/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SocketChannel.java,v 1.2 2005/06/07 19:54:12 sschneid Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.hyades.internal.execution.core.file;

import java.io.IOException;
import java.net.Socket;
import java.nio.ByteBuffer;
import java.nio.channels.Channel;
import java.nio.channels.Channels;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;

/**
 * An abstraction above the socket channel to allow a socket channel to be
 * emulated by taking a readable byte channel and a writeable byte channel and
 * combining them into a new interface. Socket channel is a class and not an
 * interface in Java, so an interface and an adapter class implementation is
 * introduced to satisfy our purposes.
 * 
 * A socket channel instance can be created from a standard socket or from a
 * standard channel.
 * 
 * @see org.eclipse.hyades.internal.execution.core.file.ISocketChannel
 * 
 * @author Scott E. Schneider
 */
class SocketChannel implements ISocketChannel {

    /**
     * An adapter that adapts a socket to look like a channel. A simple
     * interface is exposed to clients with the is open and close methods.
     * 
     * @author Scott E. Schneider
     */
    private static class ChannelInterfaceAdapter implements Channel {
        /**
         * Underlying socket that will look like a channel
         */
        private final Socket socket;

        /**
         * Adapts a standard socket to look like a standard channel interface
         * 
         * @param socket
         *            underlying socket that should expose channel behavior
         */
        ChannelInterfaceAdapter(Socket socket) {
            this.socket = socket;
        }

        /*
         * (non-Javadoc)
         * 
         * @see java.nio.channels.Channel#close()
         */
        public void close() throws IOException {
            this.socket.close();
        }

        /*
         * (non-Javadoc)
         * 
         * @see java.nio.channels.Channel#isOpen()
         */
        public boolean isOpen() {
            return !this.socket.isClosed();
        }

    }

    private final Channel channel;

    private final ReadableByteChannel readable;

    private final WritableByteChannel writable;

    /**
     * For a standard "real" channel from java.nio, simply store the instance in
     * three interface-held variables, used by the corresponding methods in this
     * class to adapt the calls (just re-forward them into the appropriate
     * delegates in most cases)
     * 
     * @param realChannel
     *            the real channel obtained from the socket channel related
     *            classes in java.nio
     */
    SocketChannel(java.nio.channels.SocketChannel realChannel) {
        this.channel = realChannel;
        this.readable = realChannel;
        this.writable = realChannel;
    }

    /**
     * A socket channel emulation is created given a standard socket, this
     * constructor will take the socket, obtain input and output streams and
     * then using the java.nio utility class it will create channels from these.
     * For the main channel interface it will adapt these appropriately to the
     * underlying socket here
     * 
     * @param socket
     */
    SocketChannel(Socket socket) throws IOException {
        this.channel = new ChannelInterfaceAdapter(socket);
        this.readable = Channels.newChannel(socket.getInputStream());
        this.writable = Channels.newChannel(socket.getOutputStream());
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.nio.channels.Channel#close()
     */
    public void close() throws IOException {
        if (this.channel != null) {
            Socket socket = ((java.nio.channels.SocketChannel) this.channel).socket();
            if (socket != null) {
                socket.close();
            }
            this.channel.close();
        }
        if (this.readable != null) {
            this.readable.close();
        }
        if (this.writable != null) {
            this.writable.close();
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.nio.channels.Channel#isOpen()
     */
    public boolean isOpen() {
        return this.channel.isOpen();
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.nio.channels.ReadableByteChannel#read(java.nio.ByteBuffer)
     */
    public int read(ByteBuffer buffer) throws IOException {
        return this.readable.read(buffer);
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.nio.channels.WritableByteChannel#write(java.nio.ByteBuffer)
     */
    public int write(ByteBuffer buffer) throws IOException {
        return this.writable.write(buffer);
    }

}