/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: HyadesTestRunner.java,v 1.7 2005/02/16 22:21:35 qiyanli Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.common.junit;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.EmptyStackException;
import java.util.Hashtable;
import java.util.Stack;

import junit.framework.AssertionFailedError;
import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestListener;
import junit.framework.TestResult;
import junit.framework.TestSuite;

import org.eclipse.hyades.test.common.event.ExecutionEvent;
import org.eclipse.hyades.test.common.event.InvocationEvent;
import org.eclipse.hyades.test.common.event.LoopEvent;
import org.eclipse.hyades.test.common.event.TypedEvent;
import org.eclipse.hyades.test.common.event.VerdictEvent;
import org.eclipse.hyades.test.common.util.BaseString;
import org.eclipse.hyades.test.common.runner.HyadesRunner;

/**
 * @author marcelop
 * @since 1.0.1
 */
public class HyadesTestRunner 
extends HyadesRunner implements TestListener
{
	protected final static boolean SYSOUT = false;
	
	private Test rootTest;
	
	private Collection succeedTests;
	private Collection failureTests;
	private Collection errorTests;
	
	private StringBuffer failureCauses;
	private StringBuffer errorCauses;
	
	private static Hashtable threadIDtoParentIDStackTable = new Hashtable();
	private static Hashtable threadIDtoParentThreadTable = new Hashtable();
	
	/**
	 * Constructor for HyadesTestRunner
	 */
	public HyadesTestRunner()
	{
		super();
		initialize();
	}
	
	public HyadesTestRunner(String args[])
	{
		super(args);
		initialize();
	}
	
	private void initialize() {
		succeedTests = Collections.synchronizedCollection(new ArrayList());
		failureTests = Collections.synchronizedCollection(new ArrayList());
		errorTests = Collections.synchronizedCollection(new ArrayList());
		
		failureCauses = new StringBuffer();
		errorCauses = new StringBuffer();
		
	}
	public static void setParentThread(Thread child, Thread parent)
	{
		threadIDtoParentThreadTable.put(new Integer(child.hashCode()), 
			new Integer(parent.hashCode()));
	}
	
	public static Integer getParentThreadID(Thread t)
	{
		return (Integer) threadIDtoParentThreadTable.get(new Integer(t.hashCode()));
	}
	
	public static Integer getParentThreadID(Integer i)
	{
		return (Integer) threadIDtoParentThreadTable.get(i);
	}
	
	public static Stack getParentStack()
	{
		int myThreadID = Thread.currentThread().hashCode();
		return (Stack) threadIDtoParentIDStackTable.get(new Integer(myThreadID));
	}
	
	public static void setParentStack(Stack parentStack)
	{
		int myThreadID = Thread.currentThread().hashCode();
		threadIDtoParentIDStackTable.put(new Integer(myThreadID), parentStack);
	}
	
	public static String peekParentEventID()
	{
		return peekParentEventID(new Integer(Thread.currentThread().hashCode()));
	}
	
	public static String peekParentEventID(Integer threadID)
	{
		Stack parentIDStack = (Stack) threadIDtoParentIDStackTable.get(threadID);

		try {		
			if ( parentIDStack != null && !parentIDStack.isEmpty())
			{
				String parent =(String) parentIDStack.peek();
				if ( parent != null && parent.length() > 0 )
				{
					return parent;
				}
			}
		}
		catch ( EmptyStackException e )
		{
		}
		
		// If we have no parent stack, or if it's empty, see if this thread
		// is a child of another thread that has a non-empty parent stack
		Integer parentThreadID = getParentThreadID(threadID);
		if ( parentThreadID == null )
		{
			return ExecutionEvent.ROOT_PARENT;
		}
		
		// Tail recursion -- don't add code after this line without a 
		// really good reason!
		return peekParentEventID(parentThreadID);
	}
	
	public static synchronized void pushParentEventID(String parentEventID)
	{
		Stack myParentIDStack = getParentStack();
		
		if ( myParentIDStack != null )
		{
			myParentIDStack.push(parentEventID);
		}
		else
		{
			myParentIDStack = new Stack();
			myParentIDStack.push(parentEventID);
			setParentStack(myParentIDStack);
		}
	}
	
	public static synchronized String popParentEventID()
	{
		Stack myParentIDStack = getParentStack();
		
		try {		
			if ( myParentIDStack != null && !myParentIDStack.isEmpty())
			{
				String parent =(String) myParentIDStack.pop();
				if ( parent == null || parent.length() == 0 )
				{
					return ExecutionEvent.ROOT_PARENT; 
				}
				else
					return parent;
			}
		}
		catch ( EmptyStackException e )
		{
		}

		return ExecutionEvent.ROOT_PARENT;
	}

	
	public static synchronized void seedParentStack(Thread t)
	{
		// This method seeds an asynchronous thread's parentID stack with the
		// parent ID from the top of the invoking thread's stack before the 
		// invocation completes (and thus, before the invoking thread can 
		// pop off the correct partent.)
		Stack newParentIDStack = new Stack();
		newParentIDStack.push(peekParentEventID());
		int newThreadID = t.hashCode();
		threadIDtoParentIDStackTable.put(new Integer(newThreadID), newParentIDStack);
	}
	
	/**
	 * Returns the root test, ie. the test that is passed to this
	 * runner.
	 * @return Test
	 */
	public Test getRoot()
	{
		return rootTest;
	}

	/**
	 * Sets the root test, ie. the test that is passed to this
	 * runner.
	 * @param root
	 */
	protected void setRoot(Test root)
	{
		rootTest = root;
	}
	
	/**
	 * Returns a collection with the hirarchy ids of the test cases that have 
	 * been succeed.
	 * @return Collection
	 */
	public Collection getSucceedTests()
	{
		return succeedTests;
	}

	/**
	 * Returns a collection with the hirarchy ids of the test cases that have 
	 * ended because of a failure.
	 * @return Collection
	 */
	public Collection getFailureTests()
	{
		return failureTests;
	}

	/**
	 * Returns a collection with the hirarchy ids of the test cases that have
	 * ended because of an error.
	 * @return Collection
	 */
	public Collection getErrorTests()
	{
		return errorTests;
	}

	/**
	 * Runs a suite extracted from a TestCase subclass.
	 * @param testClass
	 */
	public void run(Class testClass)
	{
		run(new HyadesTestSuite(testClass));
	}
	
	/**
	 * Runs a single test and collects its results. This method can be used to 
	 * start a test run from your program.
	 * @param suite
	 */
	public void run(Test test)
	{
		setRoot(test);

		TestResult result= createTestResult();
		result.addListener(this);

		runnerStarted();
		
		long startTime= getCurrentTime();
		try
		{
			test.run(result);
		}
		finally
		{
			long endTime= getCurrentTime();
			runnerExit((result == null || result.shouldStop()), (endTime-startTime));
		}		
	}

	/**
	 * Creates the TestResult to be used for the test run.
	 * @return TestResult
	 */
	protected TestResult createTestResult()
	{
		return new TestResult();
	}

	/**
	 * Returns the current timestamp.
	 * @return long
	 */
	protected long getCurrentTime()
	{
		return System.currentTimeMillis();
	}
	
	/**
	 * The runner has started. 
	 */
	protected void runnerStarted()
	{
		int count = HyadesTestUtil.countTests(new Test[]{getRoot()});
		if(SYSOUT)System.out.println("Run Started\n\tNumber of tests: " + count);
	}

	/**
	 * The runner is up to finish its activities. 
	 * @param runnerStopped
	 * @param the elapsed time
	 */
	protected void runnerExit(boolean runnerStopped, long elapsedTime)
	{
		if(SYSOUT)
		{
			if(runnerStopped)
				System.out.println("\nRun End");
			else
				System.out.println("\nRun Stop");
				
			System.out.println("\t" + HyadesTestUtil.getHierarchy(getRoot()) + "\n\t" + HyadesTestUtil.getHierarchyIds(getRoot()));
			System.out.println("\n\nResults:" 
				+ "\n\tsucceedCount: " + getSucceedTests().size()
				+ "\n\tfailureCount: " + getFailureTests().size()
				+ "\n\terrorCount: " + getErrorTests().size());
		}

		writeEvent(getDefaultVerdictEvent(getRoot()));

		TypedEvent typedEvent = new TypedEvent();
		typedEvent.setOwnerId(HyadesTestUtil.getHierarchyIds(getRoot()));
		typedEvent.setType(TypedEvent.STOP);
		typedEvent.setText(getLastEventText(runnerStopped, elapsedTime));
		typedEvent.setParentId(ExecutionEvent.ROOT_PARENT);
		writeEvent(typedEvent);
	}
	
	protected String getLastEventText(boolean runnerStopped, long elapsedTime)
	{
		return null;
	}
	
	/**
	 * @see junit.framework.TestListener#startTest(junit.framework.Test)
	 */
	public void startTest(Test test)
	{
		if(SYSOUT) System.out.println("\nStart\n\t" + HyadesTestUtil.getHierarchy(test) + "\n\t" + HyadesTestUtil.getHierarchyIds(test));
		
		TypedEvent typedEvent = new TypedEvent();
		typedEvent.setType(TypedEvent.START);

		if(test instanceof IHyadesTest)
		{
			IHyadesTest hyadesTest = (IHyadesTest)test;
			String invocationID = hyadesTest.getTestInvocationId();
			
			if(invocationID != null)
			{
				String hierarchyId = HyadesTestUtil.getHierarchyIds(hyadesTest.getParent());
				hierarchyId = HyadesTestUtil.appendToHierarchyId(hierarchyId, invocationID, hyadesTest.getIteration());
				
				InvocationEvent invocationEvent = new InvocationEvent();
				
				// Add the loop iterations to the ID to make each iteration a unique event ID
				invocationID = appendLoopIterationToID(invocationID, (IHyadesTest)hyadesTest.getParent());
				invocationEvent.setId(invocationID);
				invocationEvent.setParentId(peekParentEventID());
				
				pushParentEventID(invocationID);
				
				invocationEvent.setOwnerId(hyadesTest.getTestInvocationId());
				invocationEvent.setStatus(InvocationEvent.STATUS_SUCCESSFUL);
				invocationEvent.setReason(InvocationEvent.REASON_NONE);
				invocationEvent.setInvokedId(HyadesTestUtil.getHierarchyIds(hyadesTest));
				writeEvent(invocationEvent);
				
				// The start event should also reference the test invocation
				typedEvent.setOwnerId(hyadesTest.getTestInvocationId());

			}
			else if ( hyadesTest.getParent() != null ){
				
				// If we don't have an invocationID, and we do have a parent test
				// (meaning we're not the outermost test), then we're a loop.

				String loopId = hyadesTest.getId();
				loopId = appendLoopIterationToID(loopId, (IHyadesTest)hyadesTest.getParent());
				
				if ( hyadesTest.getIteration() == 1 )
				{
					String parentID = peekParentEventID();

					// Create a loop event
					LoopEvent loopEvent = new LoopEvent();
					loopEvent.setId(loopId);
					loopEvent.setAsychronous(!(hyadesTest.isSynchronous()));
					// loop id from test is also the id of the loop in the test suite.
					loopEvent.setOwnerId(hyadesTest.getId());
					loopEvent.setParentId(parentID);
					writeEvent(loopEvent);
					
					// Push the ID onto the parentstack
					// and make sure we remove it in in the endTest call.
					pushParentEventID(loopId);
					
					// The start event should also reference the loop element in the test model.
					typedEvent.setOwnerId(hyadesTest.getId());
				}
				else if ( hyadesTest.getIteration() > 1 )
				{
					// indicate that any child events that come along before 
					// the corresponding stop event are children of this.
					pushParentEventID(loopId);

					// The start event should also reference the loop element in the test model.
					typedEvent.setOwnerId(hyadesTest.getId());
				}
			}
		}

		typedEvent.setParentId(peekParentEventID());	
		writeEvent(typedEvent);
	}

	/**
	 * @see junit.framework.TestListener#endTest(junit.framework.Test)
	 */
	public void endTest(Test test)
	{
		if(test == getRoot())
			return;

		if(SYSOUT)
		{
			System.out.println("\nEnd\n\t" + HyadesTestUtil.getHierarchy(test) + "\n\t" + HyadesTestUtil.getHierarchyIds(test));
			if((test instanceof HyadesTestSuite) && (((HyadesTestSuite)test).getArbiter() != null))
			{
				System.out.println("\tResults:" 
					+ "\n\t\tsucceedCount: " + filterTests(test, getSucceedTests()).size()
					+ "\n\t\tfailureCount: " + filterTests(test, getFailureTests()).size()
					+ "\n\t\terrorCount: " + filterTests(test, getErrorTests()).size());
			}
		}

		if((test instanceof TestCase) && (!getFailureTests().contains(test)) && (!getErrorTests().contains(test)))
			getSucceedTests().add(test);

		writeEvent(getDefaultVerdictEvent(test));
		TypedEvent typedEvent = new TypedEvent();

		if(test instanceof IHyadesTest)
		{
			IHyadesTest hyadesTest = (IHyadesTest)test;
			String invocationID = hyadesTest.getTestInvocationId();
			
			if(invocationID != null)
			{
				String hierarchyId = HyadesTestUtil.getHierarchyIds(hyadesTest.getParent());
				hierarchyId = HyadesTestUtil.appendToHierarchyId(hierarchyId, invocationID, hyadesTest.getIteration());
				
				typedEvent.setParentId(popParentEventID());
				
				typedEvent.setOwnerId(invocationID);

			}
			else if ( hyadesTest.getParent() != null && hyadesTest.getIteration() > 0 )
			{
				typedEvent.setParentId(popParentEventID());
				typedEvent.setOwnerId(hyadesTest.getId());
			}
		}
		
		typedEvent.setType(TypedEvent.STOP);
		writeEvent(typedEvent);
	}
	
	/**
	 * Returns the default verdict event, which is pass for test
	 * cases and the arbiter evaluation for test suites.
	 * @param test
	 * @return VerdictEvent
	 */
	protected VerdictEvent getDefaultVerdictEvent(Test test)
	{
		VerdictEvent verdictEvent = null;
		if(test instanceof TestSuite)
		{
			if(test instanceof HyadesTestSuite)
			{
				HyadesTestSuite hyadesTestSuite = (HyadesTestSuite)test;
				if(hyadesTestSuite.getArbiter() != null)
				{
					verdictEvent = hyadesTestSuite.getArbiter().analyse(hyadesTestSuite, 
						filterTests(test, getSucceedTests()), filterTests(test, getErrorTests()), 
						filterTests(test, getFailureTests()), 
						errorCauses.toString(), failureCauses.toString());
						
					if(verdictEvent != null)
					{
						if(verdictEvent.getOwnerId() == null)
							verdictEvent.setOwnerId(hyadesTestSuite.getId());
					}
				}
			}
		}
		else if(test instanceof TestCase)
		{
			if((!getFailureTests().contains(test)) && (!getErrorTests().contains(test)))
			{
				verdictEvent = new VerdictEvent();
				verdictEvent.setOwnerId(((HyadesTestCase)test).getId());
				verdictEvent.setVerdict(VerdictEvent.VERDICT_PASS);
				verdictEvent.setId(computeVerdictId(peekParentEventID()));
			}
		}
		
		// If test is neither a HyadesTestSuite nor a HyadesTestCase, then verdict
		// is null, which is an acceptable return value here.
		if ( verdictEvent != null )
			verdictEvent.setParentId(peekParentEventID());		
		return verdictEvent;
	}

	/**
	 * @see junit.framework.TestListener#addError(junit.framework.Test, java.lang.Throwable)
	 */
	public void addError(Test test, Throwable throwable)
	{
		if(SYSOUT) System.out.println("\nError\n\t" + HyadesTestUtil.getHierarchy(test) + "\n\t" + HyadesTestUtil.getHierarchyIds(test));
		getErrorTests().add(test);

		VerdictEvent verdictEvent = new VerdictEvent();

		if ( test instanceof IHyadesTest )
		{
			String id = computeVerdictId(peekParentEventID());
			verdictEvent.setId(id);
			synchronized (errorCauses)
			{
				if (errorCauses.length() > 0)
					errorCauses.append(',');
				errorCauses.append(id);
			}						
		}

		verdictEvent.setParentId(peekParentEventID());
		verdictEvent.setVerdict(VerdictEvent.VERDICT_ERROR);
		verdictEvent.setReason(VerdictEvent.REASON_SEE_DESCRIPTION);
		verdictEvent.setText(BaseString.getStackTrace(throwable));
		writeEvent(verdictEvent);
	}

	/**
	 * @see junit.framework.TestListener#addFailure(junit.framework.Test, junit.framework.AssertionFailedError)
	 */
	public void addFailure(Test test, AssertionFailedError throwable)
	{
		if(SYSOUT) System.out.println("\nFailure\n\t" + HyadesTestUtil.getHierarchy(test) + "\n\t" + HyadesTestUtil.getHierarchyIds(test));
		getFailureTests().add(test);
		
		VerdictEvent verdictEvent = new VerdictEvent();

		if ( test instanceof IHyadesTest )
		{
			String id = computeVerdictId(peekParentEventID());
			verdictEvent.setId(id);
			synchronized (failureCauses)
			{
				if (failureCauses.length() > 0)
				failureCauses.append(',');
				failureCauses.append(id);
			}						
		}

		verdictEvent.setParentId(peekParentEventID());
		verdictEvent.setVerdict(VerdictEvent.VERDICT_FAIL);
		verdictEvent.setReason(VerdictEvent.REASON_SEE_DESCRIPTION);
		verdictEvent.setText(BaseString.getStackTrace(throwable));
		writeEvent(verdictEvent);
	}

	/**
	 * Writes an event.
	 * @param executionEvent
	 */	
	protected void writeEvent(ExecutionEvent executionEvent)
	{
		if(executionEvent != null)
			System.out.println(executionEvent);
	}
	
	protected Collection filterTests(Test parent, Collection tests)
	{
		Test[] testArray = (Test[])tests.toArray(new Test[tests.size()]);
		Collection filteredTests = new ArrayList(testArray.length);
		for (int i = 0, maxi = testArray.length; i < maxi; i++)
		{
			if((testArray[i] instanceof IHyadesTest) && HyadesTestUtil.isParentOf(parent, testArray[i]))
				filteredTests.add(testArray[i]);
		}
		
		return Collections.unmodifiableCollection(filteredTests);
	}
	
	protected String computeVerdictId(String parentId)
	{
		return parentId + "_verdict";
	}
	
	protected String appendLoopIterationToID( String ID, IHyadesTest parentTest )
	{
		StringBuffer iterationString = new StringBuffer("");
		
		String iterations = HyadesTestSuite.getIterationsString(parentTest);
		return ID + iterations;
/*		
		// Walk the parent tests and compute the loop iteration for this ID.
		for ( IHyadesTest test = parentTest; test != null; test = (IHyadesTest) test.getParent() )
		{
			if ( test.isLoop() )
			{
				int iteration = HyadesTestSuite.getParentIteration(test);
				
				if ( iteration > 0 )
				{
					iterationString.append('(' + Integer.toString(iteration) + ')');
				}
				else
				{
					HyadesTestSuite.debug_out("[Hey!] Parent iteration == 0 for " + test + ".  CurrentThread = " + Thread.currentThread() + ", (Async[" + (Thread.currentThread() instanceof HyadesTestSuite.AsynchronousThread) + "]");
				}
			}
		}
		if ( iterationString.length() > 0 )
		{
			ID = ID + iterationString.toString();
		}
		
		return ID;
*/

	}
}
