/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: BaseString.java,v 1.3 2005/02/16 22:21:35 qiyanli Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.test.common.util;

import java.io.ByteArrayOutputStream;
import java.io.PrintStream;
import java.io.UnsupportedEncodingException;
import java.util.Collection;
import java.util.List;
import java.util.Vector;

public class BaseString
{
	private static final String ALPHANUMERIC = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";
	private static final String NUMERIC = "0123456789";
	/**
	 * 
	 * @param o
	 * @return
	 */
	public static String toString(Object o)
	{
		if(o == null)
			return "";
			
		if(o instanceof Object[])
			return arrayToString((Object[])o);
			
		if(o instanceof List)
			return arrayToString(((List)o).toArray());

		if(o instanceof Collection)
			return arrayToString(((Collection)o).toArray());
			
		if(o instanceof String)
			return (String)o;
			
		return o.toString();
	}
	
	public static String arrayToString(Object[] array)
	{
		return arrayToString(array, " ");
	}
	
	public static String arrayToString(Object[] array, String separator)
	{
		if(array == null)
			return "";
		separator = toString(separator);	
		
		String ret = "";
		int max = array.length;
		if(max > 0)
		{
			StringBuffer sb = new StringBuffer();
			for(int i=0; i<max; i++)
				sb.append(separator).append(array[i].toString());
		
			ret = sb.toString().substring(separator.length());
		}
		
		return ret;
	}

    /**
     * 
     * @param throwable
     * @return
     */
	public static String getStackTrace(Throwable throwable)
	{
		if(throwable == null)
			return null;
			
		ByteArrayOutputStream errorStream;		
		PrintStream errorPS;		

		errorStream = new ByteArrayOutputStream();
		errorPS = new PrintStream(errorStream);

		throwable.printStackTrace(errorPS);
		errorPS.flush();
		
		String stackTrace = errorStream.toString();
		
		try
		{
			errorStream.close();
		}
		catch(Exception e)
		{
		}
		
		return stackTrace;
	}
	
	
	/**
	 * Encodes a String from unicode to the specifed encoding.
	 * This method returns null if the encoding is not supported
	 * 
	 * @param encoding
	 * @param text
	 * @return byte[]
	 */
	public static byte[] encode(String encoding, String text)
	{
		if(text != null)
		{
			if(encoding == null)
				return text.getBytes();
				
			try
			{
				return text.getBytes(encoding);
			}
			catch(UnsupportedEncodingException e)
			{
			}
		}
		
		return null;
	}
	
	
	public static String[] tokenizer(String text, String token, boolean includeToken)
	{
		if((text == null) || (token == null))
			return null;

		Vector ret = new Vector();
		int index = 0;
		do
		{
			index = text.indexOf(token);
			if(index >= 0)
			{
				if(index > 0) ret.add(text.substring(0, index));
				index += token.length();
				text = text.substring(index);
				
				if(includeToken)
					ret.add(token);
			}
		}	
		while(index >= 0);
		if(text.length() > 0)
			ret.add(text);

		return (String[]) ret.toArray(new String[ret.size()]);
	}
	
   
	/**
	 * Returns number of occurrences of searchChar within srcString
	 * example:
	 * 	srcString = "::f::f::g"
	 * 	seachrChar = ':'
	 * 	return = 6
	 */  
	 public static int occurrenceOf(String srcString, char searchChar)
	 {
		 int result = 0;
		
		 // walk backward to find if a char within srcString is in validString
		 if(srcString.length() > 0)
		 {
			 for( int i=0; i<srcString.length() ; i++ )
			 {
				 // found, increment the count
				 if(searchChar == srcString.charAt(i))
					 result++;
			 }
		 }

		 return result;
	 }
	 
	
	
	/**
	 * Returns a new string resulting from replacing all occurrences of
	 * <code>oldString</code> in the "in" string with <code>newString</code>.
	 * 
	 * @param in
	 * @param oldString
	 * @param newString
	 * @return String
	 */	
	public static String replace(String in, String oldString, String newString)
	{
		if((in == null) || (oldString == null) || (newString == null))
			return in;
			
		if(oldString.length() == 0)
			return in;
		
		if(oldString.length() == 1 && newString.length() == 1)
			return in.replace(oldString.charAt(0), newString.charAt(0));

		int lastIndex = 0;
		int newIndex = 0;
		StringBuffer sb = new StringBuffer();
		for(;;)
		{
			newIndex = in.indexOf(oldString, lastIndex);
			if(newIndex != -1)
			{
				sb.append(in.substring(lastIndex, newIndex)).append(newString);
				lastIndex = newIndex + oldString.length();
			}
			else
			{
				sb.append(in.substring(lastIndex));
				break;
			}
		}
		return sb.toString();
	}
	

	/**
	 * Returns the proper case of a string.
	 * 
	 * Examples:
	 * 		this is a text => This Is A Text
	 * 		myClass	=> MyClass
	 */
	public static String toProperCase(String text)
	{
		if((text == null) || (text.length() == 0))
			return null;
			
		if(text.length() == 1)
			return text.toUpperCase();
			
		StringBuffer sb = new StringBuffer();
		String[] pieces = tokenizer(text, " ", true);
		for(int i = 0; i < pieces.length; i++)
		{
			sb.append(pieces[i].substring(0,1).toUpperCase());
			if(pieces[i].length() > 1)
				sb.append(pieces[i].substring(1));
		}
		
		return sb.toString();		
	}	

   
	/**
	 * Returns true if all chars in srcString are in {a...z,} or {A...Z} {0...9}
	 */
	public static boolean isAlphanumeric(String srcString)
	{
		return (lastIndexOfAnyBut(srcString, ALPHANUMERIC) == -1);
	}

	/**
		 * Returns the last index within srcString that is not in the validString
		 * example:
		 * 	srcString = "abcdefg"
		 * 	validString = "bcfg"
		 * 	return = 4 (i.e. char e is not in "bcfg")  - 1st index = 0
		 */ 
		public static int lastIndexOfAnyBut(String srcString, String validString)
		{
			int result = -1;
			int srcLen = srcString.length();

			// walk backward to find if a char within srcString is in validString
			for(int i = srcLen-1; i >= 0 ; i--)
			{
				// not found, stop it
				if(validString.indexOf(srcString.charAt(i)) == -1)
				{
					result = i;
					break;
				}
			}

			return result;
		}


	/**
	 * Returns true if all chars are in '0' - '9'
	 */
	public static boolean isDigits(String srcString)
	{
		return(lastIndexOfAnyBut(srcString, NUMERIC) == -1);
	}


}