/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ProfileRuntimeWorkbenchDelegate.java,v 1.27 2005/02/16 22:24:01 qiyanli Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.launcher;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.StringTokenizer;
import java.util.TreeMap;
import java.util.Vector;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.model.LaunchConfigurationDelegate;
import org.eclipse.hyades.trace.internal.ui.TraceConstants;
import org.eclipse.hyades.trace.ui.HyadesConstants;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.util.PDCoreUtil;
import org.eclipse.jdt.launching.ExecutionArguments;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.pde.core.plugin.IPluginModelBase;
import org.eclipse.pde.internal.core.ExternalModelManager;
import org.eclipse.pde.internal.core.PDECore;
import org.eclipse.pde.internal.core.TargetPlatform;
import org.eclipse.pde.internal.ui.PDEPlugin;
import org.eclipse.pde.internal.ui.build.ClasspathHelper;
import org.eclipse.pde.internal.ui.launcher.ILauncherSettings;
import org.eclipse.pde.internal.ui.launcher.LauncherUtils;

public class ProfileRuntimeWorkbenchDelegate extends LaunchConfigurationDelegate implements ILauncherSettings {
	
	private static final String ECLIPSE_MAIN = "org.eclipse.core.launcher.Main";

	// These variables came from WorkbenchLaunchConfigurationDelegate.java (see comment below).
	private static final String KEY_BAD_FEATURE_SETUP = "WorkbenchLauncherConfigurationDelegate.badFeatureSetup";
	private static final String KEY_NO_STARTUP = "WorkbenchLauncherConfigurationDelegate.noStartup";
	private File fConfigDir = null;
	
	public void launch(ILaunchConfiguration configuration, String mode, ILaunch launch, IProgressMonitor monitor) throws CoreException {

		try {
			monitor.beginTask("", 4);
			monitor.subTask(UIPlugin.getResourceString("LAUNCHING_MESSAGES_ON_PROGRESS_VIEW"));
			fConfigDir = null;
			boolean success = ProfileLaunchUtil.performProfilingTypesLaunch(configuration);
			monitor.worked(1);
			
			if (!success) {
				monitor.setCanceled(true);
				return;
			}

			String workspace = getWorkspace(configuration);
			if (!LauncherUtils.clearWorkspace(configuration, workspace, new SubProgressMonitor(monitor, 1))) {
				monitor.setCanceled(true);
				return;
			}
			
			// clear config area, if necessary
			if (configuration.getAttribute(CONFIG_CLEAR, false))
				LauncherUtils.clearConfigArea(getConfigDir(configuration), new SubProgressMonitor(monitor, 1));
			launch.setAttribute(ILauncherSettings.CONFIG_LOCATION, getConfigDir(configuration).toString());
			monitor.worked(1);
			
			// Build up the trace argument information.
			TraceArguments traceArguments = new TraceArguments(ECLIPSE_MAIN);
			traceArguments.setClassPath(getClassPath(configuration));
			traceArguments.setParameters(getProgramArgs(configuration));
			traceArguments.setVMArguments(getVMArgs(configuration));
			traceArguments.setEnvironmentVariable(ProfileLaunchUtil.getEnvironmentVariables(configuration));
			traceArguments.setAutoMonitoring(isAutoMonitoring(configuration));
			traceArguments.setHostName(getHostName(configuration));
			traceArguments.setPortNumber(getPortNumber(configuration));
			traceArguments.setProfileFile(getProfileFile(configuration));

			ProfilingSetsManager manager = ProfilingSetsManager.instance();
			
			// Get all of the profiling-related information.
			ArrayList filters = manager.getFilters(configuration);
			Vector options = manager.getOptions(configuration);
			String projectName = getProjectName(configuration);
			String monitorName = getMonitorName(configuration);

			monitor.worked(1);

			// Launch the process and start the trace.
			if (!PDCoreUtil.launchTrace(traceArguments, filters, options, projectName, monitorName, launch)) {
				monitor.setCanceled(true);
			}
			monitor.worked(1);

		} catch (CoreException e) {
			monitor.setCanceled(true);
			throw e;
		}
	}

	/**
	 * Returns true if the user wants to auto-monitor the workbench after it has been launched.  
	 * 
	 * @param configuration the ILaunchConfiguration object to retrieve this information from.
	 * @return true if user specified to auto-montor. Defaults to true.
	 * @throws CoreException
	 */
	private boolean isAutoMonitoring(ILaunchConfiguration configuration) throws CoreException {
		return configuration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_AUTO_MONITORING, true);
	}
	
	/**
	 * Returns the hostname of where to launch the workbench.  
	 * 
	 * @param configuration the ILaunchConfiguration object to retrieve this information from.
	 * @return hostname of where to launch the workbench.
	 * @throws CoreException
	 */
	private String getHostName(ILaunchConfiguration configuration) throws CoreException {
		return configuration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_HOSTNAME, "localhost");
	}
	
	/**
	 * Returns the port number of where to launch the workbench.  
	 * 
	 * @param configuration the ILaunchConfiguration object to retrieve this information from.
	 * @return port number of the machine to launch the workbench.
	 * @throws CoreException
	 */
	private int getPortNumber(ILaunchConfiguration configuration) throws CoreException {
		IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();
		return configuration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PORT, store.getInt(HyadesConstants.LOCALHOST_PORT));
	}
	
	/**
	 * Returns the project name to store all of the profiled data.  
	 * 
	 * @param configuration the ILaunchConfiguration object to retrieve this information from.
	 * @return name of the project to use to hold the profiled data.
	 * @throws CoreException
	 */
	private String getProjectName(ILaunchConfiguration configuration) throws CoreException {
		IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();
		return configuration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_DESTINATION_PROJECT, store.getString(TraceConstants.TRACE_PROJECT_NAME));
	}
	
	/**
	 * Returns the monitor name.  
	 * 
	 * @param configuration the ILaunchConfiguration object to retrieve this information from.
	 * @return name of the monitor to use.
	 * @throws CoreException
	 */
	private String getMonitorName(ILaunchConfiguration configuration) throws CoreException {
		IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();
		return configuration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_DESTINATION_MONITOR, store.getString(TraceConstants.TRACE_MONITOR_NAME));
	}
	
	/**
	 * Returns the path to the workspace in which this workbench should use.  
	 * 
	 * @param configuration the ILaunchConfiguration object to retrieve this information from.
	 * @return path to the workspace to use.
	 * @throws CoreException
	 */
	// We should synchonize this with the call used in getProgramArguments (that comes from WorkbenchLaunchConfigurationDelegate.java).
	// *Ideally*, an API should be provided for us to retrieve this information, so we wouldn't even have to worry about sync'ing these
	// two.
	private String getWorkspace(ILaunchConfiguration configuration) throws CoreException {
		return configuration.getAttribute(LOCATION + "0", LauncherUtils.getDefaultPath().append("runtime-workbench-workspace").toOSString());
	}
	
	/**
	 * Returns the path to the profile file.  
	 * 
	 * @param configuration the ILaunchConfiguration object to retrieve this information from.
	 * @return path to the profile, or null if the user wishes to profile to display.
	 * @throws CoreException
	 */
	private String getProfileFile(ILaunchConfiguration configuration) throws CoreException {
		if (configuration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILE_TO_FILE, false))
			return configuration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_DESTINATION_FILE, (String)null);
		return null;
	}

	/**
	 * Returns a String of the classpath to use.  
	 * 
	 * @return the classpath.
	 * @throws CoreException
	 */
	// This method uses the same call found in WorkbenchLaunchConfigurationDelegate.java to get this information
	// then transforms it into a format that we can use.
	private String getClassPath(ILaunchConfiguration conf) throws CoreException {
		String[] classpath = LauncherUtils.constructClasspath(conf);
		if (classpath == null) {
			String message = PDEPlugin.getResourceString(KEY_NO_STARTUP);
			throw new CoreException(LauncherUtils.createErrorStatus(message));
		}

		if (classpath.length == 0) return null;

		String classpathList = "";
		for(int i = 0; i < classpath.length - 1; i ++) {
			classpathList = classpathList + classpath[i] + File.pathSeparatorChar;
		}
		classpathList = classpathList + classpath[classpath.length - 1];
		
		return classpathList;
	}
	
	/**
	 * Returns the list of VM Arguments to use.  
	 * 
	 * @param configuration the ILaunchConfiguration object to retrieve this information from.
	 * @return list of VM arguments.
	 * @throws CoreException
	 */
	// This method uses the same call found in WorkbenchLaunchConfigurationDelegate.java to get this information
	// then transforms it into a format that we can use.
	private String getVMArgs(ILaunchConfiguration configuration) throws CoreException {
		String[] vmArgs = new ExecutionArguments(configuration.getAttribute(VMARGS,""),"").getVMArgumentsArray();
		if (vmArgs.length == 0) {
			return null;
		}
		return ProfileLaunchUtil.convertToDelimitedString(vmArgs, ' ');
	}

	/**
	 * Returns an ArrayList of the environment variables.  
	 * 
	 * @param configuration the ILaunchConfiguration object to retrieve this information from.
	 * @return list of environment variables.
	 * @throws CoreException
	 */
	// This method calls the methods below (also found in WorkbenchLaunchConfigurationDelegate.java) to get 
	// this information then transforms it into a format that we can use.
	private String getProgramArgs(ILaunchConfiguration configuration) throws CoreException {
		String[] programArgs = getProgramArguments(configuration);
		if (programArgs.length == 0) {
			return null;
		}
		return ProfileLaunchUtil.convertToDelimitedString(programArgs, ' ');
	}
	
	/*
	 * Comment by sshiah:
	 * 
	 * All of the code below came from WorkbenchLaunchConfigurationDelegate.java, in org.eclipse.pde.internal.ui.launcher.
	 * This code processes all of the arguments provided by the user in the non-profiling tabs and builds the invocation 
	 * string that should be used. Since these helper methods are private within WorkbenchLaunchConfigurationDelegate.java, 
	 * a decision was made to re-use the existing code. 
	 * 
	 * A request has been made (bugzilla #53124) to allow access to these methods (or another solution in which we could
	 * access these processed arguments). When such access has been provided, the code below should be removed and replaced
	 * with the appropriate API call(s). Also, this class should not implement ILauncherSettings.
	 * 
	 * reference: bugzilla bugs #51979, #53124
	 */
	private String[] getProgramArguments(ILaunchConfiguration configuration) throws CoreException {
		ArrayList programArgs = new ArrayList();
		
		// If a product is specified, then add it to the program args
		if (configuration.getAttribute(USE_PRODUCT, false)) {
			programArgs.add("-product"); //$NON-NLS-1$
			programArgs.add(configuration.getAttribute(PRODUCT, "")); //$NON-NLS-1$
		} else {
			// specify the application to launch
			programArgs.add("-application"); //$NON-NLS-1$
			programArgs.add(configuration.getAttribute(APPLICATION, LauncherUtils.getDefaultApplicationName()));
		}
		
		// specify the workspace location for the runtime workbench
		String targetWorkspace =
			configuration.getAttribute(LOCATION + "0", LauncherUtils.getDefaultPath().append("runtime-workbench-workspace").toOSString()); //$NON-NLS-1$ //$NON-NLS-2$
		programArgs.add("-data"); //$NON-NLS-1$
		programArgs.add(targetWorkspace);
		
		boolean isOSGI = PDECore.getDefault().getModelManager().isOSGiRuntime();
		if (configuration.getAttribute(USEFEATURES, false)) {
			validateFeatures();
			IPath installPath = PDEPlugin.getWorkspace().getRoot().getLocation();
			programArgs.add("-install"); //$NON-NLS-1$
			programArgs.add("file:" + installPath.removeLastSegments(1).addTrailingSeparator().toString()); //$NON-NLS-1$
			programArgs.add("-update"); //$NON-NLS-1$
		} else {
			TreeMap pluginMap = LauncherUtils.getPluginsToRun(configuration);
			if (pluginMap == null) 
				return null;
				
			String primaryFeatureId = LauncherUtils.getPrimaryFeatureId();
			TargetPlatform.createPlatformConfigurationArea(
					pluginMap,
					getConfigDir(configuration),
					primaryFeatureId,
					LauncherUtils.getAutoStartPlugins(configuration));
			programArgs.add("-configuration"); //$NON-NLS-1$
			if (isOSGI)
				programArgs.add("file:" + new Path(getConfigDir(configuration).getPath()).addTrailingSeparator().toString()); //$NON-NLS-1$
			else
				programArgs.add("file:" + new Path(getConfigDir(configuration).getPath()).append("platform.cfg").toString()); //$NON-NLS-1$ //$NON-NLS-2$
			
			if (!isOSGI) {
				if (primaryFeatureId != null) {
					programArgs.add("-feature"); //$NON-NLS-1$
					programArgs.add(primaryFeatureId);					
				}
				IPluginModelBase bootModel = (IPluginModelBase)pluginMap.get("org.eclipse.core.boot"); //$NON-NLS-1$
				String bootPath = LauncherUtils.getBootPath(bootModel);
				if (bootPath != null && !bootPath.endsWith(".jar")) { //$NON-NLS-1$
					programArgs.add("-boot"); //$NON-NLS-1$
					programArgs.add("file:" + bootPath); //$NON-NLS-1$
				}
			}
		}
		
		// add the output folder names
		programArgs.add("-dev"); //$NON-NLS-1$
		if (PDECore.getDefault().getModelManager().isOSGiRuntime())
			programArgs.add(ClasspathHelper.getDevEntriesProperties(getConfigDir(configuration).toString() + "/dev.properties", true)); //$NON-NLS-1$
		else
			programArgs.add(ClasspathHelper.getDevEntries(true));

		// add tracing, if turned on
		if (configuration.getAttribute(TRACING, false)
				&& !TRACING_NONE.equals(configuration.getAttribute(TRACING_CHECKED, (String) null))) {
			programArgs.add("-debug"); //$NON-NLS-1$
			programArgs.add(
				LauncherUtils.getTracingFileArgument(
					configuration,
					getConfigDir(configuration).toString() + Path.SEPARATOR + ".options")); //$NON-NLS-1$
		}

		// add the program args specified by the user
		StringTokenizer tokenizer =
			new StringTokenizer(configuration.getAttribute(PROGARGS, "")); //$NON-NLS-1$
		while (tokenizer.hasMoreTokens()) {
			programArgs.add(tokenizer.nextToken());
		}
		
		// show splash only if we are launching the default application
		boolean showSplash = true;
		int index = programArgs.indexOf("-application"); //$NON-NLS-1$
		if (index != -1 && index <= programArgs.size() - 2) {
			if (!programArgs.get(index + 1).equals(LauncherUtils.getDefaultApplicationName())) {
				showSplash = false;
			}
		}
		if (showSplash && !programArgs.contains("-nosplash")) { //$NON-NLS-1$
			programArgs.add(0, "-showsplash"); //$NON-NLS-1$
			programArgs.add(1, computeShowsplashArgument());
		}
		return (String[])programArgs.toArray(new String[programArgs.size()]);
	}
	
	private String[] getVMArguments(ILaunchConfiguration configuration) throws CoreException {
		return new ExecutionArguments(configuration.getAttribute(VMARGS,""),"").getVMArgumentsArray(); //$NON-NLS-1$ //$NON-NLS-2$
	}
			
	private void validateFeatures() throws CoreException {
		IPath installPath = PDEPlugin.getWorkspace().getRoot().getLocation();
		String lastSegment = installPath.lastSegment();
		boolean badStructure = lastSegment == null;
		if (!badStructure) {
			IPath featuresPath = installPath.removeLastSegments(1).append("features"); //$NON-NLS-1$
			badStructure = !lastSegment.equalsIgnoreCase("plugins") //$NON-NLS-1$
					|| !featuresPath.toFile().exists();
		}
		if (badStructure) {
			throw new CoreException(LauncherUtils.createErrorStatus(PDEPlugin
					.getResourceString(KEY_BAD_FEATURE_SETUP)));
		} else {
			// Ensure important files are present
			ensureProductFilesExist(getProductPath());
		}
	}
	
	private IPath getInstallPath() {
		return PDEPlugin.getWorkspace().getRoot().getLocation();
	}
	
	private IPath getProductPath() {
		return getInstallPath().removeLastSegments(1);
	}

	private String computeShowsplashArgument() {
		IPath eclipseHome = ExternalModelManager.getEclipseHome();
		IPath fullPath = eclipseHome.append("eclipse"); //$NON-NLS-1$
		return fullPath.toOSString() + " -showsplash 600"; //$NON-NLS-1$
	}

	private void ensureProductFilesExist(IPath productArea) {
		File productDir = productArea.toFile();		
		File marker = new File(productDir, ".eclipseproduct"); //$NON-NLS-1$
		IPath eclipsePath = ExternalModelManager.getEclipseHome();
		if (!marker.exists()) 
			copyFile(eclipsePath, ".eclipseproduct", marker); //$NON-NLS-1$
		
		if (PDECore.getDefault().getModelManager().isOSGiRuntime()) {
			File configDir = new File(productDir, "configuration"); //$NON-NLS-1$
			if (!configDir.exists())
				configDir.mkdirs();		
			File ini = new File(configDir, "config.ini");			 //$NON-NLS-1$
			if (!ini.exists())
				copyFile(eclipsePath.append("configuration"), "config.ini", ini); //$NON-NLS-1$ //$NON-NLS-2$
		} else {
			File ini = new File(productDir, "install.ini"); //$NON-NLS-1$
			if (!ini.exists()) 
				copyFile(eclipsePath, "install.ini", ini);		 //$NON-NLS-1$
		}
	}

	private void copyFile(IPath eclipsePath, String name, File target) {
		File source = new File(eclipsePath.toFile(), name);
		if (source.exists() == false)
			return;
		FileInputStream is = null;
		FileOutputStream os = null;
		try {
			is = new FileInputStream(source);
			os = new FileOutputStream(target);
			byte[] buf = new byte[1024];
			long currentLen = 0;
			int len = is.read(buf);
			while (len != -1) {
				currentLen += len;
				os.write(buf, 0, len);
				len = is.read(buf);
			}
		} catch (IOException e) {
		} finally {
			try {
				if (is != null)
					is.close();
				if (os != null)
					os.close();
			} catch (IOException e) {
			}
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.LaunchConfigurationDelegate#getBuildOrder(org.eclipse.debug.core.ILaunchConfiguration, java.lang.String)
	 */
	protected IProject[] getBuildOrder(ILaunchConfiguration configuration,
			String mode) throws CoreException {
		return computeBuildOrder(LauncherUtils.getAffectedProjects(configuration));
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.LaunchConfigurationDelegate#getProjectsForProblemSearch(org.eclipse.debug.core.ILaunchConfiguration, java.lang.String)
	 */
	protected IProject[] getProjectsForProblemSearch(
			ILaunchConfiguration configuration, String mode)
			throws CoreException {
		return LauncherUtils.getAffectedProjects(configuration);
	}
	
	private File getConfigDir(ILaunchConfiguration config) {
		if (fConfigDir == null) {
			try {
				if (config.getAttribute(USEFEATURES, false)) {
					String root = getProductPath().toString();
					if (PDECore.getDefault().getModelManager().isOSGiRuntime())
						root += "/configuration"; //$NON-NLS-1$
					fConfigDir = new File(root);
				} else {
					fConfigDir = LauncherUtils.createConfigArea(config.getName());
				}
			} catch (CoreException e) {
				fConfigDir = LauncherUtils.createConfigArea(config.getName());
			}
		}
		if (!fConfigDir.exists())
			fConfigDir.mkdirs();
		return fConfigDir;
	}
}
