/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.hyades.trace.ui.internal.navigator;

import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.trace.internal.ui.PDContentProvider;
import org.eclipse.hyades.trace.internal.ui.PDLabelProvider;
import org.eclipse.hyades.trace.internal.ui.PDProjectExplorer;
import org.eclipse.hyades.trace.internal.ui.TraceConstants;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.util.TString;
import org.eclipse.hyades.ui.internal.extension.INavigatorLayout;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ITreeContentProvider;

/**
 * Presents the contents of the Profiling Monitor view in a simple, or
 * application-oriented layout. The projects, monitors and nodes are hidden,
 * and the process labels are more descriptive (they indicate which host on
 * which the process is running).
 * 
 * @author Curtis d'Entremont
 * @since 3.3
 */
public class SimpleLayout implements INavigatorLayout {
	
	private IContentProvider _contentProvider;
	private ILabelProvider _labelProvider;
	private PDProjectExplorer _viewer;
	
	/**
	 * Constructs a new FlatLayout for the given Profiling Monitor view
	 * instance.
	 * 
	 * @param viewer the Profiling Monitor view.
	 */
	public SimpleLayout(PDProjectExplorer viewer) {
		_viewer = viewer;
	}
	
	/**
	 * Returns this layout's content provider, which does not supply
	 * projects, monitors, or nodes. The content provider is lazily
	 * loaded and cached.
	 * 
	 * @return this layout's content provider.
	 * @see org.eclipse.hyades.ui.internal.extension.INavigatorLayout#getContentProvider()
	 */
	public IContentProvider getContentProvider() {
		if (_contentProvider == null) {
			_contentProvider = new SimpleContentProvider(new PDContentProvider(_viewer));
		}
		return _contentProvider;
	}
	
	/**
	 * Returns this layout's label provider, which appends an extra bit
	 * of information to the process labels which indicates on which host the
	 * process is running. The label provider is lazily loaded and cached.
	 * 
	 * @return this layout's label provider.
	 * @see org.eclipse.hyades.ui.internal.extension.INavigatorLayout#getLabelProvider()
	 */
	public ILabelProvider getLabelProvider() {
		if (_labelProvider == null) {
			_labelProvider = new SimpleLabelProvider();
		}
		return _labelProvider;
	}
}

/**
 * Hides monitors, nodes, and possibly projects, depending on the user preference.
 * This provides an application-oriented view, which is useful when all the processes
 * are running on the same machine.
 */
class SimpleContentProvider extends AbstractFilteredContentProvider {

	private IPreferenceStore _store = UIPlugin.getDefault().getPreferenceStore();
	
	/**
	 * Constructs a new simple view of the profiling elements in the profiling
	 * monitor view.
	 * 
	 * @param provider The underlying content provider that this provider filters.
	 */
	public SimpleContentProvider(ITreeContentProvider provider) {
		super(provider);
	}
	
	/**
	 * Returns true if the given element and all its children should be filtered
	 * out entirely.
	 * 
	 * @param element The element to check.
	 * @return Whether or not to filter out the element and all its children.
	 * @see org.eclipse.hyades.trace.ui.internal.navigator.AbstractFilteredContentProvider#isFiltered(java.lang.Object)
	 */
	public boolean isFiltered(Object element) {
		return false;
	}
	
	/**
	 * Returns true if the given element (but not its children) should be filtered
	 * out, or "skipped". In this case, monitors and nodes are always skipped, and
	 * projects are skipped when the user preference for this is selected.
	 * 
	 * @param element The element to check.
	 * @return Whether or not to skip the given element.
	 * @see org.eclipse.hyades.trace.ui.internal.navigator.AbstractFilteredContentProvider#isSkipped(java.lang.Object)
	 */
	public boolean isSkipped(Object element) {
		if (element instanceof IProject) {
			return !_store.getBoolean(TraceConstants.DISPLAY_PROJECTS);
		}
		return (element instanceof TRCMonitor) || (element instanceof TRCNode) || (element instanceof IFolder);
	}
}

/**
 * Overrides the default label provider to use a longer form for the process
 * label. The label takes the form "<process_name> at <host_name> [ PID: X ]".
 */
class SimpleLabelProvider extends PDLabelProvider {
	
	/**
	 * Returns the longer form of the process label, which indicates
	 * which host the process is running on.
	 * 
	 * @see org.eclipse.hyades.trace.internal.ui.PDLabelProvider#getProcessLabel(org.eclipse.hyades.models.hierarchy.TRCProcessProxy)
	 * @param process The process for which to get the label.
	 * @return An appropriate label for the given process for this layout.
	 */
	protected String getProcessLabel(TRCProcessProxy process) {
		
		/*
		 * Only show the process ID if it's a valid one (>0).
		 */
		if (process.getPid() > 0) {
			String label = UIPlugin.getResourceString("STR_PROCESS_LABEL_LONG");
			label = TString.change(label, "%1", (process.getName() == null || process.getName().length() == 0) ? UIPlugin.getResourceString("UNKNOWN") : process.getName());
			label = TString.change(label, "%2", process.getNode().getName());
			return TString.change(label, "%3", String.valueOf(process.getPid()));
		}
		else {
			String label = UIPlugin.getResourceString("STR_PROCESS_LABEL_LONG_NO_PID");
			label = TString.change(label, "%1", (process.getName() == null || process.getName().length() == 0) ? UIPlugin.getResourceString("UNKNOWN") : process.getName());
			return TString.change(label, "%2", process.getNode().getName());
		}
	}
}
