/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: LogNavigatorSynchronizer.java,v 1.14 2005/04/22 16:22:50 apnan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.log.ui.internal.navigator;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.IPath;
import org.eclipse.swt.widgets.Display;

import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;

import org.eclipse.hyades.models.hierarchy.CorrelationContainerProxy;
import org.eclipse.hyades.models.hierarchy.HierarchyPackage;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.models.hierarchy.util.HierarchyResourceSetImpl;
import org.eclipse.hyades.models.hierarchy.util.IHyadesExtendedResource;
import org.eclipse.hyades.trace.internal.ui.PDContentProvider;
import org.eclipse.hyades.trace.internal.ui.TraceConstants;
import org.eclipse.hyades.trace.ui.HyadesConstants;
import org.eclipse.hyades.trace.ui.ProfileEvent;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.util.DeleteUtil;
import org.eclipse.hyades.ui.internal.logicalfolder.LogicalFolder;
import org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater;
import org.eclipse.hyades.ui.internal.provider.ResourceChangeUpdaterProvider;
import org.eclipse.hyades.ui.util.IDisposable;

/**
 * @author marcelop
 * @since 1.1.0
 */
public class LogNavigatorSynchronizer
extends AdapterImpl implements IDisposable, IResourceChangeUpdater
{
	private LogNavigator navigator;
	
	private boolean active;
	private Collection addedResources;
	private Collection removedResources;
	private Collection changedResources;
	private Collection adaptedObjects = new ArrayList();
	private Collection resourceExtensions = new ArrayList();
	{
		resourceExtensions.add(TraceConstants.MONITOR_EXT);
		resourceExtensions.add(TraceConstants.NODE_EXT);
		resourceExtensions.add(TraceConstants.PROCESS_EXT);
		resourceExtensions.add(TraceConstants.AGENT_EXT);
		resourceExtensions.add(LogContentProvider.CORR_FILE_EXTENSION);
		resourceExtensions.add(LogContentProvider.SD_FILE_EXTENSION);
		resourceExtensions.add(LogContentProvider.SYMPTOM_FILE_EXTENSION);
	}
	
	public LogNavigatorSynchronizer(LogNavigator navigator)
	{
		this.navigator = navigator;
		active = true;
	}
	
	/**
	 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
	 */
	public void dispose()
	{		
		for (Iterator iter = adaptedObjects.iterator(); iter.hasNext();) {
			Notifier element = (Notifier)iter.next();
			element.eAdapters().remove(this);
		}
		adaptedObjects.clear();
		navigator = null;
	}
	
	public void register(ResourceChangeUpdaterProvider updaterProvider, ResourceSet resourceSet)
	{
		if(updaterProvider != null)
			updaterProvider.setResourceChangeUpdater(this);
			
		if(resourceSet != null)
		{
			if(resourceSet.eAdapters().add(this))
			{
				for(Iterator i = resourceSet.getResources().iterator(); i.hasNext();)
					handleAdd((Resource)i.next());
			}
		}
	}
	
	/**
	 * @see org.eclipse.emf.common.notify.impl.AdapterImpl#notifyChanged(org.eclipse.emf.common.notify.Notification)
	 */
	public void notifyChanged(Notification msg)
	{
		if(!isActive())
			return;
			
		switch(msg.getEventType())
		{
			case Notification.ADD:
			{
				Object newValue = msg.getNewValue();
				if(newValue instanceof Resource)
					handleAdd((Resource)newValue);
				else if(newValue instanceof TRCProcessProxy)
					handleAdd((TRCProcessProxy)newValue);
				else if(newValue instanceof TRCAgentProxy)
					handleAdd((TRCAgentProxy)newValue);
				else if(newValue instanceof TRCMonitor)
					handleAdd((TRCMonitor)newValue);
				else if(newValue instanceof CorrelationContainerProxy)
					handleAdd((CorrelationContainerProxy)newValue);	
				break;
			}
				
			case Notification.ADD_MANY:
			{
				Collection newValues = (Collection)msg.getNewValue();
				for (Iterator i = newValues.iterator(); i.hasNext();)
				{
					Object newValue = i.next();
					if(newValue instanceof Resource)
						handleAdd((Resource)newValue);
					else if(newValue instanceof TRCProcessProxy)
						handleAdd((TRCProcessProxy)newValue);
					else if(newValue instanceof TRCAgentProxy)
						handleAdd((TRCAgentProxy)newValue);
					else if(newValue instanceof TRCMonitor)
						handleAdd((TRCMonitor)newValue);
					else if(newValue instanceof CorrelationContainerProxy)
						handleAdd((CorrelationContainerProxy)newValue);
				}
				break;
			}

			case Notification.REMOVE:
			{
				Object oldValue = msg.getOldValue();

				if(oldValue instanceof Resource)
					handleRemove((Resource)oldValue);
				else if(oldValue instanceof TRCProcessProxy)
					handleRemove((TRCProcessProxy)oldValue);
				else if(oldValue instanceof TRCAgentProxy)
					handleRemove((TRCAgentProxy)oldValue);
				else if(oldValue instanceof TRCMonitor){	
					handleRemove((TRCMonitor)oldValue);
				}					
				else if(oldValue instanceof CorrelationContainerProxy){
					handleRemove((CorrelationContainerProxy)oldValue);
				}

				if(msg.getNotifier() instanceof CorrelationContainerProxy && msg.getFeature() == HierarchyPackage.eINSTANCE.getCorrelationContainerProxy_CorrelatedAgents())
				{				
					CorrelationContainerProxy correlationContainerProxy = (CorrelationContainerProxy)msg.getNotifier();
					if(correlationContainerProxy.getCorrelatedAgents().size()==0)
						correlationContainerProxy.setMonitor(null);
					getNavigator().getViewer().refresh(msg.getNotifier());
				}
				break;
			}

			case Notification.REMOVE_MANY:
				Collection oldValues = (Collection)msg.getOldValue();
				for (Iterator i = oldValues.iterator(); i.hasNext();)
				{
					Object oldValue = i.next();
					if(oldValue instanceof Resource)
						handleRemove((Resource)oldValue);
					else if(oldValue instanceof TRCProcessProxy)
						handleRemove((TRCProcessProxy)oldValue);
					else if(oldValue instanceof TRCAgentProxy)
						handleRemove((TRCAgentProxy)oldValue);
					else if(oldValue instanceof TRCMonitor){	
						handleRemove((TRCMonitor)oldValue);
					}
					else if(oldValue instanceof CorrelationContainerProxy)
						handleRemove((CorrelationContainerProxy)oldValue);
				}
				break;
				
			case Notification.SET:
				if(msg.getNotifier() instanceof TRCAgentProxy)
					handleSet((TRCAgentProxy)msg.getNotifier());
				else if(msg.getNotifier() instanceof CorrelationContainerProxy){
					refreshTree(msg.getNotifier());
				}
				break;
		}
	}
	
	protected void handleRemove(TRCMonitor proxy)
	{
		proxy.eAdapters().remove(this);
		adaptedObjects.remove(proxy);
		for(Iterator i = proxy.getCorrelationContainerProxies().iterator(); i.hasNext();)
			handleRemove((CorrelationContainerProxy)i.next());
	}

	/**
	 * @param proxy
	 */
	protected void handleRemove(CorrelationContainerProxy proxy) {
		
		LogicalFolder corrLogicalFolder = navigator.getContentProvider().getCorrelationFolder();
		proxy.eAdapters().remove(this);
		adaptedObjects.remove(proxy);
		if(corrLogicalFolder.getChildren().remove(proxy)){
			removeFromTree(proxy);
		}
		
	}

	protected void handleAdd(Resource resource)
	{
		if(TraceConstants.PROCESS_EXT.equals(resource.getURI().fileExtension()))
		{
			if(resource.eAdapters().add(this))			
			{
				adaptedObjects.add(resource);
				for (Iterator i = resource.getContents().iterator(); i.hasNext();)
				{
					EObject eObject = (EObject)i.next();
					if(eObject instanceof TRCProcessProxy)
						handleAdd((TRCProcessProxy)eObject);
				}
			}
		}else if(TraceConstants.MONITOR_EXT.equals(resource.getURI().fileExtension())){
			if(resource.eAdapters().add(this)){
				adaptedObjects.add(resource);
				for (Iterator i = resource.getContents().iterator(); i.hasNext();)
				{
					EObject eObject = (EObject)i.next();
					if(eObject instanceof TRCMonitor)
						handleAdd((TRCMonitor)eObject);
				}
			}
		}
	}
	
	/**
	 * @param monitor
	 */
	private void handleAdd(TRCMonitor monitor) {
		if(monitor.eAdapters().add(this))
		{
			adaptedObjects.add(monitor);
			for(Iterator i = monitor.getCorrelationContainerProxies().iterator(); i.hasNext();)
				handleAdd((CorrelationContainerProxy)i.next());
		}

		
	}

	/**
	 * @param proxy
	 */
	private void handleAdd(CorrelationContainerProxy proxy) {

		LogicalFolder corrLogicalFolder = navigator.getContentProvider().getCorrelationFolder();
		if(proxy.eAdapters().add(this))
		{
			adaptedObjects.add(proxy);
			if(!corrLogicalFolder.getChildren().contains(proxy))
			{
				corrLogicalFolder.getChildren().add(proxy);
				addToTree(navigator.getContentProvider().getCorrelationFolder(), proxy);
			}
		}
	}

	protected void handleAdd(TRCProcessProxy processProxy)
	{
		if(processProxy.eAdapters().add(this))
		{
			adaptedObjects.add(processProxy);
			for(Iterator i = processProxy.getAgentProxies().iterator(); i.hasNext();)
				handleAdd((TRCAgentProxy)i.next());
		}
	}

	protected void handleAdd(TRCAgentProxy agentProxy)
	{
		if(HyadesConstants.LOG_AGENT_TYPE.equals(agentProxy.getType()))
		{
			LogicalFolder logLogicalFolder = navigator.getContentProvider().getLogFolder();
			refreshTree(logLogicalFolder);
		}
	}

	protected void handleRemove(Resource resource)
	{
				
		if(TraceConstants.PROCESS_EXT.equals(resource.getURI().fileExtension()))
		{
			resource.eAdapters().remove(this);
			adaptedObjects.remove(resource);
			for (Iterator i = resource.getContents().iterator(); i.hasNext();)
			{
				EObject eObject = (EObject)i.next();
				if(eObject instanceof TRCProcessProxy)
					handleRemove((TRCProcessProxy)eObject);
			}
		}else if(TraceConstants.MONITOR_EXT.equals(resource.getURI().fileExtension())){
			adaptedObjects.remove(resource);
			for (Iterator i = resource.getContents().iterator(); i.hasNext();)
			{
				EObject eObject = (EObject)i.next();
				if(eObject instanceof TRCMonitor)
					handleRemove((TRCMonitor)eObject);
			}
			
		}else{
			if(TraceConstants.AGENT_EXT.equals(resource.getURI().fileExtension()))
			{
				resource.eAdapters().remove(this);
				adaptedObjects.remove(resource);
				for (Iterator i = resource.getContents().iterator(); i.hasNext();)
				{
					EObject eObject = (EObject)i.next();
					if(eObject instanceof TRCAgentProxy)
						handleRemove((TRCAgentProxy)eObject);
				}
			}		
		}
	}
	
	protected void handleRemove(TRCProcessProxy processProxy)
	{
		processProxy.eAdapters().remove(this);
		adaptedObjects.remove(processProxy);
		for(Iterator i = processProxy.getAgentProxies().iterator(); i.hasNext();)
			handleRemove((TRCAgentProxy)i.next());
	}

	protected void handleRemove(TRCAgentProxy agentProxy)
	{
		if(HyadesConstants.LOG_AGENT_TYPE.equals(agentProxy.getType()))
		{			
			LogicalFolder logLogicalFolder = navigator.getContentProvider().getLogFolder();
			refreshTree(logLogicalFolder);					
		}
	}
	
	protected void handleSet(TRCAgentProxy agentProxy)
	{
		if(HyadesConstants.LOG_AGENT_TYPE.equals(agentProxy.getType()))
			refreshTree(agentProxy);
	}
	
	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#setActive(boolean)
	 */
	public void setActive(boolean enable)
	{
		active = enable;
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#isActive()
	 */
	public boolean isActive()
	{
		return active;
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#started()
	 */
	public void started()
	{
		addedResources = new ArrayList();
		removedResources = new ArrayList();
		changedResources = new ArrayList();
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#ended()
	 */
	public void ended()
	{
		LogicalFolder sdFolder = navigator.getContentProvider().getSDFolder();
		int initialSize = sdFolder.getChildren().size();
		Collection sdbList = null;
		if(!removedResources.isEmpty()){
			for(Iterator i = removedResources.iterator(); i.hasNext();)
			{
				IResource resource = (IResource)i.next();
				//a file resource was deleted
				if(resource.getType() == IResource.FILE){
					//check if it's an sdb
					if(LogContentProvider.SD_FILE_EXTENSION.equals(resource.getFileExtension()) || LogContentProvider.SYMPTOM_FILE_EXTENSION.equals(resource.getFileExtension())){
						//remove the sdb from the log navigator
						sdFolder.getChildren().remove(resource);
						if(sdbList==null){
							sdbList = new ArrayList();
						}
						sdbList.add(resource);
					}
				}
			}
		}
		boolean doRefresh = (initialSize != sdFolder.getChildren().size());		

		if(DeleteUtil.isDeleteInProgress()){
			
			if(doRefresh)
				refreshTree(sdFolder);
			
			return;
		}
	
		if(sdbList!=null){
			//sdb resources were already handled
			removedResources.removeAll(sdbList);
		}
		boolean globalRefresh = false;
		
		if(!removedResources.isEmpty())
		{
			PDContentProvider.resetMonitors();
			for(Iterator i = removedResources.iterator(); i.hasNext();)
			{
				IResource resource = (IResource)i.next();
				//a file resource was deleted
				if(resource.getType() == IResource.FILE){
						
					EObject[] eObjects = EMFUtil.getEObjects(HierarchyResourceSetImpl.getInstance(), (IFile)resource);
					for(int j = 0, maxj = eObjects.length; j < maxj; j++)
					{
						DeleteUtil.deleteEObject(eObjects[j],true,false,true,getNavigator().getID());
					}


				}else{
					//a project or folder was deleted - cleanup all emf resources underneath that project/folder
					IPath fullPath = resource.getFullPath();
					IResource[] resources = (IResource[])sdFolder.getChildren().toArray(new IResource[sdFolder.getChildren().size()]);
					Collection sdbsToRemove = new ArrayList();
					for(int j = 0, maxj = resources.length; j < maxj; j++)
					{
						if(fullPath.isPrefixOf(resources[j].getFullPath())){
							sdbsToRemove.add(resources[j]);
						}

					}
					if(!sdbsToRemove.isEmpty()){
						sdFolder.getChildren().removeAll(sdbsToRemove);
					}
					Resource[] emfResources = EMFUtil.getResources(fullPath, HierarchyResourceSetImpl.getInstance());
					for(int j = 0, maxj = emfResources.length; j < maxj; j++)
					{
						if(emfResources[j]!=null && emfResources[j] instanceof IHyadesExtendedResource){
							((IHyadesExtendedResource)emfResources[j]).delete();
							globalRefresh = true;
						}

					}					
				}
			}
			
			//sdFolder.getChildren().removeAll(removedResources);
			doRefresh = (initialSize != sdFolder.getChildren().size());
		}

		if(!addedResources.isEmpty())
		{
			sdFolder.getChildren().addAll(addedResources);
			if(!doRefresh)
				addToTree(sdFolder, addedResources.toArray());
		}

		if(!changedResources.isEmpty())
		{
			if(!doRefresh)
				refreshTree(changedResources.toArray());
		}
		
		if(doRefresh)
			refreshTree(sdFolder);
		
		if(globalRefresh){
			ProfileEvent event = UIPlugin.getDefault().getProfileEvent();
			event.setSource(null);
			event.setType(ProfileEvent.UNSPECIFIED);			
			UIPlugin.getDefault().notifyProfileEventListener(event);			
		}
		
		addedResources.clear();
		addedResources = null;
		removedResources.clear();
		removedResources = null;
		changedResources.clear();
		changedResources = null;		
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#add(org.eclipse.core.resources.IResource, org.eclipse.core.resources.IResource[])
	 */
	public boolean add(IResource parent, IResource[] affectedResources)
	{
		for (int i = 0, maxi = affectedResources.length; i < maxi; i++)
		{
			if(LogContentProvider.SD_FILE_EXTENSION.equals(affectedResources[i].getFileExtension())
			|| LogContentProvider.SYMPTOM_FILE_EXTENSION.equals(affectedResources[i].getFileExtension()))
			{
				if(navigator.getContentProvider().getSDFolder().getChildren().add(affectedResources[i]))
					addedResources.add(affectedResources[i]);
			}
		}
		return false;
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#remove(org.eclipse.core.resources.IResource, org.eclipse.core.resources.IResource[])
	 */
	public boolean remove(IResource parent, IResource[] affectedResources)
	{
		for (int i = 0, maxi = affectedResources.length; i < maxi; i++)
		{
			if(affectedResources[i].getType() != IResource.FILE)
			{
				removedResources.add(affectedResources[i]);
			}
			else if(resourceExtensions.contains(affectedResources[i].getFileExtension()))
			{
				//if(navigator.getContentProvider().getSDFolder().getChildren().remove(affectedResources[i])){					
					removedResources.add(affectedResources[i]);
				//}

			}
		}
		return false;
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#replaced(org.eclipse.core.resources.IResource)
	 */
	public boolean replaced(IResource affectedResource)
	{
		return false;
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#updateChildrenType(org.eclipse.core.resources.IResource)
	 */
	public boolean updateChildrenType(IResource affectedResource)
	{
		return false;
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#updateContent(org.eclipse.core.resources.IResource, org.eclipse.core.resources.IResource)
	 */
	public boolean updateContent(IResource parent, IResource affectedResource)
	{
		return false;
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#updateProperties(org.eclipse.core.resources.IResource)
	 */
	public boolean updateProperties(IResource affectedResource)
	{
		if(LogContentProvider.SD_FILE_EXTENSION.equals(affectedResource.getFileExtension())
				|| LogContentProvider.SYMPTOM_FILE_EXTENSION.equals(affectedResource.getFileExtension()))
			changedResources.add(affectedResource);
		return false;
	}
	
	protected void addToTree(final Object parent, final Object element)
	{
		Display.getDefault().asyncExec(new Runnable()
		{
			public void run()
			{
				getNavigator().getTreeViewer().add(parent, element);
			}
		});
	}

	protected void addToTree(final Object parent, final Object[] elements)
	{
		Display.getDefault().asyncExec(new Runnable()
		{
			public void run()
			{
				LogNavigator navigator = getNavigator();
				if(navigator!=null){
					navigator.getTreeViewer().add(parent, elements);
				}
				
			}
		});
	}

	protected void removeFromTree(final Object element)
	{
		Display.getDefault().syncExec(new Runnable()
		{
			public void run()
			{
				if(navigator!=null && navigator.getTreeViewer() != null && !navigator.getTreeViewer().getControl().isDisposed()){
					navigator.getTreeViewer().remove(element);
				}				
			}
		});
	}

	protected void removeFromTree(final Object[] elements)
	{
		Display.getCurrent().asyncExec(new Runnable()
		{
			public void run()
			{
				if(navigator!=null && navigator.getTreeViewer() != null && !navigator.getTreeViewer().getControl().isDisposed()){
					navigator.getTreeViewer().remove(elements);
				}
			}
		});
	}
	
	protected void refreshTree(final Object element)
	{
		Display.getDefault().asyncExec(new Runnable()
		{
			public void run()
			{
				if(navigator!=null && navigator.getTreeViewer() != null && !navigator.getTreeViewer().getControl().isDisposed()){
					navigator.getTreeViewer().refresh(element, true);
				}				
			}
		});
	}
	
	protected void refreshTree(final Object[] elements)
	{
		Display.getDefault().asyncExec(new Runnable()
		{
			public void run()
			{
				for (int i = 0, maxi = elements.length; i < maxi; i++)
					getNavigator().getTreeViewer().refresh(elements[i], true);
			}
		});
	}
	
	protected LogNavigator getNavigator()
	{
		return navigator;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#refreshContent(org.eclipse.core.resources.IResource)
	 */
	public void refreshContent(IResource affectedResource) {
		// TODO Auto-generated method stub
		
	}
	
}
