/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: CbeEmfUtility.java,v 1.3 2005/02/16 22:20:28 qiyanli Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.log.ui.internal.util;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.List;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.EMap;
import org.eclipse.hyades.loaders.cbe.CBEUtils;
import org.eclipse.hyades.logging.events.cbe.AssociatedEvent;
import org.eclipse.hyades.logging.events.cbe.AssociationEngine;
import org.eclipse.hyades.logging.events.cbe.AvailableSituation;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.ComponentIdentification;
import org.eclipse.hyades.logging.events.cbe.ConfigureSituation;
import org.eclipse.hyades.logging.events.cbe.ConnectSituation;
import org.eclipse.hyades.logging.events.cbe.ContextDataElement;
import org.eclipse.hyades.logging.events.cbe.CreateSituation;
import org.eclipse.hyades.logging.events.cbe.DependencySituation;
import org.eclipse.hyades.logging.events.cbe.DestroySituation;
import org.eclipse.hyades.logging.events.cbe.EventFactory;
import org.eclipse.hyades.logging.events.cbe.ExtendedDataElement;
import org.eclipse.hyades.logging.events.cbe.FeatureSituation;
import org.eclipse.hyades.logging.events.cbe.MsgCatalogToken;
import org.eclipse.hyades.logging.events.cbe.MsgDataElement;
import org.eclipse.hyades.logging.events.cbe.OtherSituation;
import org.eclipse.hyades.logging.events.cbe.ReportSituation;
import org.eclipse.hyades.logging.events.cbe.RequestSituation;
import org.eclipse.hyades.logging.events.cbe.Situation;
import org.eclipse.hyades.logging.events.cbe.StartSituation;
import org.eclipse.hyades.logging.events.cbe.StopSituation;
import org.eclipse.hyades.logging.events.cbe.impl.EventFactoryContext;
import org.eclipse.hyades.models.cbe.CBEAvailableSituation;
import org.eclipse.hyades.models.cbe.CBECommonBaseEvent;
import org.eclipse.hyades.models.cbe.CBEComponentIdentification;
import org.eclipse.hyades.models.cbe.CBEConfigureSituation;
import org.eclipse.hyades.models.cbe.CBEConnectSituation;
import org.eclipse.hyades.models.cbe.CBEContextDataElement;
import org.eclipse.hyades.models.cbe.CBECreateSituation;
import org.eclipse.hyades.models.cbe.CBEDependencySituation;
import org.eclipse.hyades.models.cbe.CBEDestroySituation;
import org.eclipse.hyades.models.cbe.CBEExtendedDataElement;
import org.eclipse.hyades.models.cbe.CBEFeatureSituation;
import org.eclipse.hyades.models.cbe.CBEMsgDataElement;
import org.eclipse.hyades.models.cbe.CBEOtherSituation;
import org.eclipse.hyades.models.cbe.CBEReportSituation;
import org.eclipse.hyades.models.cbe.CBERequestSituation;
import org.eclipse.hyades.models.cbe.CBESituation;
import org.eclipse.hyades.models.cbe.CBEStartSituation;
import org.eclipse.hyades.models.cbe.CBEStopSituation;
import org.eclipse.hyades.models.hierarchy.CorrelationContainerProxy;
import org.eclipse.hyades.models.hierarchy.CorrelationEngine;

/**
 * Utility for converting Common Base Events
 */
public class CbeEmfUtility {

	/**
	 * Factory for creating Common Base Events
	 */
	protected static EventFactory _eventFactory = EventFactoryContext.getInstance().getEventFactoryHome("org.eclipse.hyades.logging.events.cbe.impl.SimpleEventFactoryHomeImpl").getEventFactory("org.eclipse.hyades.logging.ui");
	
	/**
	 * Converts a CBECommonBaseEvent EMF object to a Logging Util ICommonBaseEvent object
	 * 
	 * @param cbe the EMF CBE record
	 * @return the Logging Util CBE record
	 */
	public static CommonBaseEvent convertCbe(CBECommonBaseEvent cbe, Hashtable associationMaps) {
		// getEventFactory() initializes the event factory
		CommonBaseEvent cbe2 = _eventFactory.createCommonBaseEvent();
		
		cbe2.setLocalInstanceId(cbe.getLocalInstanceId());
		
		cbe2.setGlobalInstanceId(cbe.getGlobalInstanceId());
		
		cbe2.setCreationTime(CbeEmfUtility.formatCreationTime(cbe));
		
		cbe2.setSeverity(cbe.getSeverity());
		
		cbe2.setPriority(cbe.getPriority());
		
		// ReporterComponentId
		CBEComponentIdentification rcid = cbe.getReporterComponentId();
		if (rcid != null) {
			ComponentIdentification rcid2 = _eventFactory.createComponentIdentification();
			cbe2.setReporterComponentId(rcid2);
			
			rcid2.setLocation(rcid.getLocation());
			rcid2.setLocationType(rcid.getLocationType());
			rcid2.setApplication(rcid.getApplication());
			rcid2.setExecutionEnvironment(rcid.getExecutionEnvironment());
			rcid2.setComponent(rcid.getComponent());
			rcid2.setSubComponent(rcid.getSubComponent());
			rcid2.setComponentIdType(rcid.getComponentIdType());
			rcid2.setInstanceId(rcid.getInstanceId());
			rcid2.setProcessId(rcid.getProcessId());
			rcid2.setThreadId(rcid.getThreadId());
			rcid2.setComponentType(rcid.getComponentType());
		}
		
		// SourceComponentId
		CBEComponentIdentification scid = cbe.getSourceComponentId();
		if (scid != null) {
			ComponentIdentification scid2 = _eventFactory.createComponentIdentification();
			cbe2.setSourceComponentId(scid2);
			
			scid2.setLocation(scid.getLocation());
			scid2.setLocationType(scid.getLocationType());
			scid2.setApplication(scid.getApplication());
			scid2.setExecutionEnvironment(scid.getExecutionEnvironment());
			scid2.setComponent(scid.getComponent());
			scid2.setSubComponent(scid.getSubComponent());
			scid2.setComponentIdType(scid.getComponentIdType());
			scid2.setInstanceId(scid.getInstanceId());
			scid2.setProcessId(scid.getProcessId());
			scid2.setThreadId(scid.getThreadId());
			scid2.setComponentType(scid.getComponentType());
		}
		
		// Situation
		CBESituation sit = cbe.getSituation();
		if (sit != null) {
			Situation sit2 = _eventFactory.createSituation();
			cbe2.setSituation(sit2);

			sit2.setCategoryName(sit.getCategoryName());

			if (sit instanceof CBEStartSituation) {
				CBEStartSituation sittype = (CBEStartSituation) sit;
				StartSituation sittype2 = _eventFactory.createStartSituation();
				sit2.setSituationType(sittype2);	

				sittype2.setReasoningScope(sittype.getReasoningScope());
				sittype2.setSuccessDisposition(sittype.getSuccessDisposition());
				sittype2.setSituationQualifier(sittype.getSituationQualifier());
			}
			else if (sit instanceof CBEStopSituation) {
				CBEStopSituation sittype = (CBEStopSituation) sit;
				StopSituation sittype2 = _eventFactory.createStopSituation();
				sit2.setSituationType(sittype2);	

				sittype2.setReasoningScope(sittype.getReasoningScope());
				sittype2.setSuccessDisposition(sittype.getSuccessDisposition());
				sittype2.setSituationQualifier(sittype.getSituationQualifier());	
			}
			else if (sit instanceof CBEConnectSituation) {
				CBEConnectSituation sittype = (CBEConnectSituation) sit;
				ConnectSituation sittype2 = _eventFactory.createConnectSituation();
				sit2.setSituationType(sittype2);	

				sittype2.setReasoningScope(sittype.getReasoningScope());
				sittype2.setSuccessDisposition(sittype.getSuccessDisposition());
				sittype2.setSituationDisposition(sittype.getSituationDisposition());	
			}
			else if (sit instanceof CBERequestSituation) {
				CBERequestSituation sittype = (CBERequestSituation) sit;
				RequestSituation sittype2 = _eventFactory.createRequestSituation();
				sit2.setSituationType(sittype2);	

				sittype2.setReasoningScope(sittype.getReasoningScope());
				sittype2.setSuccessDisposition(sittype.getSuccessDisposition());
				sittype2.setSituationQualifier(sittype.getSituationQualifier());	
			}
			else if (sit instanceof CBEConfigureSituation) {
				CBEConfigureSituation sittype = (CBEConfigureSituation) sit;
				ConfigureSituation sittype2 = _eventFactory.createConfigureSituation();
				sit2.setSituationType(sittype2);	

				sittype2.setReasoningScope(sittype.getReasoningScope());
				sittype2.setSuccessDisposition(sittype.getSuccessDisposition());
			}
			else if (sit instanceof CBEAvailableSituation) {
				CBEAvailableSituation sittype = (CBEAvailableSituation) sit;
				AvailableSituation sittype2 = _eventFactory.createAvailableSituation();
				sit2.setSituationType(sittype2);	

				sittype2.setReasoningScope(sittype.getReasoningScope());
				sittype2.setOperationDisposition(sittype.getOperationDisposition());
				sittype2.setAvailabilityDisposition(sittype.getAvailabilityDisposition());
				sittype2.setProcessingDisposition(sittype.getProcessingDisposition());
			}
			else if (sit instanceof CBEReportSituation) {
				CBEReportSituation sittype = (CBEReportSituation) sit;
				ReportSituation sittype2 = _eventFactory.createReportSituation();
				sit2.setSituationType(sittype2);	

				sittype2.setReasoningScope(sittype.getReasoningScope());
				sittype2.setReportCategory(sittype.getReportCategory());
			}
			else if (sit instanceof CBECreateSituation) {
				CBECreateSituation sittype = (CBECreateSituation) sit;
				CreateSituation sittype2 = _eventFactory.createCreateSituation();
				sit2.setSituationType(sittype2);	

				sittype2.setReasoningScope(sittype.getReasoningScope());
				sittype2.setSuccessDisposition(sittype.getSuccessDisposition());
			}
			else if (sit instanceof CBEDestroySituation) {
				CBEDestroySituation sittype = (CBEDestroySituation) sit;
				DestroySituation sittype2 = _eventFactory.createDestroySituation();
				sit2.setSituationType(sittype2);	

				sittype2.setReasoningScope(sittype.getReasoningScope());
				sittype2.setSuccessDisposition(sittype.getSuccessDisposition());
			}
			else if (sit instanceof CBEFeatureSituation) {
				CBEFeatureSituation sittype = (CBEFeatureSituation) sit;
				FeatureSituation sittype2 = _eventFactory.createFeatureSituation();
				sit2.setSituationType(sittype2);	

				sittype2.setReasoningScope(sittype.getReasoningScope());
				sittype2.setFeatureDisposition(sittype.getFeatureDisposition());
			}
			else if (sit instanceof CBEDependencySituation) {
				CBEDependencySituation sittype = (CBEDependencySituation) sit;
				DependencySituation sittype2 = _eventFactory.createDependencySituation();
				sit2.setSituationType(sittype2);	

				sittype2.setReasoningScope(sittype.getReasoningScope());
				sittype2.setDependencyDisposition(sittype.getDependencyDisposition());
			}
			else if (sit instanceof CBEOtherSituation) {
				CBEOtherSituation sittype = (CBEOtherSituation) sit;
				OtherSituation sittype2 = _eventFactory.createOtherSituation();
				sit2.setSituationType(sittype2);	

				sittype2.setReasoningScope(sittype.getReasoningScope());
			}
		}
		
		// ContextDataElements
		{
			EList ctes = cbe.getContextDataElements();
			ContextDataElement[] ctes2 = new ContextDataElement[ctes.size()];
			
			for (int i = 0; i < ctes.size(); ++i) {
				CBEContextDataElement cte = (CBEContextDataElement) ctes.get(i);
				ContextDataElement cte2 = _eventFactory.createContextDataElement();
				
				cte2.setType(cte.getType());
				cte2.setName(cte.getName());
				cte2.setContextValue(cte.getContextValue());
				cte2.setContextId(cte.getContextId());
				
				ctes2[i] = cte2;
			}
		}
		
		// MsgDataElement
		CBEMsgDataElement mde = cbe.getMsgDataElement();
		if (mde != null) {
			MsgDataElement mde2 = _eventFactory.createMsgDataElement();
			cbe2.setMsgDataElement(mde2);
			
			mde2.setMsgId(mde.getMsgId());
			mde2.setMsgIdType(mde.getMsgIdType());
			mde2.setMsgCatalogId(mde.getMsgCatalogId());
			
			EList msgCatalogTokens = mde.getMsgCatalogTokens();
			MsgCatalogToken[] msgCatalogTokens2 = new MsgCatalogToken[msgCatalogTokens.size()];
			
			for (int i = 0; i < msgCatalogTokens.size(); ++i) {
				String msgCatalogToken = (String) msgCatalogTokens.get(i);
				MsgCatalogToken msgCatalogToken2 = _eventFactory.createMsgCatalogToken();
				
				msgCatalogToken2.setValue(msgCatalogToken);
				
				msgCatalogTokens2[i] = msgCatalogToken2;		
			}
			
			mde2.setMsgCatalogTokens(msgCatalogTokens2);
			mde2.setMsgCatalog(mde.getMsgCatalog());
			mde2.setMsgCatalogType(mde.getMsgCatalogType());
			mde2.setMsgLocale(mde.getMsgLocale());
		}
		
		cbe2.setMsg(cbe.getMsg());
		
		cbe2.setRepeatCount(cbe.getRepeatCount());
		
		cbe2.setElapsedTime(cbe.getElapsedTime());
		
		// AssociatedEvents
		{
			Enumeration keys = associationMaps.keys();
			while (keys.hasMoreElements()) {
				CorrelationContainerProxy corProxy = (CorrelationContainerProxy)(keys.nextElement());
				if(corProxy!=null && corProxy.getCorrelationContainer()!=null){
					EMap correlations = corProxy.getCorrelationContainer().getCorrelations();
					if(correlations!=null && correlations.get(cbe)!=null){
						AssociationEngine asoEgn = (AssociationEngine) associationMaps.get(corProxy);
						
						AssociatedEvent asoEvt = _eventFactory.createAssociatedEvent();
						asoEvt.setAssociationEngine(asoEgn.getId());
						cbe2.getAssociatedEvents().add(asoEvt);
						
						List resolvedEvents = (List) correlations.get(cbe);
						for (int i = 0; i < resolvedEvents.size(); i++) {
							Object cbeEvent = resolvedEvents.get(i);
							if (cbeEvent instanceof CBECommonBaseEvent){
								asoEvt.addResolvedEvent(((CBECommonBaseEvent)cbeEvent).getGlobalInstanceId());							
							}
						}
					}
				}
			}
		}

		
		cbe2.setExtensionName(cbe.getExtensionName());
		
		// ExtendedDataElemets
		{
			EList edes = cbe.getExtendedProperties();
	
			for (int i = 0; i < edes.size(); ++i) {
				if (edes.get(i) instanceof CBEExtendedDataElement) {
					CBEExtendedDataElement ede = (CBEExtendedDataElement) edes.get(i);
					ExtendedDataElement ede2 = _eventFactory.createExtendedDataElement();
					cbe2.addExtendedDataElement(ede2);
					
					setEdeFields(ede, ede2);
				}
			}
		}
		
		cbe2.setSequenceNumber(cbe.getSequenceNumber());
		
		return cbe2;
	}

	/**
	 * Sets the fields of an Extended Data Element
	 * 
	 * @param ede The EMF CBEExtendedDataElement object (source)
	 * @param ede2 The Logging utils IExtendedDataElement object (target)
	 */
	protected static void setEdeFields(CBEExtendedDataElement ede, ExtendedDataElement ede2) {
		ede2.setName(ede.getName());
		ede2.setType(ede.getType());
		
		EList values = ede.getValues();
		if (values.size() > 0) {
			String[] values2 = new String[values.size()];
			for (int j = 0; j < values.size(); ++j) {
				String value = (String) values.get(j);
				values2[j] = value;
			}
			ede2.setValues(values2);
		}
		
		String hexValue = ede.getHexValue();
		if (hexValue != null) {
			ede2.setHexValue(hexValue);
		}
		
		EList children = ede.getChildren();
		for (int j = 0; j < children.size(); ++j) {
			CBEExtendedDataElement childEde = (CBEExtendedDataElement) children.get(j);
			ExtendedDataElement childEde2 = _eventFactory.createExtendedDataElement();
			ede2.addChild(childEde2);
			
			setEdeFields(childEde, childEde2);
		}
	}

	
	
	/**
	 * Formats the creation time of EMF CBECommonBaseEvent object into xsd:dateTime format
	 * 
	 * @param event the EMF CBE record
	 * @return the xsd:dateTime format string
	 */
	protected static String formatCreationTime(CBECommonBaseEvent event) {
		String res="0";
		
		// if time is not specified, use "0"
		if (event.getCreationTime() != 0)
		{
			try {
				short value=0;
				try {
					value = event.getTimeZone();
				} catch (Exception e) {
//					Utilities.printStackTrace(e);
				}
				
				double ct = CBEUtils.computeAdjustedCreationTime(event);
				Double l2 = new Double(ct);
				
				//SimpleDateFormat dateFormat = LoadersUtils.getPartialCIMDateFormat();
				SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd" + " " + "HH:mm:ss");
				
				res = dateFormat.format(new Date(((Double)l2).longValue()/1000 + value*60*1000));
				String microseconds = ""+(((Double)l2).longValue() % 1000000);
				res = res + "."+"000000".substring(0,6-microseconds.length()) + microseconds;
				res = res + (value>0?"+00":"-00").substring(0,3-(""+Math.abs(value/60)).length())+Math.abs(value/60);
				res = res + ":00".substring(0,3-(""+Math.abs(value%60)).length())+Math.abs(value%60);
				res = res.replace(' ', 'T');
			} catch (Exception e) {
//				Utilities.printStackTrace(e);
			}
		}
		return res;
	}

	public static AssociationEngine createAssociationEngine(CorrelationEngine corE) {
		AssociationEngine asoE = _eventFactory.createAssociationEngine();
		asoE.setId(corE.getId());
		asoE.setName(corE.getName());
		asoE.setType(corE.getType());
		return asoE;
	}
	
}
