/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: Splitter.java,v 1.4 2005/02/16 22:20:28 qiyanli Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.log.ui.internal.util;

import org.eclipse.swt.*;
import org.eclipse.swt.widgets.*;
import org.eclipse.swt.graphics.*;

/**
 * 
 */
public class Splitter extends Composite {

	protected SplitterLayout fLayout;
	protected int fDirection;
	protected int fGap= 3;          // size of Sash
	protected boolean fResizable;
	
	static public class Weight {
		public int fSize;       // initial size
		public int fGrow;       // how to scale
		public int fShrink;     // how to scale
		
		public Weight(int size) {
			fSize= size;
			fGrow= 50;
			fShrink= 50;
		}
		
		public Weight(int size, int w) {
			fSize= size;
			fGrow= w;
			fShrink= w;
		}
		
		public Weight(int size, int g, int s) {
			fSize= size;
			fGrow= g;
			fShrink= s;
		}
	};

	/**
	 * Layout for maximized state
	 */
	static public class MaximizeLayout extends Layout {
		Control fChild;
		Point fOldSize;
		
		MaximizeLayout(Control w) {
			fChild= w;
			fOldSize= w.getSize();
		}
		protected Point computeSize(Composite composite, int wHint, int hHint, boolean flushCache) {
			return fChild.getSize();
		}
		protected void layout(Composite composite, boolean flushCache) {
			fChild.setBounds(composite.getClientArea());
		}
	}

	/**
	 * Layout for splitted state
	 */
	public class SplitterLayout extends Layout {
		
		protected Point computeSize(Composite composite, int wHint, int hHint, boolean flushCache) {
			Control[] children= composite.getChildren();
			int max= 0, total= 0;
			
			if (fDirection == SWT.HORIZONTAL) {
				for (int i= 0; i < children.length; i++) {
					Point pt= children[i].computeSize(SWT.DEFAULT, SWT.DEFAULT, flushCache);
					total+= pt.x;
					max= Math.max(max, pt.y);
				}   
				return new Point(total, max);
			}
			
			for (int i= 0; i < children.length; i++) {
				Point pt= children[i].computeSize(SWT.DEFAULT, SWT.DEFAULT, flushCache);
				total+= pt.y;
				max= Math.max(max, pt.x);
			}
			return new Point(max, total);
		}

		protected void layout(Composite composite, boolean flushCache) {
			Rectangle clientArea= composite.getClientArea();
			Control[] children= composite.getChildren();
			int count= children.length;

			int newSize= (fDirection == SWT.HORIZONTAL) ? clientArea.width : clientArea.height;

			
										//          horizontal  vertical
			int totalSize= 0;           // accum    width   or      height
			int[] size= new int[count]; // holds    width   or      height  per child
			int totalGrow= 0;
			int totalShrink= 0;

			//System.out.println("newsize: " + newSize);

			// get current (or initial) sizes and calculate total size
			for (int i= 0; i < count; i++) {
				int e= getWindowSize(children[i], flushCache);
				//System.out.println("   size: " + e);
				totalSize+= e;
				size[i]= e;
				
				Weight w= getWeight(children[i]);
				totalGrow+= w.fGrow;
				totalShrink+= w.fShrink;
			}
			
			//System.out.println("oldsize: " + totalSize);
			
			int diff= newSize - totalSize;
			//System.out.println("diff: " + diff);
			
			// scale size
			boolean allfixed= true;
			int total= 0;
			for (int i= 0; i < count; i++) {
				Weight w= getWeight(children[i]);
				
				if (diff > 0) {
					if (w.fGrow > 0) {
						int d= (diff * w.fGrow) / totalGrow;
						size[i]+= d;
						allfixed= false;
						//System.out.println("   added[" + i + "]: " + d);
					}
				} else {
					if (w.fShrink > 0) {
						int d= (diff * w.fShrink) / totalShrink;
						size[i]+= d;
						allfixed= false;
					}
				}
				total+= size[i];
			}

			// distribute any rest
			if (!allfixed) {
				for (int i= 0; total < newSize; i++) {
					if (!isFixed(children[i % count])) {
						size[i % count]++;
						total++;
					}
				}
			}

			// position and size children
			if (fDirection == SWT.HORIZONTAL) {
				int x= clientArea.x;
				//System.out.println("setWidth:");
				for (int i= 0; i < count; i++) {
					children[i].setBounds(x, clientArea.y, size[i], clientArea.height);
					x+= size[i];
					//System.out.println("   " + i + ": " + size[i]);
				}
			} else {
				int y= clientArea.y;
				for (int i= 0; i < count; i++) {
					children[i].setBounds(clientArea.x, y, clientArea.width, size[i]);
					y+= size[i];
				}
			}
			
		}
	}
	
/**
 * Splitter constructor comment.
 */
public Splitter(Composite parent, int direction, boolean resizable) {
	super(parent, SWT.NONE);
	fDirection = direction;
	fResizable = resizable;
	fLayout = new SplitterLayout();
	setLayout(fLayout);
}
/**
 * Add a Sash after the last added window.
 */
public Sash addSash() {
	final Sash sash =
		new Sash(this, fDirection == SWT.HORIZONTAL ? SWT.VERTICAL : SWT.HORIZONTAL);
	if (fResizable) {
		sash.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event event) {
				dragSash(sash, event);
			}
		});
	}
	return sash;
}
void dragSash(Sash sash, Event event) {
	Rectangle clientArea = getClientArea();
	Control[] children = getChildren();
	int count = children.length;

	if (event.detail == SWT.DRAG) {
		// constrain feedback
		int limit = 20;
		if (fDirection == SWT.HORIZONTAL) {
			event.x = Math.min(Math.max(limit, event.x), clientArea.width - limit);
		} else {
			event.y = Math.min(Math.max(limit, event.y), clientArea.height - limit);
		}
		return;
	}

	Rectangle bounds = event.getBounds();

	int ix = -1;
	for (int i = 0; i < count; i++) {
		Control child = children[i];
		if (child == sash) {
			ix = i;
			break;
		}
	}

	//Assert.isTrue(ix >= 1 && ix < count-1);

	Control w1 = children[ix - 1];
	Rectangle b1 = w1.getBounds();

	Rectangle oldBounds = sash.getBounds();

	Control w2 = children[ix + 1];
	Rectangle b2 = w2.getBounds();

	if (fDirection == SWT.HORIZONTAL) {
		int shift = bounds.x - oldBounds.x;
		b1.width += shift;
		b2.x += shift;
		b2.width -= shift;
		if (b1.width < 20 || b2.width < 20) {
			return;
		}
	} else {
		int shift = bounds.y - oldBounds.y;
		b1.height += shift;
		b2.y += shift;
		b2.height -= shift;
		if (b1.height < 20 || b2.height < 20) {
			return;
		}
	}

	w1.setBounds(b1.x, b1.y, b1.width, b1.height);
	sash.setBounds(bounds.x, bounds.y, bounds.width, bounds.height);
	w2.setBounds(b2.x, b2.y, b2.width, b2.height);
}
/**
 * Flip pane orientation.
 */
public void flipDirection() {

	fDirection = (fDirection == SWT.HORIZONTAL) ? SWT.VERTICAL : SWT.HORIZONTAL;

	Control[] children = getChildren();
	for (int i = 0; i < children.length; i++) {
		Control w = children[i];
		if (w instanceof Sash) {
			Sash s = addSash();
			s.moveAbove(w);
			w.dispose();
		} else {
			// swap height and width to keep proportions after the flip
			// the layout proportionally adjusts them later
			Point e = w.getSize();
			w.setSize(e.y, e.x);
		}
	}
}
/**
 */
public Point[] getSizes() {
	Control[] children = getChildren();
	Point[] p = new Point[children.length];
	for (int i = 0; i < children.length; i++)
		p[i] = children[i].getSize();
	return p;
}
Weight getWeight(Control window) {
	Object data = window.getLayoutData();
	if (data instanceof Weight)
		return (Weight) data;

	Weight w = null;
	if (window instanceof Sash) {
		Point ee = window.computeSize(SWT.DEFAULT, SWT.DEFAULT, true);
		int e = (fDirection == SWT.HORIZONTAL) ? ee.x : ee.y;
		w = new Weight(e, 0);
	} else {
		if (data instanceof Integer) {
			w = new Weight(((Integer) data).intValue());
		} else {
			w = new Weight(50);
		}
	}

	window.setLayoutData(w);
	return w;
}
int getWindowSize(Control window, boolean flushCache) {

	/*
	if (isFixed(window)) {
		Point s= window.computeSize(SWT.DEFAULT, SWT.DEFAULT, flushCache);
		return Math.min(s.x, s.y);
	}
	*/

	Point ee = window.getSize();

	//Point ee= new Point(0, 0);

	int e = (fDirection == SWT.HORIZONTAL) ? ee.x : ee.y;

	if (e == 0)
		e = getWeight(window).fSize;

	return e;
}
/**
 * Maximize the given window to fill the area of this Splitter.
 * If w is already maximized reset to old layout.
 * If w is not a child of the Splitter strange things happen.
 */
void internalMaximize(Control w) {
	Layout layout = getLayout();
	boolean maximized = layout instanceof MaximizeLayout;

	Control[] ws = getChildren();
	for (int i = 0; i < ws.length; i++)
		if (ws[i] != w)
			ws[i].setVisible(maximized);

	if (maximized) {
		w.setSize(((MaximizeLayout) layout).fOldSize);
		setLayout(fLayout);
	} else {
		setLayout(new MaximizeLayout(w));
	}

	// walk up
	w = getParent();
	if (w instanceof Splitter)
		 ((Splitter) w).internalMaximize(this);
	else
		layout(true);
}
boolean isFixed(Control window) {

	if (window instanceof Sash)
		return true;

	return getWeight(window).fGrow <= 0;
}
/**
 * Maximize the given window to fill the area of this Splitter.
 * If w is already maximized reset to old layout.
 * If w is not a child of the Splitter strange things happen.
 */
public void maximize(Control w) {
	setRedraw(false);
	internalMaximize(w);
	setRedraw(true);
}
/**
 */
public void setSizes(Point[] sizes) {
	setRedraw(false);
	Control[] children = getChildren();
	for (int i = 0; i < sizes.length; i++)
		children[i].setSize(sizes[i]);
	layout(true);
	setRedraw(true);
}
}
