/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: LogContentProvider.java,v 1.45 2005/04/07 23:13:43 ewchan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.log.ui.internal.views;

import java.util.AbstractList;
import java.util.ArrayList;
import java.util.Collections;
import java.util.ConcurrentModificationException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.eclipse.core.resources.IResourceStatus;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.EMap;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.EcoreFactory;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.hyades.loaders.hierarchy.Constants;
import org.eclipse.hyades.log.ui.internal.LogUIPlugin;
import org.eclipse.hyades.log.ui.internal.util.FeatureNode;
import org.eclipse.hyades.log.ui.internal.util.FilterTableElement;
import org.eclipse.hyades.log.ui.internal.util.LogFilterCriteria;
import org.eclipse.hyades.log.ui.internal.util.LogUIConstants;
import org.eclipse.hyades.log.ui.internal.util.RecordFilterSearchUtil;
import org.eclipse.hyades.log.ui.internal.util.SortElement;
import org.eclipse.hyades.log.ui.internal.util.Sorter;
import org.eclipse.hyades.log.ui.internal.util.TerminalNode;
import org.eclipse.hyades.log.ui.internal.util.TreeNode;
import org.eclipse.hyades.models.cbe.CBECommonBaseEvent;
import org.eclipse.hyades.models.cbe.CBEDefaultEvent;
import org.eclipse.hyades.models.cbe.CBEPackage;
import org.eclipse.hyades.models.cbe.util.LogQueryBuilder;
import org.eclipse.hyades.models.hierarchy.AbstractDefaultEvent;
import org.eclipse.hyades.models.hierarchy.CorrelationContainerProxy;
import org.eclipse.hyades.models.hierarchy.HierarchyPackage;
import org.eclipse.hyades.models.hierarchy.TRCAgent;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.extensions.ExtensionsFactory;
import org.eclipse.hyades.models.hierarchy.extensions.Query;
import org.eclipse.hyades.models.hierarchy.extensions.QueryResult;
import org.eclipse.hyades.models.hierarchy.extensions.ResultEntry;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleOperand;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleSearchQuery;
import org.eclipse.hyades.models.hierarchy.extensions.TimeBasedCorrelationQuery;
import org.eclipse.hyades.models.hierarchy.plugin.HierarchyPlugin;
import org.eclipse.hyades.models.hierarchy.util.HierarchyResourceSetImpl;
import org.eclipse.hyades.models.hierarchy.util.IFilterElement;
import org.eclipse.hyades.models.hierarchy.util.ILogFilterCriteria;
import org.eclipse.hyades.models.hierarchy.util.ISortElement;
import org.eclipse.hyades.models.hierarchy.util.PerfUtil;
import org.eclipse.hyades.models.hierarchy.util.internal.QueryUtils;
import org.eclipse.hyades.trace.ui.HyadesUtil;
import org.eclipse.hyades.trace.ui.filters.internal.dialogs.FilterUIUtil;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.viewers.DecoratingLabelProvider;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.custom.CLabel;
import org.eclipse.swt.custom.ViewForm;

/**
 * This class is used to provide CBEDefaultEvent entries to the Log Viewer.
 */
public class LogContentProvider implements IStructuredContentProvider, ITreeContentProvider {

	/**
	 * @author slavescu
	 */
	public static class ReferenceList extends AbstractList{
		static class ReferenceInfo
		{
			public ReferenceInfo(int agentIndex, int index) {
				this.agentIndex = (byte)agentIndex;
				this.defaultEventIndex = index;
			}
			protected byte agentIndex;
			protected int defaultEventIndex;
		}
		ArrayList agentURIs = new ArrayList();
		ArrayList references = new ArrayList();
		/* (non-Javadoc)
		 * @see java.util.Collection#size()
		 */
		public int size() {
			return references.size();
		}

		/* (non-Javadoc)
		 * @see java.util.Collection#contains(java.lang.Object)
		 */
		public boolean contains(Object o) {
			return indexOf(o) != -1;
		}

		/**
		 * @param i
		 * @return
		 */
		private Object getRealObject(int i) {
			try
			{
				Object o = references.get(i);
				if(o!=null && o instanceof ReferenceInfo)
				{
					ReferenceInfo referenceInfo = (ReferenceInfo)references.get(i);
					TRCAgent agent = (TRCAgent)HierarchyResourceSetImpl.getInstance().getEObject((URI)agentURIs.get((int)referenceInfo.agentIndex), true);
					o = agent.getDefaultEvents().get(referenceInfo.defaultEventIndex);
				}
				return o;
			}
			catch (Exception e) {
			}
			return null;
		}

		/* (non-Javadoc)
		 * @see java.util.Collection#clear()
		 */
		public void clear() {
			int expectedModCount=modCount;
			agentURIs.clear();
			references.clear();
			if(expectedModCount!=modCount)
				throw new ConcurrentModificationException();
			modCount++;
		}

		/* (non-Javadoc)
		 * @see java.util.List#get(int)
		 */
		public Object get(int index) {
			return getRealObject(index);
		}

		/* (non-Javadoc)
		 * @see java.util.List#add(int, java.lang.Object)
		 */
		public void add(int index, Object o) {
			try
			{
				int expectedModCount = modCount;
				if(o instanceof CBEDefaultEvent)
				{	
					CBEDefaultEvent defaultEvent = (CBEDefaultEvent)o;
					int defaultEventIndex = ((EList)defaultEvent.eContainer().eGet(HierarchyPackage.eINSTANCE.getTRCAgent_DefaultEvents())).indexOf(defaultEvent);
					URI uri = EcoreUtil.getURI(defaultEvent.eContainer());
					int agentIndex = agentURIs.indexOf(uri);
					if(agentIndex==-1)
					{
						agentURIs.add(uri);
						agentIndex = agentURIs.size()-1;
					}
					ReferenceInfo referenceInfo = new ReferenceInfo(agentIndex,defaultEventIndex);
					references.add(index,referenceInfo);
				}
				else
					references.add(index, o);
				if(expectedModCount!=modCount)
				{
					throw new ConcurrentModificationException();
				}
			}catch (Exception e) {
				throw new RuntimeException("Error "+e.getLocalizedMessage()+" when adding object:"+o);
			}
		}

		/* (non-Javadoc)
		 * @see java.util.AbstractList#set(int, java.lang.Object)
		 */
		public Object set(int index, Object o) {
			Object res = getRealObject(index);
			try
			{
				int expectedModCount = modCount;
				if(o instanceof CBEDefaultEvent)
				{	
					CBEDefaultEvent defaultEvent = (CBEDefaultEvent)o;
					int defaultEventIndex = ((EList)defaultEvent.eContainer().eGet(HierarchyPackage.eINSTANCE.getTRCAgent_DefaultEvents())).indexOf(defaultEvent);
					URI uri = EcoreUtil.getURI(defaultEvent.eContainer());
					int agentIndex = agentURIs.indexOf(uri);
					if(agentIndex==-1)
					{
						agentURIs.add(uri);
						agentIndex = agentURIs.size()-1;
					}
					ReferenceInfo referenceInfo = new ReferenceInfo(agentIndex,defaultEventIndex);
					references.set(index,referenceInfo);
				}
				else
					references.set(index, o);
				if(expectedModCount!=modCount)
				{
					throw new ConcurrentModificationException();
				}
			}catch (Exception e) {
				throw new RuntimeException("Error "+e.getLocalizedMessage()+" when adding object:"+o);
			}
			return res;
		}
		/* (non-Javadoc)
		 * @see java.util.List#remove(int)
		 */
		public Object remove(int index) {
			int expectedModCount = modCount;
			Object res = getRealObject(index);
			references.remove(index);
			if(expectedModCount!=modCount)
			{
				throw new ConcurrentModificationException();
			}
			return res;
		}

		/* (non-Javadoc)
		 * @see java.util.List#indexOf(java.lang.Object)
		 */
		public int indexOf(Object o) {
			int expectedModCount = modCount;
			for (int i = 0; i < references.size(); i++) {
				if(o == getRealObject(i))
				{
					if(expectedModCount!=modCount)
						throw new ConcurrentModificationException();
					return i;
				}
			}
			return -1;
		}

		/**
		 * 
		 */
		public void trimToSize() {
			agentURIs.trimToSize();
			references.trimToSize();
		}
	}
	private int _visible = 0;
	private int _total = 0;
	private ViewForm _viewer;
	private String _title = LogUIPlugin.getResourceString("STR_LOG_PANE_TITLE");
	private Sorter sorter;
	private LogViewer _logViewer;

	private RecordFilterSearchUtil recordFilterSearch = null;
	private int _currentPage = 0;
	private int _totalPages = 0;
	private int PAGE_SIZE = 100;
	private final String ASSOCIATED_EVENTS = "associatedEvents";
	private List sortAttributes;
	private String filters;
	private List filterTableElements;
	private int _visible_roots;
	private int _total_roots;
	private List allElements;
	private boolean filterSevOne;
	private boolean filterSevTwo;
	private boolean filterSevThree;
	private boolean pageActionCalled=false;
	private Query query;
	private IAction analyzeAction;
	private ILogFilterCriteria criteria;
	private CorrelationContainerProxy input;
	private boolean bAnalyzed=false;	

	public LogContentProvider(ViewForm viewer, LogViewer logViewer) {
		super();
		_viewer = viewer;
		_logViewer = logViewer;
		recordFilterSearch = new RecordFilterSearchUtil();
	}

	public void dispose() {
		reset();
		recordFilterSearch=null;
		_viewer = null;
		_logViewer = null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.viewers.ITreeContentProvider#getChildren(java.lang.Object)
	 */
	public Object[] getChildren(Object parentElement) {
		return getElementsAll(parentElement).toArray();
	}

	public List getPageList(Object inputElement)
	{
		PerfUtil p = new PerfUtil("getPageList", true);
		List pagedList = new ArrayList();
		String pgSize = HierarchyPlugin.getPlugin().getPluginPreferences().getString(Constants.VIEW_PAGE_SIZE);
		if (pgSize != null) {
			try {
				PAGE_SIZE = Integer.parseInt(pgSize);
			} catch (Exception e) {
			}
		}

		if (isModelChanged() || prepareFiltersAndSortAttributes()) {
//			rebuildAllElements(inputElement);
			rebuildAllElements(inputElement);
			_visible_roots = _visible;
			_total_roots = _total;

			_totalPages = _visible / PAGE_SIZE;
			_totalPages = (_visible % PAGE_SIZE > 0 && _visible > 0 ? _totalPages + 1 : _totalPages);

			_currentPage = Math.max(1, _currentPage);
			_currentPage = Math.min(_totalPages, _currentPage);

			if (_visible < PAGE_SIZE || _visible == 0) {
				return allElements;
			}

			int startPos = (_currentPage - 1) * PAGE_SIZE;
			int endPos = (Math.min((_currentPage) * PAGE_SIZE, _visible));
			for (int i = startPos; i < endPos; i++) {
				Object element = allElements.get(i);
				pagedList.add(element);
			}
		} else {
			_visible = _visible_roots;
			_total = _total_roots;
			_totalPages = _visible / PAGE_SIZE;
			_totalPages = (_visible % PAGE_SIZE > 0 && _visible > 0 ? _totalPages + 1 : _totalPages);

			_currentPage = Math.max(1, _currentPage);
			_currentPage = Math.min(_totalPages, _currentPage);

			if (_visible < PAGE_SIZE || _visible == 0) {
				return allElements;
			}

			int startPos = (_currentPage - 1) * PAGE_SIZE;
			int endPos = (Math.min((_currentPage) * PAGE_SIZE, _visible));
			for (int i = startPos; i < endPos; i++) {
				Object element = allElements.get(i);
				pagedList.add(element);
			}

		}
		p.stopAndPrintStatus();
		return pagedList;
	}
	
	protected void analyze() {
		if(analyzeAction!=null && bAnalyzed==false)
		{
			try {
				analyzeAction.run();
			}
			catch (Exception e) {
				LogUIPlugin.log(e);
			}
		}
	}

	public void setAnalyzed(boolean newValue){
		bAnalyzed = newValue;
	}
	public void setAnalyzeAction(IAction r){
		analyzeAction = r;
	}
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(java.lang.Object)
	 */
	public Object[] getElements(Object inputElement) {
		List pagedList = getPageList(inputElement); 
		setTitle();
		updatePageActions();
		analyze();		
		return pagedList.toArray();
	}

	
	public boolean isInputXMI(Object inputElement){
		List agentProxies = (List)HyadesUtil.getLogAgents((EObject)inputElement);
		int s = agentProxies.size();
		TRCAgentProxy proxy = null;
		TRCAgent agent = null;
		for(int i=0;i<s;i++){
			proxy = (TRCAgentProxy)agentProxies.get(i);
			agent = (TRCAgent)proxy.eGet(HierarchyPackage.eINSTANCE.getTRCAgentProxy_Agent(), false);
			if(agent==null){
				continue;
			}
			if(EcoreUtil.getURI(agent).path().endsWith("xmi"))
			{				
				return true;
			}			
		}
		return false;		
	}
	
	private void rebuildAllElements(Object inputElement){
		
		if(!(inputElement instanceof EObject)){
			allElements = new ArrayList();
			_visible=0;
			_total=0;
			return;			
		}
		List agentProxies = (List)HyadesUtil.getLogAgents((EObject)inputElement);		
		if(isInputXMI(inputElement)){
			rebuildAllElementsXMI((TRCAgentProxy[]) agentProxies.toArray(new TRCAgentProxy[agentProxies.size()]));
		}else{
			rebuildAllElementsSQL(agentProxies);
		}
	}
	
	private SimpleSearchQuery getQueryForSQL(List agentProxies)
	{
		SimpleSearchQuery filter = _logViewer.getCurrentFilter();
		if(filter==null)
		{
			filter = QueryUtils.getEmptyQuery();
		}
		else
			if (filter instanceof TimeBasedCorrelationQuery) {
				SimpleSearchQuery tbcq = ExtensionsFactory.eINSTANCE.createSimpleSearchQuery();
				tbcq.setCount(filter.isCount());
				tbcq.setDistinct(filter.isDistinct());
				tbcq.setMaxElements(filter.getMaxElements());
				tbcq.setName(filter.getName());
				tbcq.setStartWith(filter.getStartWith());
				tbcq.setWhereExpression(filter.getWhereExpression());
				tbcq.getOutputElements().addAll(filter.getOutputElements());
				tbcq.getOrderByExpresions().addAll(filter.getOrderByExpresions());
				tbcq.getSources().addAll(filter.getSources());
				tbcq.getSubQuery().addAll(filter.getSubQuery());
				filter = tbcq;
			}

		SimpleOperand Operand = ExtensionsFactory.eINSTANCE.createSimpleOperand();
		Operand.setType(CBEPackage.eINSTANCE.getCBECommonBaseEvent());
		filter.getOutputElements().add(Operand);
		
		EObject inputObject;
		int s = agentProxies.size();
		for (int i = 0; i < s; i++) {
			inputObject = (EObject) agentProxies.get(i);
			if (inputObject instanceof TRCAgentProxy) {
				TRCAgent agent = (TRCAgent) ((TRCAgentProxy)inputObject).eGet(HierarchyPackage.eINSTANCE.getTRCAgentProxy_Agent(), false);
				if (agent != null) {
					String uri = EcoreUtil.getURI(agent).toString();
					if (!filter.getSources().contains(uri))
						filter.getSources().add(uri);
				}
			} else if (inputObject instanceof CorrelationContainerProxy) {
				for (Iterator iter = ((CorrelationContainerProxy)inputObject).getCorrelatedAgents().iterator(); iter.hasNext();) {
					TRCAgentProxy agentProxy = (TRCAgentProxy) iter.next();
					TRCAgent agent = (TRCAgent) agentProxy.eGet(HierarchyPackage.eINSTANCE.getTRCAgentProxy_Agent(), false);
					if (agent != null) {
						String uri = EcoreUtil.getURI(agent).toString();
						if (!filter.getSources().contains(uri))
							filter.getSources().add(uri);
					}
				}
			}
		}

		filter.setDistinct(true);
		
		filter.getOrderByExpresions().addAll(LogQueryBuilder.createOrderByClause(CBEPackage.eINSTANCE.getCBECommonBaseEvent(), getFilterCriteria().getSortColumns()));
		
		return filter;
	}
	
	/**
	 * @param inputElement
	 */
	private void rebuildAllElementsSQL(List agentProxies) {
		
		try
		{
			
			if(agentProxies.size()==0)
			{
				allElements = new ArrayList();
				_visible=0;
				_total=0;
				return;
			}

			for (Iterator iter = agentProxies.iterator(); iter.hasNext();) {
				TRCAgentProxy element = (TRCAgentProxy) iter.next();
				//load the agent in order to set the CommonBaseEvent-Agent reference
				element.getAgent();
			}
			
			if (FilterUIUtil.ENABLE_NEW_FILTERING && FilterUIUtil.ENABLE_NEW_FILTERING_FOR_SQL)
				query = getQueryForSQL(agentProxies);
			else
				query = LogQueryBuilder.createQuery(agentProxies, getFilterCriteria());
			
			if(query==null){
				allElements = new ArrayList();
				_visible=0;
				_total=0;
				return;				
			}
			SimpleSearchQuery countAllQuery = (SimpleSearchQuery)EcoreUtil.copy(query);
			countAllQuery.setCount(true);
			QueryResult queryResult = LogQueryBuilder.executeQuery(query, HierarchyResourceSetImpl.getInstance());
			_visible = 0;
			_total = 0;
			if(queryResult==null)
			{
				allElements = new ArrayList();
				_visible=0;
				_total=0;
				return;
			}
			ResultEntry resultEntry=null;
			if(queryResult.getResultEntries().size()>0)
			{
				resultEntry =(ResultEntry)queryResult.getResultEntries().get(0); 
				allElements = (List)resultEntry.getValue();
				_visible=allElements.size();
			}
			
			countAllQuery.setWhereExpression(null);
			countAllQuery.getOrderByExpresions().clear();

			queryResult = LogQueryBuilder.executeQuery(countAllQuery, HierarchyResourceSetImpl.getInstance());
			if(queryResult==null)
			{
				allElements = new ArrayList();
				_visible=0;
				_total=0;
				return;
			}

			if(queryResult.getResultEntries().size()>0)
			{
				resultEntry =(ResultEntry)queryResult.getResultEntries().get(0);
				List countList = (List)resultEntry.getValue();
				if(countList.size()>0)
					_total = ((Integer)countList.get(0)).intValue();
			}
		}
		catch (Exception e) {			
			LogUIPlugin.log(e);
            Status errorStatus = new Status(Status.ERROR, ResourcesPlugin.PI_RESOURCES, IResourceStatus.INTERNAL_ERROR,e.getLocalizedMessage() , e);

            ErrorDialog.openError(LogUIPlugin.getActiveWorkbenchShell(), LogUIPlugin.getResourceString("LOGGING_MESSAGE"), LogUIPlugin.getResourceString("ERROR_LOADING_LOG_RECORDS"), errorStatus);
			
			allElements = new ArrayList();
			_visible=0;
			_total=0;			
		}

	}

	/**
	 * @param inputElement
	 */
	private void rebuildAllElementsXMI(Object inputElement) {
		allElements = getElementsAll(inputElement);
		((ArrayList)allElements).trimToSize(); // 			((ReferenceList)allElements).trimToSize();
	}

	/**
	 * @return
	 */
	private boolean isModelChanged() {
		
		if (allElements == null || _logViewer.isHandlingRefresh())
		{
			prepareFiltersAndSortAttributes();		
			return true;
		}
	
		//TODO: MS add better checking
		return false;
	}

	/**
	 *  
	 */
	private void updatePageActions() {
		try {
			pageActionCalled=false;
			
			_logViewer.getPageUpAction().setEnabled(_currentPage - 1 > 0);
			_logViewer.getPageDownAction().setEnabled((_totalPages - _currentPage) > 0);
			_logViewer.getGotPageAction().setEnabled(_totalPages > 1);

		} catch (Exception e) {
			// TODO: handle exception
		}
	}

	/**
	 * Returns an array of CBEDefaultEvent elements under a monitor or node or agent filtered by severity
	 * 
	 * @param element
	 * @see #getFilteredRecords(List)
	 * @return an array of CBEDefaultEvent objects
	 */
	public List getElementsAll(Object element) {

		List wlist = new ArrayList();//new ReferenceList();
		//List pdList = new ArrayList();//new ReferenceList();
		_visible = 0;
		_total = 0;

		if (element instanceof TRCAgentProxy) {
			getAgentRecords((TRCAgentProxy) element, wlist);
		}else		
		if(element instanceof TRCAgentProxy[]){			
			getAgentRecords(((TRCAgentProxy[])element), wlist);			
		}
		else if (element instanceof EObject && ((EObject) element).eClass().getEPackage() == CBEPackage.eINSTANCE) {
			getFeatureNodes((EObject) element, wlist);
		} else if (element instanceof FeatureNode) {
			getFeatureNodeElements((FeatureNode) element, wlist);
		}else if(element instanceof TreeNode){
			if(((TreeNode)element).getElement() instanceof List && ((List)((TreeNode)element).getElement()).size()>0){
				List elements = (List)((TreeNode)element).getElement();
				int s = elements.size();
				DecoratingLabelProvider labelProvider = ((DecoratingLabelProvider)_logViewer.getLogPage().getView().getViewer().getLabelProvider());
				for(int i=0;i<s;i++){
					wlist.add(new TerminalNode(element,(EObject)elements.get(i),labelProvider.getText(elements.get(i))));
					
				}
			}			
		}

		return wlist;
	}

	public boolean hasChildren(Object element) {

		boolean result = false;

		//check if the object is in the CommonBaseEvent model, i.e in the CBE package
		if (element instanceof EObject && ((EObject) element).eClass().getEPackage() == CBEPackage.eINSTANCE) {

			final EClass eClass = ((EObject) element).eClass();
			for (Iterator iter = eClass.getEAllStructuralFeatures().iterator(); iter.hasNext();) {
				EStructuralFeature feature = (EStructuralFeature) iter.next();
				// if not string array attribute add feature node (if it's not an empty list)
				if ((feature instanceof EReference && feature.getFeatureID() != CBEPackage.CBE_COMMON_BASE_EVENT__SYMPTOMS) || (feature instanceof EAttribute && feature.isMany() && feature.getEType().getInstanceClass() != String.class)) {
					Object refs = ((EObject) element).eGet(feature, true);
					if (refs != null) {
						if (refs instanceof EList) {
							if (((EList) refs).size() > 0) {

								result = true;
								break;
							}
						} else {
							result = true;
							break;
						}

					}
				}
			}

		} else if (element instanceof FeatureNode) {

			final EStructuralFeature feature = ((FeatureNode) element).getFeature();
			final EObject cElem = ((FeatureNode) element).getElement();
			if (feature instanceof EReference) {
				if(feature.getName()==ASSOCIATED_EVENTS){
					return true;
				}
				if (feature.isMany()) {
					EList refs = (EList) cElem.eGet(feature, true);
					result = refs.size() > 0;
				} else {
					Object ref = cElem.eGet(feature, true);
					result = ref != null;
				}

			}
		} else if (element instanceof TerminalNode) {
			result = false;
		}else if(element instanceof TreeNode){
			if(((TreeNode)element).getElement() instanceof List){
				return ((List)((TreeNode)element).getElement()).size()>0;
			}
			return false;
		}

		return result;
	}

	public Object getParent(Object element) {
		return null;
	}
	public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		if(newInput instanceof CorrelationContainerProxy){
			input = (CorrelationContainerProxy)newInput;
		}else{
			input = null;
		}
		reset();
	}
	private void reset() {
		filters = null;
		sortAttributes = null;
		filterTableElements = null;
		allElements = null;		
		_currentPage = 0;
		_totalPages = 0;
		pageActionCalled=false;		
	}

	public boolean isDeleted(Object element) {
		return false;
	}

	private void updateSeverity()
	{
		if(filters==null)
			return;

		filterSevOne = true;
		filterSevTwo = true;
		filterSevThree = true;

		int idx = filters.indexOf(' ');
		
		if (idx == -1) {
			return;
		}
		filterSevOne = (filters.charAt(idx-1)=='1');

		idx = filters.indexOf(' ',idx+1);
		if (idx == -1) {
			return;
		}
		filterSevTwo = (filters.charAt(idx-1)=='1');
		
		idx = filters.indexOf(' ',idx+1);
		if (idx == -1) {
			return;
		}
		filterSevThree = (filters.charAt(idx-1)=='1');
	}
	
	/**
	 * Filters and sorts the list of CBEDefaultEvent's according to the filter and sort options saved in the preferences store CBEDefaultEvent objects are considered to be: - of severity one if their severity property satisfies the condition 50
	 * <= severity <=70 - of severity two if 30 <= severity <50 - of severity three if 0 <= severity <30
	 */
	private List getFilteredRecords(List pdList) {

			if (sortAttributes != null && sortAttributes.size() > 0) {
				Collections.sort(pdList, getLogSorter(sortAttributes));
			}
	
			ArrayList data = new ArrayList();
	
			if ((filters == null || filters.equals("")) && filterTableElements == null) {
				for (int i = 0; i < pdList.size(); i++) {
					data.add(pdList.get(i));
				}
				_total += pdList.size();
				_visible += pdList.size();
				return data;
			}
	
			for (int i = 0; i < pdList.size(); i++) {
				Object entry = pdList.get(i);
	
				try {
					int sev = ((CBECommonBaseEvent) entry).getSeverity();
					
					if (((sev >= 50 && sev <= 70 && filterSevOne) || (sev >= 30 && sev < 50 && filterSevTwo) || (sev >= 0 && sev < 30 && filterSevThree)) && isAdvancedFilterAppliedTo((CBECommonBaseEvent)entry)) {
						data.add(entry);
						_visible++;
					}
	
				} catch (ClassCastException e) {
					// we might encounter nodes that are not CommonBaseEvent's and don't have severity
					data.add(entry);
					_visible++;
				}
	
				_total++;
			}
			return data;
	}

	/**
	 * @param entry
	 * @return
	 */
	private boolean isAdvancedFilterAppliedTo(CBECommonBaseEvent entry) {
		if(filterTableElements!=null){
			int s = filterTableElements.size();
			return recordFilterSearch.isAdvFilterApply(entry, (IFilterElement[])filterTableElements.toArray(new IFilterElement[s]));
		}
		return true;
	}

	private boolean prepareFiltersAndSortAttributes() {
		List sortAttributes1 = parsePreferenceString(LogUIConstants.PD_SORT_LOG_OPTIONS);

		String filters1 = LogUIPlugin.getDefault().getPreferenceStore().getString(LogUIConstants.PD_SEV_OPTIONS);
		List filterTableElements1 = FilterTableElement.createFilterTableElementFromString(LogUIPlugin.getDefault().getPreferenceStore().getString(LogUIConstants.PD_ADV_FILTER_OPTIONS));

		if ((sortAttributes != null && !sortAttributes.equals(sortAttributes1)) || (sortAttributes1 != null && !sortAttributes1.equals(sortAttributes))) {
			return swapFiltersAndSortAttributes(sortAttributes1, filters1, filterTableElements1);
		}
		
		if ((filters != null && !filters.equals(filters1)) || (filters1 != null && !filters1.equals(filters))) {
			return swapFiltersAndSortAttributes(sortAttributes1, filters1, filterTableElements1);
		}
		
		if ((filterTableElements != null && !filterTableElements.equals(filterTableElements1)) || (filterTableElements1 != null && !filterTableElements1.equals(filterTableElements))) {
			return swapFiltersAndSortAttributes(sortAttributes1, filters1, filterTableElements1);
		}
		return false;
	}

	private boolean swapFiltersAndSortAttributes(List sortAttributes1, String filters1, List filterTableElements1) {
		sortAttributes = sortAttributes1;
		filters = filters1;
		filterTableElements = filterTableElements1;
		updateSeverity();
		return true;
	}

	/**
	 * Sets the title of the log viewer
	 *  
	 */
	private void setTitle() {
		String[] variables = new String[4];
		variables[0] = "" + _visible;
		variables[1] = "" + _total;
		variables[2] = "" + _currentPage;
		variables[3] = "" + _totalPages;

		String title = (new StringBuffer(_title).append(" ").append(LogUIPlugin.getResourceString("STR_SHOW_INFO", variables))).toString();

		//		title = HyadesUtil.change(title, "%2", String.valueOf(_total));
		//		title = HyadesUtil.change(title, "%1", String.valueOf(_visible));

		 ((CLabel) _viewer.getTopLeft()).setText(title);

		 _logViewer.updateFilterAppliedDescription();
	}
	
	public ILogFilterCriteria getFilterCriteria(){
		int s = 0;
		if(criteria==null){
			criteria = new LogFilterCriteria();
		}
		if(filterTableElements!=null)
		{
			int  analyzedFilterIndex = -1;
			s = filterTableElements.size();
			for(int i=0;i<s;i++){
				if(((FilterTableElement)filterTableElements.get(i)).getAttribute().equals("analyzed")){
					analyzedFilterIndex = i;
					break;
				}
			}
			if(analyzedFilterIndex>-1){				
				if(s>1){	
					IFilterElement[] filter = new FilterTableElement[s-1];
					int index = 0;
					for(int i=0;i<s;i++){
						if(i!=analyzedFilterIndex){
							filter[index++] = (FilterTableElement)filterTableElements.get(i);
						}
					}
					criteria.setFilters(filter);
				}else{
					criteria.setFilters(new IFilterElement[0]);
				}
			}else{				
				criteria.setFilters((IFilterElement[])filterTableElements.toArray(new IFilterElement[s]));
			}
		}
		else{			
			criteria.setFilters(new IFilterElement[0]);
		}
		
		s = sortAttributes.size();
		ISortElement[] sortElements = null; 
		if(s>0){
			sortElements = new ISortElement[s];
			for(int i=0;i<s;i++){
				sortElements[i] = new SortElement((String)sortAttributes.get(i), true); 
			}
			criteria.setSortColumns(sortElements);
			
		}else{
			criteria.setSortColumns(new ISortElement[0]);
		}
		
		Map options = new HashMap();
		options.put(ILogFilterCriteria.OPTION_FILTER_SEV1, filterSevOne==true?"1":"0");
		options.put(ILogFilterCriteria.OPTION_FILTER_SEV2, filterSevTwo==true?"1":"0");
		options.put(ILogFilterCriteria.OPTION_FILTER_SEV3, filterSevThree==true?"1":"0");
		criteria.setFilterOptions(options);
		return criteria;
	}
	
	public Query getQuery(){
		return query;
	}

	private List parsePreferenceString(String option) {

		ArrayList newSortAttributes = null;
		String firstSort = "";
		IPreferenceStore store = LogUIPlugin.getDefault().getPreferenceStore();

		String sortData = LogUIPlugin.getDefault().getPreferenceStore().getString(option);

		if (sortData != null) {
			int i = sortData.indexOf(" ");
			String column = "";
			int j = 0;
			newSortAttributes = new ArrayList();
			while (i != -1) {
				column = sortData.substring(0, i);
				j = column.indexOf(":");
				if (j > -1 && column.substring(j + 1).equals("1")) {
					newSortAttributes.add(firstCharToUpperCase(column.substring(0, j)));
				}
				sortData = sortData.substring(i + 1);
				i = sortData.indexOf(" ");
			}
			int k = sortData.indexOf(":");
			if (k > -1 && sortData.substring(k + 1).equals("1")) {
				newSortAttributes.add(firstCharToUpperCase(sortData.substring(0, k)));
			}

		}

		return newSortAttributes;

	}
	
	private String firstCharToUpperCase(String name) {
		char[] charName = name.toCharArray();
		char c = Character.toUpperCase(charName[0]);
		charName[0] = c;
		return new String(charName);
	}
	
	private Sorter getLogSorter(List sortAttributes) {

		if (sorter == null) {
			sorter = new Sorter(sortAttributes);
		}

		sorter.setAttributes(sortAttributes);

		return sorter;
	}

	
	private void getAgentRecords(TRCAgentProxy agentProxy, List wlist){
		getAgentRecords(new TRCAgentProxy[]{agentProxy},wlist);
	}
	
	private void getAgentRecords(TRCAgentProxy[] agents, List wlist){
		
		List logRecords = new ArrayList(); 
		List agList = new ArrayList(); 
		_total=0;
		for (int i = 0; i < agents.length; i++) {
			if(agents[i]!=null && agents[i].getAgent()!=null){
				if (FilterUIUtil.ENABLE_NEW_FILTERING){
					_total+=agents[i].getAgent().getDefaultEvents().size();
					agList.add(agents[i].getAgent());
				}
				else
					logRecords.addAll(agents[i].getAgent().getDefaultEvents());
			}
		}
		if (FilterUIUtil.ENABLE_NEW_FILTERING)
		{
			List result = _logViewer.getFilteredLogRecords(agList);
			_visible = result.size();
			wlist.addAll(result);
			
			if (sortAttributes != null && sortAttributes.size() > 0) {
				Collections.sort(wlist, getLogSorter(sortAttributes));
			}			
		}
		else
		{
			wlist.addAll(getFilteredRecords(logRecords));
			logRecords.clear();
		}
		
	}

	/**
	 * Creates a list of features of the EObject passed as parameter, wrapped into a FeatureNode object FeatureNode will be used to get the actual elements of the feature, this is the reason why the current EObject is also saved into the FeatureNode
	 * 
	 * @param element
	 * @param wlist
	 */

	private void getFeatureNodes(EObject element, List wlist) {

		try {

			EClass eClass = (element).eClass();

			if(allElements instanceof ArrayList)
				generateAssociatedEvents(element, wlist);
			//loop through all the features of the current object and return those who are not the symptom reference or byte arrays
			for (Iterator iter = eClass.getEAllStructuralFeatures().iterator(); iter.hasNext();) {
				EStructuralFeature feature = (EStructuralFeature) iter.next();
				if (feature.getContainerClass().equals(AbstractDefaultEvent.class)) {
					continue;
				}
				// if not string array attribute add feature node (if it's not an empty list)
				if ((feature instanceof EReference && feature.getFeatureID() != CBEPackage.CBE_COMMON_BASE_EVENT__SYMPTOMS) || (feature instanceof EAttribute && feature.isMany() && feature.getEType().getInstanceClass() != String.class)) {
					Object refs = ((EObject) element).eGet(feature, true);
					if (refs != null) {
						if (refs instanceof EList) {
							if (((EList) refs).size() > 0) {
								wlist.add(new FeatureNode((EObject) element, feature));
							}
						} else {
							wlist.add(new FeatureNode((EObject) element, feature));
						}
					}

				}
			}
		} catch (Exception e) {
			e.printStackTrace();
			wlist.clear();
		}
	}

	private void generateAssociatedEvents(EObject element, List wlist) {
		EReference associatedEvents = null;
		//check for internal correlations, add them to the list of associated events
		if(element instanceof CBEDefaultEvent && ((CBEDefaultEvent) element).getAgent()!=null){
			boolean hasInternalCorrelations = false;
			List internalCorrelations = ((CBEDefaultEvent) element).getAgent().getAgentProxy().getInternalCorrelations();
			int s = internalCorrelations.size();
			CorrelationContainerProxy container = null;
			
			for(int i=0;i<s;i++){
				container = (CorrelationContainerProxy)internalCorrelations.get(i);
				if(container.getCorrelationContainer()!=null){
					EMap correlations = container.getCorrelationContainer().getCorrelations();
					if(correlations!=null && correlations.get(element)!=null){
						hasInternalCorrelations = true;
						break;
					}
				}
			}
			if(hasInternalCorrelations){
				associatedEvents = EcoreFactory.eINSTANCE.createEReference();
				associatedEvents.setName(ASSOCIATED_EVENTS);
				wlist.add(new FeatureNode((EObject) element, associatedEvents));					
			}
		}
		//if the viewer input is a CorrelationContainerProxy add the "associated events" to the list of features
		if(input!=null && input.getCorrelationContainer()!=null){
			EMap correlations = input.getCorrelationContainer().getCorrelations();
			if(correlations!=null && correlations.get(element)!=null){
				if(associatedEvents==null){
					associatedEvents = EcoreFactory.eINSTANCE.createEReference();
					associatedEvents.setName(ASSOCIATED_EVENTS);
					wlist.add(new FeatureNode((EObject) element, associatedEvents));
				}
			}
		}
	}

	/**
	 * Returns in the wlist parameter the list of elements of the feature passed by the element parameter The model object cached in the FeatureNode paramater is queried for the elements of the feature also cached as part of the FeatureNode paramater
	 * 
	 * @param element
	 * @param wlist
	 */
	private void getFeatureNodeElements(FeatureNode element, List wlist) {
		try {

			final EStructuralFeature feature = element.getFeature();
			final EObject cElem = element.getElement();
			if (feature instanceof EReference) {

				//if the feature is "associated events" then look up the associated events in the internal correlations (if any) and the correlation container
				if(feature.getName()==ASSOCIATED_EVENTS){
					//add internal correlations first
					List internalCorrelations = ((CBEDefaultEvent) cElem).getAgent().getAgentProxy().getInternalCorrelations();
					int s;
					List corrEngines = null;
					if(internalCorrelations!=null && (s = internalCorrelations.size())>0){

						CorrelationContainerProxy container = null;
						corrEngines = new ArrayList();
						for(int i=0;i<s;i++){
							container = (CorrelationContainerProxy)internalCorrelations.get(i);
							if(container.getCorrelationContainer()!=null){
								EMap correlations = container.getCorrelationContainer().getCorrelations();
								if(correlations!=null && correlations.get(cElem)!=null && !wlist.contains(container.getCorrelationEngine())){
									if(!corrEngines.contains(container.getCorrelationEngine())){	
										wlist.add(new TreeNode(container.getCorrelationEngine(), correlations.get(cElem)));									
										corrEngines.add(container.getCorrelationEngine());
									}
								}
							}
						}
					}
					
					//add associated events from the correlation container
					if(input!=null && input.getCorrelationContainer()!=null){
						EMap correlations = input.getCorrelationContainer().getCorrelations();
						if(correlations!=null && correlations.get(cElem)!=null && !wlist.contains(input.getCorrelationEngine())){
							if(corrEngines==null){
								corrEngines = new ArrayList();
							}
							if(!corrEngines.contains(input.getCorrelationEngine())){
								wlist.add(new TreeNode(input.getCorrelationEngine(), correlations.get(cElem)));								
							}
						}
					}
					return;
				}
				//return the elements of a containment by value or create a terminal node (nodes that don't expand anymore, avoiding an infinite tree)

				if (feature.isMany()) {

					EList refs = (EList) cElem.eGet(feature, true);
					if (((EReference) feature).isContainment()) {

						wlist.addAll(refs);

					} else {

						int index = 0;

						EObject elem = null;
						String label = "";
						for (Iterator iter = refs.iterator(); iter.hasNext();) {
							elem = (EObject) iter.next();
							//wlist.add(new TerminalNode(elem, elem.eClass().getName()+"["+index+"] - "));
							EStructuralFeature f = elem.eClass().getEStructuralFeature(CBEPackage.CBE_COMMON_BASE_EVENT__MSG);

							if (f == CBEPackage.eINSTANCE.getCBECommonBaseEvent_Msg()) {
								label = (String) elem.eGet(f);
							} else if ((f = elem.eClass().getEStructuralFeature(CBEPackage.CBE_DEFAULT_EVENT__EXTENSION_NAME)) == CBEPackage.eINSTANCE.getCBEDefaultEvent_ExtensionName()) {
								label = (String) elem.eGet(f);
							} else {
								label = elem.eClass().getName();
							}
							wlist.add(new TerminalNode(element.getElement(), elem, label));
							index++;
						}
					}
				} else {
					EObject ref = (EObject) cElem.eGet(feature);
					if (((EReference) feature).isContainment()) {

						wlist.add(ref);

					} else {
						wlist.add(new TerminalNode(element.getElement(), ref, ref.eClass().getName()));
					}
				}
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	/**
	 *  
	 */
	public boolean pageUp() {
		if (_currentPage > 1) {
			_currentPage--;
		}
		pageActionCalled=true;
		// return true if is not first
		return _currentPage != 1;
	}

	/**
	 *  
	 */
	public boolean pageDown() {
		if (_currentPage < _totalPages) {
			_currentPage++;
		}
		pageActionCalled=true;
		// return true if is not last page
		return _currentPage != _totalPages;
	}

	public boolean goToPage(int pageNr) {
		if (pageNr > 0 && pageNr <= _totalPages) {
			_currentPage = pageNr;
		}
		pageActionCalled=true;
		// return true if is not last page
		return _currentPage != _totalPages;
	}

	/**
	 * @return
	 */
	public int getTotalPages() {
		return _totalPages;
	}

	/**
	 * @param record
	 * @return
	 */
	public boolean revealObject(EObject record) {

		if(allElements==null)
			return false;
		int index = allElements.indexOf(record);
		if(index==-1)
		{
			return false;
		}
		
		index++;
		int _prevPage=_currentPage;
		int newPage = index/ PAGE_SIZE;
		newPage = (index % PAGE_SIZE > 0 && index > 0 ? newPage + 1 : newPage);
		goToPage(newPage);
		return newPage!=_prevPage;
	}
	
	/**
	 * 
	 * @return the cached list of elements (filtered and sorted)
	 */
	public List getCachedElements()
	{
		return allElements;
	}
}
