/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ImportLogWizard.java,v 1.48 2005/06/08 15:43:10 apnan Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.log.ui.internal.wizards;

import java.net.URL;
import java.text.DateFormat;
//import java.util.ArrayList;
import java.util.ArrayList;
import java.util.Date;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.StringTokenizer;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.MultiStatus;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Preferences;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.IJobChangeEvent;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.core.runtime.jobs.JobChangeAdapter;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.hyades.internal.execution.local.common.CommandElement;
import org.eclipse.hyades.internal.execution.local.common.Constants;
import org.eclipse.hyades.internal.execution.local.common.DataProcessor;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.internal.execution.local.control.AgentFactory;
import org.eclipse.hyades.internal.execution.local.control.AgentListener;
import org.eclipse.hyades.internal.execution.local.control.InactiveProcessException;
import org.eclipse.hyades.internal.execution.local.control.NoSuchApplicationException;
import org.eclipse.hyades.internal.execution.local.control.Node;
import org.eclipse.hyades.internal.execution.local.control.NodeFactory;
import org.eclipse.hyades.internal.execution.local.control.NotConnectedException;
import org.eclipse.hyades.internal.execution.local.control.Process;
import org.eclipse.hyades.internal.execution.local.control.ProcessFactory;
import org.eclipse.hyades.internal.execution.local.control.ProcessListener;
import org.eclipse.hyades.loaders.util.HyadesResourceExtensions;
import org.eclipse.hyades.loaders.util.LoadersUtils;
import org.eclipse.hyades.loaders.util.XMLLoader;
import org.eclipse.hyades.log.ui.internal.LogUIPlugin;
import org.eclipse.hyades.log.ui.internal.navigator.LogNavigator;
import org.eclipse.hyades.log.ui.internal.util.GLAFilterHelper;
import org.eclipse.hyades.log.ui.internal.util.TraceMergeUIDialog;
import org.eclipse.hyades.logging.adapter.model.internal.adapter.util.AdapterResourceFactoryImpl;
import org.eclipse.hyades.logging.parsers.Parser;
import org.eclipse.hyades.logging.parsers.ParserConstants;
import org.eclipse.hyades.models.hierarchy.HierarchyFactory;
import org.eclipse.hyades.models.hierarchy.TRCAgent;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.models.hierarchy.plugin.HierarchyPlugin;
import org.eclipse.hyades.models.hierarchy.util.IHyadesExtendedResource;
import org.eclipse.hyades.models.hierarchy.util.IHyadesResourceExtension;
import org.eclipse.hyades.models.hierarchy.util.PerfUtil;
import org.eclipse.hyades.security.util.TString;
import org.eclipse.hyades.trace.internal.ui.PDPerspective;
import org.eclipse.hyades.trace.internal.ui.TraceConstants;
import org.eclipse.hyades.trace.ui.HyadesConstants;
import org.eclipse.hyades.trace.ui.HyadesUtil;
import org.eclipse.hyades.trace.ui.ProfileEvent;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.core.TraceAssociationManager;
import org.eclipse.hyades.trace.ui.internal.piclient.XMLTraceDataProcessor;
import org.eclipse.hyades.trace.ui.internal.util.PDCoreUtil;
import org.eclipse.hyades.ui.HyadesUI;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.extension.IAssociationDescriptor;
import org.eclipse.hyades.ui.extension.IAssociationMapping;
import org.eclipse.hyades.ui.extension.IAssociationMappingRegistry;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.window.Window;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.swt.custom.BusyIndicator;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IImportWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.internal.Workbench;


/**
 * Standard workbench wizard for importing log files
 * from a local or remote file system into the
 * workspace.
 */
public class ImportLogWizard extends Wizard implements IImportWizard, Runnable {
    //~ Instance fields ----------------------------------------------------------------------------

    protected IStructuredSelection selection;
    protected IWorkbench workbench;
    protected LogParserItem _parser;
    protected LogNavigator fViewer;
    protected Shell fShell;
    protected String errorMessage = null;
    protected String logParserId;
    protected String reasonAction = null;
    private LogParserItem currentSelectedParser;
    private boolean bAppend = false;
	private final String CANCEL = "cancel";
	private ImportLogWizardPage wizardPage;
	private boolean shouldOpenLogView = false;	
	private String detailMsg = null;
	private StatusListener monitorThread = null;
	private TRCAgent displayAgent;
	private LogImportJobListener jobListener = new LogImportJobListener();
	private ArrayList jobPool = null;
	
    //~ Constructors -------------------------------------------------------------------------------

    public ImportLogWizard() {

        IDialogSettings workbenchSettings = LogUIPlugin.getDefault().getDialogSettings();

        IDialogSettings section = workbenchSettings.getSection("org.eclipse.hyades.log.ui.internal.wizards.ImportLogFile");

        if (section == null) {
            section = workbenchSettings.addNewSection("org.eclipse.hyades.log.ui.internal.wizards.ImportLogFile");
        }

        setDialogSettings(section);
        
        fShell = Workbench.getInstance().getActiveWorkbenchWindow().getShell();
    }

    //~ Methods ------------------------------------------------------------------------------------

    public LogParserItem getSelectedParser(){
		return _parser;
    }

	public List getLogFileElements(){
		return wizardPage.getLogFileElements();
	}

    public void addPages() {
        super.addPages();

        setWindowTitle(LogUIPlugin.getResourceString("IMPORT_LOG_FILE_WIZARD_TITLE"));

        wizardPage = new ImportLogWizardPage();
        addPage(wizardPage);

    }

    public void init(IWorkbench workbench, IStructuredSelection currentSelection) {
        this.workbench = workbench;

        selection = currentSelection;

        setNeedsProgressMonitor(true);
    }

    public boolean performFinish() {
        if(!performPageFinish()){
        	return false;
        }
        errorMessage = null;
        BusyIndicator.showWhile(Display.getDefault(),ImportLogWizard.this); 
        return true;
    }

    public void run() {

        /*
         * Ask the user if they want to switch perspective
         */
        synchronized(this){ 
		Display.getDefault().syncExec(new Runnable() {
				public void run() {
					/*
					 * The log view should not be opened until this block
					 * has finished and the user said they want to switch
					 * perspective.
					 */

					try {
						/* find out if the perspective should change */
						IWorkbenchWindow window = PlatformUI.getWorkbench()
								.getActiveWorkbenchWindow();
						boolean shouldSwitch = HyadesUIPlugin
								.getInstance()
								.openPerspectiveSwitchDialog(
										window.getShell(),
										PDPerspective.ID_TRACE_PERSPECTIVE,
										UIPlugin.getDefault()
												.getPreferenceStore(),
										HyadesUI.SWITCH_TO_PERSPECTIVE_KEY,
										LogUIPlugin
												.getResourceString("CONFIRM_PERSP_SWITCH_MSG"));
						if (shouldSwitch) {
							UIPlugin.getActivePage();
						}
						if (PDPerspective.ID_TRACE_PERSPECTIVE
								.equals(window.getActivePage()
										.getPerspective().getId())) {
							fViewer = (LogNavigator) UIPlugin
									.getActivePage()
									.showView(
											PDPerspective.ID_LOG_NAVIGATOR_VIEW);
//							StructuredViewer viewer = fViewer.getViewer();
//							viewer.refresh();
							shouldOpenLogView = true;
						} else {
							shouldOpenLogView = false;
						}
					} catch (Exception e) {
						HyadesUIPlugin.logError(e);
					}
				}
		});
       }

    	List elements = getLogFileElements();				
		
		//jobListener.addJobStatusChangeListener(monitorThread);
		if(elements.size()>0){	
			jobPool = new ArrayList(elements.size());
		}
		IStatus status = null;		
		String errMsg = "";
		
		for(int i=0; i<elements.size(); i++){
			status = importLog((LogFileElement)elements.get(i));
			if(status.getSeverity()==IStatus.ERROR || (status.getChildren().length>0 && status.getChildren()[0].getSeverity()==IStatus.ERROR)){
				Map input = getUserInput((LogFileElement)elements.get(i),false);
				String param = (String)input.get("file_path");
				if(param==null){
					param = ((LogFileElement)elements.get(i)).getParser().getName();
				}				
				errMsg = LogUIPlugin.getResourceString("IMPORT_LOG_FILE_GENERIC_ERR_MSG", new String[]{param});									
				ErrorDialog.openError(getShell(), LogUIPlugin.getResourceString("LOGGING_MESSAGE"),errMsg, status);
			}
		}
		LogImportJob job = null;
		int nrOfLocalImports = getLocalImportsCount();
		
		if(nrOfLocalImports>0){
			monitorThread = new StatusListener(nrOfLocalImports);			
		}
		for(int i=0; i<jobPool.size(); i++){
			job = (LogImportJob)jobPool.get(i);
			if(job instanceof LocalLogImportJob){
				++nrOfLocalImports;
			}
			if(job!=null){
				job.schedule();
			}
		}
		if(monitorThread!=null)
			monitorThread.start();
		
		final TRCAgent refreshAgent = displayAgent;

		/*
		 * Don't try to open the log view until the user said
		 * they want to switch perspective. (See synchronized
		 * block above).
		 */
		synchronized(this) {
			Display.getDefault().asyncExec(new Runnable() {
				public void run(){	   
					if(shouldOpenLogView) {
						((TreeViewer)fViewer.getViewer()).setExpandedState(fViewer.getContentProvider().getLogFolder(), true);
						fViewer.getViewer().setSelection(new StructuredSelection(refreshAgent.getAgentProxy()));
						openLogView(refreshAgent);
					}
				}
			});
		}
    }

	private void refreshView(final Object object){

		Display.getDefault().asyncExec(new Runnable() {
				public void run() {
					//update ui
					ProfileEvent event = UIPlugin.getDefault().getProfileEvent();
					event.setSource(object);
					event.setType(ProfileEvent.REFRESH_VIEWS);
					UIPlugin.getDefault().notifyProfileEventListener(event);
				}
			});		
	}
    
    protected boolean performPageFinish() {
        return wizardPage.finish();
    }

    private String getParserCommand(LogFileElement element, Hashtable input) {
        String cmd = element.getParser().getParserClass().concat(" ");

        //Hashtable input = getUserInput(element, true);
        
        //Add the client's language and country codes for locale-specific message resolution on the remote host:
        Locale defaultLocale = Locale.getDefault();
        
        String localeProperty = defaultLocale.getLanguage().trim();

        if(localeProperty.length() > 0){

            input.put(ParserConstants.CLIENT_LOCALE_LANGUAGE_KEY, localeProperty);

            localeProperty = defaultLocale.getCountry().trim();

            if(localeProperty.length() > 0){            
                input.put(ParserConstants.CLIENT_LOCALE_COUNTRY_KEY, localeProperty);
            }
            
            localeProperty = defaultLocale.getVariant().trim();

            if(localeProperty.length() > 0){            
                input.put(ParserConstants.CLIENT_LOCALE_VARIANT_KEY, localeProperty);
            }
        }        
        
        Enumeration enum = input.keys();

        while (enum.hasMoreElements()) {
			Object key = enum.nextElement();
			String value = input.get(key).toString();
			cmd = cmd.concat("\"").concat(key.toString()).concat("=");
			if(value.endsWith("\\") || value.endsWith("/")){
				cmd = cmd.concat(value.substring(0,value.length()-1)).concat("\" ");
			}
			else{
				cmd = cmd.concat(value).concat("\" ");			
			}            
        }

        return cmd.trim();
    }

    /**
	 * @param parser
	 * @return
	 */
	private Hashtable getUserInput(LogFileElement element, boolean parserParam) {

		Hashtable table = new Hashtable();
		List fields = element.getParser().getFields();
		int i = fields.size();

		List values = element.getValues();
		for (i=0; i<fields.size();i++) {
			ParserField field = (ParserField) fields.get(i);
			table.put(field.getId(), values.get(i));			
		}
		
		if(parserParam){
		List params = element.getParser().getParserParameters();
		for (int j=0;j<params.size();j++) {
			ParserParameter param = (ParserParameter) params.get(j);
			table.put(param.getName(),param.getValue());
		}
		}
		return table;
	}

	/*
     *
     */
    private TRCAgent createAgent(TRCProcessProxy process, LogFileElement element,TRCAgentProxy mergeAgentProxy) {
        TRCAgentProxy agent = null;
        final String name = getAgentName(element);
        
        String type = HyadesConstants.LOG_AGENT_TYPE;

        if (mergeAgentProxy!=null) { //user select an agent to merge
            if (bAppend) { 
                return mergeAgentProxy.getAgent();
            } else { //replace
            	((IHyadesExtendedResource)mergeAgentProxy.getAgent().eResource()).delete();
            	mergeAgentProxy.setAgent(null);
            	mergeAgentProxy.getProcessProxy().eResource().setModified(true);
            	mergeAgentProxy.setProcessProxy(null);
            }
        }
        else { //user did not select to merge any agent.
	        EList agents = process.getAgentProxies();
	        Object[] agentProxys = agents.toArray();
	
	        for (int i=0;i<agents.size();i++) {
	            agent = (TRCAgentProxy) agentProxys[i];
	
	            if (agent.getName().equals(name)) {
	            	errorMessage = null;
					Display.getDefault().syncExec(new Runnable(){
						public void run(){	            
							String msg = LogUIPlugin.getResourceString("STR_EXISTING_DATA_DESC_1");
							msg = TString.change(msg, "%1", name);
							TraceMergeUIDialog dialog = new TraceMergeUIDialog(getShell(), msg);
							int ret = dialog.open();
							if(ret==Window.CANCEL){
								errorMessage = CANCEL;												
							}
							if(ret==2){
								bAppend = true;
							}
							if(ret==3){
								bAppend = false;
							}
						}
					}
					);
	                
	                if(errorMessage == CANCEL)
	                	return null;
	                	
	                if (bAppend) {
	                    return agent.getAgent();
	                } else {
	                	((IHyadesExtendedResource)agent.getAgent().eResource()).delete();
	                	agent.setAgent(null);
	                   	agent.getProcessProxy().eResource().setModified(true);
	                	agent.setProcessProxy(null);
	                }
	            }
	        }
        }

		Date currentDate = new Date();
        String timestamp = DateFormat.getDateTimeInstance(DateFormat.FULL, DateFormat.FULL).format(currentDate);
        String rID = timestamp;

        rID = HyadesUtil.change(rID, " ", "");
        rID = HyadesUtil.change(rID, ":", "");
		
		rID += "-"+currentDate.getTime();
		

        String agentName = new StringBuffer(process.getNode().getMonitor().getName()).append("_").append(process.getNode().getName()).append("_").append(process.getPid()).append("_").append(rID).append("_").append(element.getParser().getDescription()).toString();

        agentName = HyadesUtil.change(agentName, " ", "");

        String pPath = process.eResource().getURI().toString();
        IPath path = new Path(pPath);

        if (path.segmentCount() > 1) {
            pPath = path.removeLastSegments(1).toString();
        }

        String fileName = agentName + getAgentNamePostfix(element);
        IPath filePath = new Path(pPath).append(fileName);

        URI uri = URI.createURI(filePath.toString());

        Resource agDoc = Resource.Factory.Registry.INSTANCE.getFactory(uri).createResource(uri);
		agDoc.setModified(true);
        EList agExt = agDoc.getContents();

        UIPlugin.getDefault().getResourceSet().getResources().add(agDoc); // prevents reloading later

        HierarchyFactory factory = UIPlugin.getDefault().getPerftraceFactory();

        //create process element
        agent = factory.createTRCAgentProxy();
        agent.setName(name);
        agent.setType(type);

        agent.setProcessProxy(process);
        process.getAgentProxies().add(agent);

        TRCAgent ag = factory.createTRCAgent();

        ag.setAgentProxy(agent);
        ag.setType(type);
        agExt.add(ag);

        // adding support for multiple files
//        SaveUtil.addDocument(agDoc);

        return ag;
    }
    
    private String getAgentName(LogFileElement element){
		String fPath = (String)getUserInput(element, false).get("file_path");
        String name = element.getParser().getName() + (fPath!=null ? " "+fPath : "");
    	return name;
    }

    /**
	 * @param element
	 * @return
	 */
	private String getAgentNamePostfix(LogFileElement element) {
		String res=null;
		Preferences store = HierarchyPlugin.getPlugin().getPluginPreferences();
		if(element.isUseLargeLogSupport() && store.contains(IHyadesResourceExtension.LARGE_RESOURCE_SUPPORT_CURRENT_DB_TYPE))
		{
			String activeDBType = store.getString(IHyadesResourceExtension.LARGE_RESOURCE_SUPPORT_CURRENT_DB_TYPE);
			for (Iterator iter = HyadesResourceExtensions.getInstance().entrySet().iterator(); iter.hasNext();) {
				Map.Entry entry = (Map.Entry) iter.next();
				if(entry.getKey() instanceof String)
				{
					IHyadesResourceExtension hyadesResourceFactory = (IHyadesResourceExtension)entry.getValue();
					String postfix = (String)entry.getKey();
					if(postfix.endsWith(".trcadb") && hyadesResourceFactory.getStoreType(postfix).equals(activeDBType))
					{
						return postfix;
					}
				}
			}
		}
		if(res==null)
			res = ".trcaxmi";
		return res;
	}

    /*
     *
     */
    private TRCProcessProxy createProcess(IContainer container, TRCNode node, LogFileElement element) {
        int pID = 0;

        String logPath = "";
        Hashtable elems = getUserInput(element, false);

        Iterator iterator = elems.keySet().iterator();

        if (iterator.hasNext()) {
            logPath += elems.get(iterator.next()).toString();
        }

        while (iterator.hasNext()) {
            logPath += (" " + elems.get(iterator.next()).toString());
        }

        TRCProcessProxy process = null;
        EList processes = node.getProcessProxies();
        Iterator i = processes.iterator();
        int counter = 0;

        while (i.hasNext()) {
            process = (TRCProcessProxy) i.next();
            counter++;

            if ((process.getName() != null) && process.getName().equals(logPath)) {
            	process.eResource().setModified(true);
                return process;
            }
        }

        TRCMonitor monitor = node.getMonitor();

        String rID = (new Date()).toString();

        rID = HyadesUtil.change(rID, " ", "");
        rID = HyadesUtil.change(rID, ":", "");

        String processName = node.getMonitor().getName() + "_" + node.getName() + "_" + counter + "_" + pID;

        String fileName = processName + "." + TraceConstants.PROCESS_EXT;
        IPath path = container.getFullPath().append(fileName);
        URI uri = URI.createURI("platform:/resource" + path.toString());

        Resource pDoc = Resource.Factory.Registry.INSTANCE.getFactory(uri).createResource(uri);
        pDoc.setModified(true);
        EList pExt = pDoc.getContents();

        UIPlugin.getDefault().getResourceSet().getResources().add(pDoc); // prevents reloading later

        // adding support for multiple files
//        SaveUtil.addDocument(pDoc);

        HierarchyFactory factory = UIPlugin.getDefault().getPerftraceFactory();

        process = factory.createTRCProcessProxy();
        process.setPid(pID);

        /* Ensure the runtime UUID is set on the process object */
        process.setRuntimeId(String.valueOf(pID));

        process.setName(logPath);
        process.setLaunchMode(0); //attach                       

        process.setNode(node);
        pExt.add(process);
        final TRCProcessProxy fProcess = process;
		
		refreshView(fProcess);		
        
        return process;
    }

    /**
     * Initializes states of the controls from the preference store.
     */
    private void initializeValues() {
    	
        IConfigurationElement[] config = Platform.getExtensionRegistry().getConfigurationElementsFor("org.eclipse.hyades.logging.parsers", "logParser");

        for (int idx = 0; idx < config.length; idx++) {
            IConfigurationElement elem = config[idx];

            if (logParserId.trim().equals(elem.getAttribute("id").trim())) {
                _parser = new LogParserItem(elem.getAttribute("id"), elem.getAttribute("name"), elem.getAttribute("ui_name"), elem.getAttribute("description"));
                _parser.setConfigurationElement(elem);
                _parser.setClass(elem.getAttribute("class"));
                _parser.setUIType(elem.getAttribute("ui_type"));

                String iconPath = elem.getAttribute("icon");

                if ((iconPath != null) && iconPath.length() > 0) {
                    URL pluginURL = Platform.getBundle(elem.getDeclaringExtension().getNamespace()).getEntry("/");

                    try {
                        URL url = new URL(pluginURL, iconPath);
                        ImageDescriptor image = ImageDescriptor.createFromURL(url);

                        _parser.setImageDescriptor(image);
                    } catch (Exception exc) {
                        // if the URL does not have a valid format, just log and ignore the exception
                        IStatus status = new Status(IStatus.ERROR, LogUIPlugin.getPluginId(), 2, "exception.malformedURL", exc); //$NON-NLS-1$

                        LogUIPlugin.getDefault().getLog().log(status);

                        continue;
                    }
                }

                IConfigurationElement[] felems = elem.getChildren("field");

                for (int i = 0; i < felems.length; i++) {
                    IConfigurationElement felem = felems[i];
                    ParserField pfield = new ParserField(felem.getAttribute("id"), felem.getAttribute("name"));
					pfield.setConfigurationElement(felem);
                    pfield.setDefaultValue(felem.getAttribute("defaultValue"));
                    pfield.setContextId(felem.getAttribute("helpContextId"));
                    pfield.setTooltip(felem.getAttribute("tooltip"));
                    pfield.setFilters(felem.getAttribute("browseType"));
                    pfield.useBrowse(felem.getAttribute("useBrowse").equals("true"));
                    pfield.setUiType(felem.getAttribute("ui_type"));
					pfield.setListener(felem.getAttribute("listener"));

                    _parser.addEntry(pfield);
                }
                
				IConfigurationElement[] fparameters = elem.getChildren("parserParameter");
				for(int i=0; i<fparameters.length; i++)
				{
				   IConfigurationElement fparameter = fparameters[i];
				   ParserParameter para = new ParserParameter(fparameter.getAttribute("name"),fparameter.getAttribute("value"));
				   _parser.addParameter(para);			   
				}
            }
        }
    }

    private void openLogView(TRCAgent agent) {
    	
		IAssociationMappingRegistry registry = TraceAssociationManager.getTraceViewMappingRegistry();
		IAssociationMapping mapping = registry.getAssociationMapping(HyadesUIPlugin.EP_ANALYZER_EXTENSIONS);
    	
		if(mapping == null)
			return;
    	
		IAssociationDescriptor[] array = mapping.getAssociationDescriptors(agent.getType());

		if(array.length > 0)
		{
			IAssociationDescriptor desc = array[0];
			try {
					
				IAction action = (IAction)desc.createImplementationClassInstance();
				
				if(action != null)
				{					
					action.run();
				}								
			} catch (Exception exc) {
				exc.printStackTrace();
			}			
		}
		IWorkbenchPage page = UIPlugin.getActivePage();
		try {
			fViewer = (LogNavigator) page.showView(PDPerspective.ID_LOG_NAVIGATOR_VIEW);
			StructuredViewer viewer = fViewer.getViewer();
			if(viewer!=null){							
				viewer.refresh();
			}
		}
		catch (Exception e) {
			HyadesUIPlugin.logError(e);
		}
    }

    //~ Inner Classes ------------------------------------------------------------------------------

    /* RKD:  ProcessListenerImpl to be instantiated and used by the importers */
    class ProcessListenerImpl implements ProcessListener {
        TRCAgentProxy trcAgent;
        TRCProcessProxy trcProcess;

        public ProcessListenerImpl(TRCProcessProxy process, TRCAgentProxy agent) {
            trcProcess = process;
            trcAgent = agent;
        }

        public void processExited(Process process) {
            trcAgent.setActive(false);
            trcAgent.setAttached(false);
            trcAgent.setMonitored(false);

            Display.getDefault().asyncExec(new Runnable() {
                    public void run() {
                        //update ui
                    	if (shouldOpenLogView)
                    		fViewer.getViewer().refresh(trcAgent);
                    }
                });
        }

        public void processLaunched(Process process) {
            try {
                if (trcProcess != null) {
                    trcProcess.setPid(Integer.parseInt(process.getProcessId()));
                    trcProcess.setRuntimeId(process.getUUID());
                    trcProcess.setActive(true);

                    /*Display d = fShell.getDisplay();

                    d.asyncExec(new Runnable() {
                            public void run() {
                                //update ui
                                fViewer.getViewer().refresh(trcProcess);
                                fViewer.selectObject(trcAgent);
                            }
                        });*/
                }
            } catch (Exception exc) {
                exc.printStackTrace();
            }
        }
    }
    
    class LogAgentListener implements AgentListener{
		
		private RemoteLogImportJob job;
    	private TRCAgentProxy aproxy;		
		private TRCProcessProxy trcProcess ;
		private boolean isActive = false;

		public LogAgentListener(RemoteLogImportJob job, TRCAgentProxy aproxy, TRCProcessProxy trcProcess) {
			this.job = job;
			this.aproxy = aproxy;
			this.trcProcess = trcProcess;
		}
		public void agentActive(Agent agent) {	        
			aproxy.setActive(true);
			job.setLogAgentActive(true);
			job.resetTimeOut();
			aproxy.setAttached(true);
			aproxy.setMonitored(true);			
			LoadersUtils.registerAgentInstance(aproxy, agent);

			Display.getDefault().asyncExec(new Runnable() {
					public void run() {
						//update ui
						ProfileEvent event = UIPlugin.getDefault().getProfileEvent();

						event.setSource(aproxy);
						event.setType(ProfileEvent.START_MONITOR);
						UIPlugin.getDefault().notifyProfileEventListener(event);
					}
				});

			try {
				XMLTraceDataProcessor dataProcessor = new XMLTraceDataProcessor(aproxy){
					/* (non-Javadoc)
					 * @see org.eclipse.hyades.trace.ui.internal.piclient.XMLTraceDataProcessor#dataServerExited()
					 */
					public void dataServerExited()
					{
						super.dataServerExited();
						LoadersUtils.deregisterAgentInstance(aproxy);
						Display.getDefault().asyncExec(new Runnable() {
								public void run() {
									//update ui
									ProfileEvent event = UIPlugin.getDefault().getRefreshViewEvent(aproxy);
									UIPlugin.getDefault().notifyProfileEventListener(event);
								}
							});
						
						
				
//											loadEvent("</CommonBaseEvents>".getBytes(), "</CommonBaseEvents>".getBytes().length);
					}
				};
//									dataProcessor.loadEvent("<CommonBaseEvents>".getBytes(), "<CommonBaseEvents>".getBytes().length);

				agent.startMonitoring(dataProcessor);
        
			} catch (Exception e) {
				LogUIPlugin.log(e);
				//final String agentErrorMessage = e.toString();
				errorMessage = LogUIPlugin
				.getResourceString("IMPORT_LOG_FILE_RAC_ERROR_");
				reasonAction = e.getLocalizedMessage();
				MultiStatus status = new MultiStatus(LogUIPlugin.getPluginId(), 0, errorMessage, e);
				if(reasonAction!=null){
					status.add(new Status(IStatus.ERROR, LogUIPlugin.getPluginId(), 0, reasonAction, e));
				}else{
					status.add(new Status(IStatus.ERROR, LogUIPlugin.getPluginId(), 0, e.getStackTrace().toString(), e));
				}
				job.setStatus(status);
				job.setFinished(true);
//				Display.getDefault().asyncExec(new Runnable() {
//						public void run() {
//							MessageDialog.openError(UIPlugin.getActiveWorkbenchShell(), LogUIPlugin.getResourceString("LOGGING_MESSAGE"), agentErrorMessage);
//						}
//					});
			}
		}

		public void agentInactive(Agent agent) {
			job.setLogAgentActive(false);
			aproxy.setActive(false);
			aproxy.setAttached(false);
			aproxy.setMonitored(false);

			trcProcess.setActive(false);
			//System.out.println("SET FINISHED = true");
			job.setFinished(true);
			Display.getDefault().asyncExec(new Runnable() {
					public void run() {
						//update ui
						ProfileEvent event = UIPlugin.getDefault().getProfileEvent();

						event.setSource(aproxy);
						event.setType(ProfileEvent.TERMINATE);
						UIPlugin.getDefault().notifyProfileEventListener(event);
					}
				});
		}

		public void error(Agent agent, String errorId, String message) {
			reasonAction = errorId.concat(": ").concat(message);
			errorMessage = LogUIPlugin
			.getResourceString("IMPORT_LOG_FILE_RAC_ERROR_");

			MultiStatus status = new MultiStatus(LogUIPlugin.getPluginId(), 0, errorMessage, null);
			if(reasonAction!=null)
				status.add(new Status(IStatus.ERROR, LogUIPlugin.getPluginId(), 0, reasonAction, null));
			job.setStatus(status);
			job.setFinished(true);

//			Display.getDefault().asyncExec(new Runnable() {
//					public void run() {
//						MessageDialog.openError(UIPlugin.getActiveWorkbenchShell(), LogUIPlugin.getResourceString("LOGGING_MESSAGE"), errMsg);
//					}
//				});
		}

		public void handleCommand(Agent agent, CommandElement command) {
		}
	    
    }
    
    class ErrorAgentListener implements AgentListener{

    	private String errorAgentMessage = null;
    	private boolean isActive = false;
    	
		public ErrorAgentListener(){
		}
		
		public void agentActive(Agent agent) {
			isActive = true;
			try {
				agent.startMonitoring(new DataProcessor() {
					public void incommingData(byte[] buffer, int length, java.net.InetAddress peer) {
//						final String agentErrorMessage = new String(buffer, 0, length);
						errorAgentMessage = new String(buffer, 0, length);						
//						Display.getDefault().asyncExec(new Runnable() {
//							public void run() {
////								fShell = Workbench.getInstance().getActiveWorkbenchWindow().getShell();
////								MessageDialog.openError(fShell, LogUIPlugin.getResourceString("LOGGING_MESSAGE"), agentErrorMessage);
//							}
//						});
					}

					public void incommingData(char[] buffer, int length, java.net.InetAddress peer) {
					}

					public void invalidDataType(byte[] data, int length, java.net.InetAddress peer) {
					}

					public void waitingForData() {
					}
				});
			} catch (Exception e) {
//				final String agentErrorMessage = e.toString();
				errorAgentMessage = e.toString();

				Display.getDefault().asyncExec(new Runnable() {
						public void run() {
//							fShell = Workbench.getInstance().getActiveWorkbenchWindow().getShell();
//							MessageDialog.openError(fShell, LogUIPlugin.getResourceString("LOGGING_MESSAGE"), agentErrorMessage);
						}
					});
			}
		}

		public void agentInactive(Agent agent) {
			isActive = false;
		}

		public void error(Agent agent, String errorId, String message) {
//			final String errMsg = errorId.concat(": ").concat(message);
			errorAgentMessage = errorId.concat(": ").concat(message);			
//			Display.getDefault().asyncExec(new Runnable() {
//					public void run() {
////						fShell = Workbench.getInstance().getActiveWorkbenchWindow().getShell();
////						MessageDialog.openError(fShell, LogUIPlugin.getResourceString("LOGGING_MESSAGE"), errMsg);
//					}
//				});
		}

		public void handleCommand(Agent agent, CommandElement command) {
		}
		
		String getErrorMessage(){
			return errorAgentMessage;
		}
		
		boolean isActive(){
			return isActive;
		}
	} 
    
    class LocalLogImportJob extends LogImportJob{
    	
    	private Parser parser = null;
    	private XMLLoader xmlLoader = null;
    	private LocalLogParserLoader parserLoader; 
    	
    	LocalLogImportJob(String name, LogFileElement element, Parser parser, XMLLoader xmlLoader, LocalLogParserLoader parserLoader, TRCAgent trcAgent){
    		super(name);
    		this.element = element;
    		this.parser = parser;
    		this.xmlLoader =xmlLoader;
    		this.parserLoader = parserLoader;
    		this.trcAgent = trcAgent;
    	}
    	

    	Parser getParser(){
    		return parser;
    	}
    	
    	TRCAgent getTRCAgent(){
    		return trcAgent;
    	}
    	
    	IProgressMonitor getMonitor(){
    		return mon;
    	}
    	
    	boolean hasFinished(){
    		return hasFinished;
    	}
    	
    	void setFinished(boolean newValue){
    		hasFinished = newValue;
    	}
    	    	
    	int[] getProgressStatus(){
    		parseStatusString();
    		return status;
    	}
    	
    	String getTaskName(){
    		return getAgentName(element);
    	}
    	
    	IStatus stop(){
    		
    		
    		try{
    			parser.stop();
    		} catch (Exception e) {    				
    			return new Status(IStatus.ERROR, LogUIPlugin.getPluginId(), 0, LogUIPlugin
    					.getResourceString("IMPORT_LOG_FILE_STOP_PARSE_MSG"), e);
    		}    		
    		return new Status(IStatus.OK, LogUIPlugin.getPluginId(), 0, "", null);
    	}
    	
    	private void parseStatusString(){
    		//System.out.println("LocalLogImport ---- getStatus is called");
    		
    		
    		statusString = parser.getStatus();
    		
    		if(statusString==null){
    			return;
    		}
    		//System.out.println("STATUS_STRING = "+statusString);
    		if(!statusString.startsWith("total=") || statusString.indexOf("processed")<0){
    			return;
    		}    		
    		
   			if(status[0]==0){
	    		int stopTotal = statusString.indexOf(" ");
	    			status[0] = new Integer(statusString.substring(6,stopTotal)).intValue();
		    		//System.out.println("TOTAL="+status[0]);    			
    		}
    		int startProcessed = statusString.indexOf("processed");
    		int endProcessed = statusString.indexOf(" ",startProcessed);
    		if(endProcessed<0){
    			endProcessed = statusString.length();
    		} 	    		
    		status[1] = new Integer(statusString.substring(startProcessed+10,endProcessed)).intValue();
    		//System.out.println("PROCESSED="+status[1]);    		
    	}
    	
    	/* (non-Javadoc)
		 * @see org.eclipse.core.runtime.jobs.Job#run(org.eclipse.core.runtime.IProgressMonitor)
		 */
		protected IStatus run(IProgressMonitor monitor) {
			mon = monitor;
			if(monitorThread!=null){
				monitorThread.register(LocalLogImportJob.this);
			}
			return runOnLocalHost(monitor);
		}
		
		protected IStatus runOnLocalHost(IProgressMonitor monitor){
						
			LocalGLALogger glaLogger = new LocalGLALogger();
			//glaLogger.setXMLLoader(xmlLoader);
			
			parserLoader.setParserLogger(glaLogger);
			try{	
				xmlLoader.loadEvent("<CommonBaseEvents>".getBytes(),
						"<CommonBaseEvents>".getBytes().length);				
				parserLoader.startParsing();
				xmlLoader.loadEvent("</CommonBaseEvents>".getBytes(),
						"</CommonBaseEvents>".getBytes().length);				


			} catch (final Exception e) {
				e.printStackTrace();				
				errorMessage = LogUIPlugin
				.getResourceString("IMPORT_LOG_FILE_RAC_ERROR_");
				reasonAction = e.getLocalizedMessage();
		        MultiStatus status = new MultiStatus(LogUIPlugin.getPluginId(), 0, errorMessage, e);
		        if(reasonAction!=null)	
		        	status.add(new Status(IStatus.ERROR, LogUIPlugin.getPluginId(), 0, reasonAction, e));
				if(glaLogger.getLogDetails()!=null){
					detailMsg = glaLogger.getLogDetails().trim();
					status.add(new Status(IStatus.ERROR, LogUIPlugin.getPluginId(), 0, detailMsg, e));
				}
				setStatus(status);
		        stop();
			}finally{
				try{	
					xmlLoader.cleanUp();
				}catch(Exception e){
					//fail silently
				}
			}
			
			if(jobStatus==null){				
				MultiStatus status = new MultiStatus(LogUIPlugin.getPluginId(), 0, "", null);
				status.add(new Status(IStatus.OK, LogUIPlugin.getPluginId(), 0, "", null));
				setStatus(status);
			}
			//System.out.println("LOCAL JOB " + getTaskName() +" FINISHED..."+jobStatus);
			return jobStatus;
			
		}
		
		
    }
    
    class RemoteLogImportJob extends LogImportJob implements AgentListener{
    	
    	private Node node = null;
    	private Process process = null;
    	private Agent agent = null;    	
    	private boolean monitoringFinished = false;// variable used to keep the progress and GLA error monitoring status
    	private long startTime = 0;
    	private int timeout = 2000; //in case of an error wait for the  main error message or the error details.
    	private int importTimeOut = 25000;//if no log agent active is received in 25 sec. abort the import
    	private List errorMessages = null;
    	private ErrorAgentListener errorListener = null;
    	private boolean isLogAgentActive = false;
    	private boolean logAgentActiveEventReceived = false;
    	private final int MAX_NR_OF_ERRORS_DISPLAYED = 100;
    	
    	RemoteLogImportJob(String name, LogFileElement element, Node node, Process process, Agent agent, TRCAgent trcAgent){
    		super(name);
    		this.element = element;
    		this.node = node;
    		this.process = process;
    		this.trcAgent = trcAgent;    		
    	}

    	Process getProcess(){
    		return process;
    	}
    	
    	TRCAgent getTRCAgent(){
    		return trcAgent;
    	}
    	
    	IProgressMonitor getMonitor(){
    		return mon;
    	}
    	
    	boolean hasFinished(){
    		return hasFinished;
    	}
    	
    	boolean isMonitoring(){
    		return !monitoringFinished;
    	}
    	
    	void setMonitoringFinished(boolean newValue){
    		monitoringFinished = newValue;
    	}
    	
    	void setFinished(boolean newValue){
    		hasFinished = newValue;
    	}
 	    	
    	int[] getProgressStatus(){
    		parseStatusString();
    		return status;
    	}
    	
    	String getTaskName(){
    		return getAgentName(element);
    	}
    	
    	void setErrorListener(ErrorAgentListener listener){
    		errorListener = listener;
    	}
    	
    	List getErrorMessages(){
    		return errorMessages;
    	}
    	
    	boolean isLogAgentActive(){
    		return isLogAgentActive;
    	}
    	
    	void setLogAgentActive(boolean newValue){
    		isLogAgentActive = newValue;
    		if(newValue){
    			logAgentActiveEventReceived = true;
    		}
    	}
    	
    	IStatus stop(){
    		
    		try {
    			if(process != null && process.isActive())								
    			{
    				Node node = process.getNode();
    				node.killProcess(process);
    			}
    		}
    		catch(NotConnectedException exc1)
			{
    			exc1.printStackTrace();
			}		
    		catch(InactiveProcessException exc2)
			{
    			exc2.printStackTrace();
			}
    		ProfileEvent pEvent = UIPlugin.getDefault().getProfileEvent();
    		pEvent.setSource(process);
    		pEvent.setType(ProfileEvent.TERMINATE);
    		final ProfileEvent fpEvent = pEvent; 
			Display.getDefault().asyncExec(new Runnable() {
				public void run() {
					//update ui
					UIPlugin.getDefault().notifyProfileEventListener(fpEvent);				}
			});	
    		return new Status(IStatus.OK, LogUIPlugin.getPluginId(), 0, "", null);
    	}
    	
    	private void parseStatusString(){
    		
    		if(statusString==null){
    			return;
    		}
//    		System.out.println("STATUS_STRING = "+statusString);
    		if(statusString.startsWith("total=") && statusString.indexOf("processed")>0){    			
       			if(status[0]==0){
    	    		int stopTotal = statusString.indexOf(" ");
    	    			status[0] = new Integer(statusString.substring(6,stopTotal)).intValue();
    		    		//System.out.println("TOTAL="+status[0]);    			
        		}
        		int startProcessed = statusString.indexOf("processed");
        		int endProcessed = statusString.indexOf(" ",startProcessed);
        		if(endProcessed<0){
        			endProcessed = statusString.length();
        		} 	    		
        		status[1] = new Integer(statusString.substring(startProcessed+10,endProcessed)).intValue();
        		//System.out.println("PROCESSED="+status[1]);    			
    		}else if (statusString.startsWith("glalog=")){
    			if(errorMessages==null){
    				errorMessages = new ArrayList();
    			}
				String errString = statusString.substring(7);
    			if(!errorMessages.contains(errString)){	
    				errorMessages.add(errString);
				}				
    		}/*else{
    			// this must be the main error message
    			if(errorMessages==null){
    				errorMessages = new ArrayList();
    			}
    			errorMessages.add(0,statusString);    			
    		}*/   		
    	}
    	
    	/* (non-Javadoc)
		 * @see org.eclipse.core.runtime.jobs.Job#run(org.eclipse.core.runtime.IProgressMonitor)
		 */
		protected IStatus run(IProgressMonitor monitor) {
			mon = monitor;
			IStatus status = runOnRemoteHost(monitor);
			String tempAdapterPath = element.getTempAdapterPath();
			if(tempAdapterPath!=null && tempAdapterPath.length()>0){
				GLAFilterHelper.deleteTempAdapter(node, tempAdapterPath);
			}
			return status;
		}
		
		protected IStatus runOnRemoteHost(IProgressMonitor monitor){
	        try {
	            process.launch();
	        }
	        // The RemoteLogParserLoader application is not configured in the RAC's serviceconfig.xml file: 
	         catch (NoSuchApplicationException n) {
	               // trying to import a new log type with an old RAC
	               final String parserClass = element.getParser().getId().trim();
	               if ((!parserClass.equals("com.ibm.etools.logging.parsers.WASActivityLogParser"))){
	                  errorMessage = LogUIPlugin.getResourceString("IMPORT_LOG_FILE_RAC_ERROR_");
	                  reasonAction = LogUIPlugin.getResourceString("IMPORT_LOG_FILE_RAC_REASON_OLD_RAC");
	  		          MultiStatus status = new MultiStatus(LogUIPlugin.getPluginId(), 0, errorMessage, n);	
			          status.add(new Status(IStatus.ERROR, LogUIPlugin.getPluginId(), 0, reasonAction, n));
			          return status;
	               }
	               Hashtable userInput = getUserInput(element, true);
	               String was_home = ((String) (userInput.get("was_home")));
	               final String activityPath = ((String) (userInput.get("file_path")));
	               //pre V5 process
	               process = ProcessFactory.createProcess(node, "ActivityLogParser", ("com.ibm.etools.logging.was.WASActivityLogParser".concat(" \"").concat(activityPath).concat("\" \"").concat(was_home).concat("\"")));
	               process.addProcessListener(new ProcessListenerImpl(trcAgent.getAgentProxy().getProcessProxy(), trcAgent.getAgentProxy()));
	               final Agent preV5ProcessAgent = AgentFactory.createAgent(process, "WAS Activity Log Agent", HyadesConstants.LOG_AGENT_TYPE);
	               preV5ProcessAgent.setAutoAttach(true);
	               preV5ProcessAgent.addAgentListener(new LogAgentListener(RemoteLogImportJob.this, trcAgent.getAgentProxy(), trcAgent.getAgentProxy().getProcessProxy()));
	               final Agent preV5ErrorAgent = AgentFactory.createAgent(process, "WAS Activity Log Error Agent", HyadesConstants.LOG_AGENT_TYPE);
	               preV5ErrorAgent.setAutoAttach(true);
	               ErrorAgentListener errorAgentListener = new ErrorAgentListener();
	               preV5ErrorAgent.addAgentListener(errorAgentListener);
	               setErrorListener(errorAgentListener);
	               agent = preV5ErrorAgent;
	               try {
	               		process.launch();
	               }
	               catch (Exception e) {
	                   errorMessage = LogUIPlugin.getResourceString("IMPORT_LOG_FILE_RAC_ERROR_");
	                   reasonAction = LogUIPlugin.getResourceString("IMPORT_LOG_FILE_RAC_REASON_UNKNOWN");
	                   LogUIPlugin.log(e);
	                   MultiStatus status = new MultiStatus(LogUIPlugin.getPluginId(), 0, errorMessage, e);	
	                   status.add(new Status(IStatus.ERROR, LogUIPlugin.getPluginId(), 0, reasonAction, e));
	                   return status;
	               }	               
	             
	        } catch (Exception e) {
	            errorMessage = LogUIPlugin.getResourceString("IMPORT_LOG_FILE_RAC_ERROR_");
	            reasonAction = LogUIPlugin.getResourceString("IMPORT_LOG_FILE_RAC_REASON_UNKNOWN");
	            LogUIPlugin.log(e);
	            MultiStatus status = new MultiStatus(LogUIPlugin.getPluginId(), 0, errorMessage, e);	
	            status.add(new Status(IStatus.ERROR, LogUIPlugin.getPluginId(), 0, reasonAction, e));
	            return status;
	        }
//	      	monitor progress activity	
	         return monitorProgress(monitor);
		}

		private IStatus monitorProgress(IProgressMonitor monitor){
						
			boolean taskBegan = false;
			int workedUnits = 0;			
			while(isActive()){									
				try{
					status = getProgressStatus();
					
				}catch(Exception e){
					e.printStackTrace();
				}
				if (!taskBegan && status[0] > 0) {
					monitor.beginTask(getTaskName(), status[0]);
//					System.out.println("TOTAL WORK="+status[0]+" element="+getTaskName());
					taskBegan = true;
				}
				
				if(status[0]!=0 && status[1]!=0 && status[0]<=status[1]){
					setFinished(true);
					monitor.done();					
				}
				
				if (monitor.isCanceled()) {
					stop();
					setFinished(true);
					setStatus(new Status(IStatus.CANCEL, LogUIPlugin.getPluginId(), 0, "", null));
					monitor.done();
				}					
				if (status[1] > 0) {						
					
					monitor.worked(status[1]-workedUnits);					
					int percent = (status[1]*100)/status[0];
					monitor.subTask(getTaskName()+ " "+percent+"%" );
					workedUnits = status[1];
				}else{
					monitor.subTask(getTaskName());
				}
				try{					
//					 wait for 2 second(s) and resume scanning progress status
					Thread.sleep(2000);
				}catch(Exception e){
					
				}				
			}
			if(hasErrors()){
				generateErrorMessage();
			}
	        if(jobStatus==null){
	        	jobStatus = new Status(IStatus.OK, LogUIPlugin.getPluginId(), 0, "", null);
	        }
	        monitor.done();
	        //System.out.println("REMOTE JOB FINISHED "+jobStatus);
			return jobStatus;
			
		}
		
		
		private boolean hasErrors(){
			return (errorListener!=null && errorListener.getErrorMessage()!=null) || (errorMessages!=null && !errorMessages.isEmpty()); 
		}
		
		private boolean isActive(){
			if(!monitoringFinished){				
				if(!isLogAgentActive && !logAgentActiveEventReceived){
					if(!timeOut(importTimeOut)){
						return true;
					}else{
		        		String errMsg = LogUIPlugin.getResourceString("AgentControllerInvalidConfigError");
						IStatus status = new Status(IStatus.ERROR, LogUIPlugin.getPluginId(), 0, errMsg, null);
		        		setStatus(status);
		        		return false;
					}
				}

				if(!hasFinished || errorListener.isActive()){					
					return true;
				}else{
					//if both GLA and status monitoring agents have finished
					//allow a timeout until the job is finished to receive all error messages					
					return !timeOut(timeout);
				}					
			}
			else{ 
				if(!hasFinished || errorListener.isActive()){
					return true;
				}else{
					//if both GLA and status monitoring agents have finished
					//allow a timeout until the job is finished to receive all error messages
					return !timeOut(timeout);
				}
			}			
		}
		
		private boolean timeOut(int delta){
			if(startTime == 0){			
				startTime = System.currentTimeMillis();
			}
			long currentTime = System.currentTimeMillis();
			if(currentTime-startTime>delta){
				return true;
			}
			return false;
		}
		
		void resetTimeOut(){
			startTime = 0;
		}
		
    	public void agentActive(Agent agent) {
    		try {
    			
    			DataProcessor dataProcessor = new DataProcessor() {
	    				public void incommingData(byte[] buffer, int length, java.net.InetAddress peer) {    					
	
	    					statusString = new String(buffer, 0, length);
	    					parseStatusString();	
//	    					generateErrorMessage();
	    				}
	    				
	    				public void incommingData(char[] buffer, int length, java.net.InetAddress peer) {
	    				}
	    				
	    				public void invalidDataType(byte[] data, int length, java.net.InetAddress peer) {
	    				}
	    				
	    				public void waitingForData() {
	    				}
	    		};    			
    			agent.startMonitoring(dataProcessor);
    		} catch (Exception e) {
    		}
    	}
    	
    	public void agentInactive(Agent agent) {             
//            generateErrorMessage();
            setFinished(true);
	    	setMonitoringFinished(true);
    	}
    	
    	private void generateErrorMessage(){

    		if(errorMessages!=null && errorMessages.size()>0){        		
            	String mainErrMsg = (String)errorMessages.get(0); 
            	MultiStatus status = null;
            	if(errorListener!=null && errorListener.getErrorMessage()!=null && errorListener.getErrorMessage().trim().length()>0){
    				mainErrMsg = errorListener.getErrorMessage();
    				StringTokenizer stringTokenizer = new StringTokenizer(mainErrMsg, "\r\n");
    				while(stringTokenizer.hasMoreTokens())
    				{
    					String token = stringTokenizer.nextToken();
    					if(token.length()>0){
    						if(status==null){
    							status = new MultiStatus(LogUIPlugin.getPluginId(), 0, token, null);				
    						}else{
    							status.add(new Status(IStatus.ERROR, LogUIPlugin.getPluginId(), 0, token, null));			
    						}

    					}
    				}
    				if(status==null){
    					status = new MultiStatus(LogUIPlugin.getPluginId(), 0, mainErrMsg, null);
    				}
            	}
    				
                StringBuffer errMsg = new StringBuffer();
                final String NL = System.getProperties().getProperty("line.separator");
                int min = Math.min(MAX_NR_OF_ERRORS_DISPLAYED, errorMessages.size());
        		for(int i=0;i<min;i++){
        			errMsg.append(errorMessages.get(i));
        			status.add(new Status(IStatus.ERROR, LogUIPlugin.getPluginId(), 0, errorMessages.get(i).toString(), null));
                }
        		setStatus(status);
            }else{
    			if(errorListener!=null && errorListener.getErrorMessage()!=null && errorListener.getErrorMessage().trim().length()>0){    				
//    				Status status = new Status(IStatus.ERROR, LogUIPlugin.getPluginId(), 0, errorListener.getErrorMessage(), null);
    				IStatus status = null;
    				StringTokenizer stringTokenizer = new StringTokenizer(errorListener.getErrorMessage(), "\r\n");
    				if(stringTokenizer.countTokens()>1){    				
	    				while(stringTokenizer.hasMoreTokens())
	    				{
	    					String token = stringTokenizer.nextToken();
	    					if(token.length()>0){
	    						if(status==null){
	    							status = new MultiStatus(LogUIPlugin.getPluginId(), 0, token, null);				
	    						}else{
	    							((MultiStatus)status).add(new Status(IStatus.ERROR, LogUIPlugin.getPluginId(), 0, token, null));			
	    						}
	
	    					}
	    				}
    				}
    				if(status==null){
    					status = new Status(IStatus.ERROR, LogUIPlugin.getPluginId(), 0, errorListener.getErrorMessage(), null);
    				}
    				
    				setStatus(status);   
    			}
            }
    	}
    	
    	public void error(Agent agent, String errorId, String message) {
//    		final String errMsg = errorId.concat(": ").concat(message);
//    		Display.getDefault().asyncExec(new Runnable() {
//    			public void run() {
//    				fShell = Workbench.getInstance().getActiveWorkbenchWindow().getShell();
//    				MessageDialog.openError(fShell, LogUIPlugin.getResourceString("LOGGING_MESSAGE"), errMsg);
//    			}
//    		});
    	}
    	
    	public void handleCommand(Agent agent, CommandElement command) {
    	}
		
    }

    abstract class LogImportJob extends Job{
    	
    	protected LogFileElement element = null;
    	protected TRCAgent trcAgent = null;
    	protected boolean hasFinished = false;//GLA processing status   
    	protected int[] status = new int[2];
    	protected IProgressMonitor mon = null;
    	protected String statusString = null;
    	protected IStatus jobStatus = null;
    	
    	LogImportJob(String name){
    		super(name);
    	}
    	abstract int[] getProgressStatus(); 
    	
    	IStatus getStatus(){
    		return jobStatus;
    	}
    	
    	public void setStatus(IStatus newStatus){
    		jobStatus = newStatus;
    	}
    	abstract void setFinished(boolean newValue);
    	abstract boolean hasFinished();
    	abstract IStatus stop();
    	abstract TRCAgent getTRCAgent();
    }
    
    class LogImportJobListener extends JobChangeAdapter{
    	
    	/* (non-Javadoc)
    	 * @see org.eclipse.core.runtime.jobs.IJobChangeListener#done(org.eclipse.core.runtime.jobs.IJobChangeEvent)
    	 */
    	public void done(IJobChangeEvent event) {
    		if(event.getResult().getSeverity()==IStatus.CANCEL){
    			((LogImportJob)event.getJob()).stop();
    		}
    		((LogImportJob)event.getJob()).setFinished(true);
    		((LogImportJob)event.getJob()).getTRCAgent().getAgentProxy().setActive(false);
    		final IJobChangeEvent fEvent = event;
    		Display.getDefault().asyncExec(new Runnable() {
    			public void run() {
    				//update ui
    				if(fViewer!=null)	
    					fViewer.getViewer().refresh(((LogImportJob)fEvent.getJob()).getTRCAgent().getAgentProxy());
    			}
    		});		
    		if(((LogImportJob)event.getJob()).getTRCAgent()==displayAgent){								
    			refreshView(((LogImportJob)event.getJob()).getTRCAgent().getAgentProxy());
    		}
    	}
    }
    
    
    class StatusListener extends Thread{
    	
    	private int s;
    	private LocalLogImportJob[] jobs;
    	
    	public StatusListener(int s){    		
    		this.s = s;
    		initialize();
    	}
    	
    	private void initialize(){
    		jobs = new LocalLogImportJob[s];
    	}
    	    	
    	/*
		 * (non-Javadoc)
		 * 
		 * @see java.lang.Thread#run()
		 */
		public void run() {
			
			int[] status = null;
			int timeout = 120000;// allow 2 min. for the parser to return
			// status information;
			long[] startTime = new long[s];
			Parser parser = null;
			Agent agent = null;
			long currentTime;
			boolean[] tasksBegan = new boolean[s];
			int[] unitsWorked = new int[s];
//			System.out.println("SIZE="+s);
			for(int k=0;k<s;k++){
				tasksBegan[k] = false;
			}
			LocalLogImportJob element = null;
			boolean end = false;
			while(!end){
				for(int i=0;i<s;i++){
					
					synchronized(jobs){
						element = jobs[i];
//						if(element!=null)
//							System.out.println("i="+i+" element="+element.getTaskName());
						if(element!=null && !element.hasFinished()){
		
							if(startTime[i]==0){
								startTime[i]=System.currentTimeMillis();
							}					
							try{	
								status = element.getProgressStatus();
		
							}catch(Exception e){
								e.printStackTrace();
							}
							if (!tasksBegan[i] && status[0] > 0) {
								element.getMonitor().beginTask(element.getTaskName(), status[0]);
//								System.out.println("TOTAL WORK="+status[0]);
								tasksBegan[i] = true;
							}
		
		//					currentTime = System.currentTimeMillis();
		//					if (currentTime - startTime[i] > timeout) {
		//						tasksEnded[i] = true;
		//					}
							if(status[0]!=0 && status[1]!=0 && status[0]<=status[1]){
								element.setFinished(true);
								element.getMonitor().done();
							}
		
							if (element.getMonitor().isCanceled()) {
								element.setStatus(new Status(IStatus.CANCEL, LogUIPlugin.getPluginId(), 0, "", null));
								element.stop();
								element.setFinished(true);
								element.getMonitor().done();
							}					
							if (status[1] > 0) {						
								element.getMonitor().worked(status[1]-unitsWorked[i]);
//								System.out.println("WORKED="+(status[1]-unitsWorked[i]));								
								int percent = status[1]*100/status[0];
								element.getMonitor().subTask(" "+percent+"%" );
								unitsWorked[i] = status[1];								
							}else{
								element.getMonitor().subTask(element.getTaskName());
							}
						}
						// verify if all jobs have ended, if so stop monitoring
						end = true;						
						boolean jobStatus = false;
						for(int j=0;j<s;j++){						
//							if(jobs[j]!=null)
////								System.out.println("jobs["+j+"] "+jobs[j].getTaskName());
							jobStatus = jobs[j]!=null ? (jobs[j]).hasFinished() :  false;
							if(jobStatus)
								jobs[j].getMonitor().done();
							end = end && jobStatus;  
						}
					}
//					System.out.println("MONITORING THREAD ENDED="+end);
				}
				try{	
					sleep(2000);// wait for 2 seconds and resume scanning status
				}catch(Exception e){
					
				}				
			}
			//listener.removeJobStatusChangeListener(StatusListener.this);
		}
		
		public synchronized LocalLogImportJob[] getJobs(){
			return jobs;
		}
				
		public synchronized void register(LocalLogImportJob job){			
			LocalLogImportJob element = null;
			int s = jobs.length;
			for(int j=0;j<s;j++){										
				if(jobs[j]==null){
					jobs[j] = job;
//					System.out.println("REGISTERED ["+j+"] "+job.getTaskName());
					break;
				}
			}			
		}
		
		public synchronized boolean hasRegistered(LocalLogImportJob job){
			LocalLogImportJob element = null;
			int s = jobs.length;
			for(int j=0;j<s;j++){										
				if(jobs[j]==job){
					return true;
				}
			}
			return false;
		}		
    }
    
//	abstract interface IJobStatusChangeListener {
//	
//	/**
//	 * Notifies this listener that the given status has changed.
//	 * 
//	 * @param	status	the new status
//	 */
//	public void statusChanged(IJobStatus status);
//	}
	
//    class JobElement{
//
//    	private Job job;
//    	private Parser parser;
//    	private Agent agent;
//    	private IProgressMonitor mon;
//    	private String label;
//    	private boolean status = false;
//    	
//    	JobElement(Job job, Parser parser, Agent agent, IProgressMonitor mon, String label){
//    		this.job = job;
//    		this.parser = parser;
//    		this.agent = agent;
//    		this.mon = mon;
//    		this.label = label;
//    	}
//    	
//		/**
//		 * @return Returns the job.
//		 */
//		public Job getJob() {
//			return job;
//		}
//		/**
//		 * @return Returns the monitor.
//		 */
//		public IProgressMonitor getMonitor() {
//			return mon;
//		}
//		/**
//		 * @return Returns the parser.
//		 */
//		public Parser getParser() {
//			return parser;
//		}
//
//		/**
//		 * @return Returns the parser.
//		 */
//		public Agent getAgent() {
//			return agent;
//		}
//		
//		/**
//		 * @return Returns the label.
//		 */
//		public String getLabel() {
//			return label;
//		}
//		
//		/**
//		 * @return Returns if the job has ended of not.
//		 */
//		public boolean getStatus() {
//			return status;
//		}
//		
//		public void setStatus(boolean newStatus){
//			status = newStatus;
//		}
//		
//    }
    

	/* (non-Javadoc)
	 * @see org.eclipse.core.runtime.jobs.Job#run(org.eclipse.core.runtime.IProgressMonitor)
	 */
	protected IStatus importLog(LogFileElement element) {
    	
    	boolean localHost = true;
    	Parser parser = null;
    	TRCProcessProxy process;

	    Node node;
	    PerfUtil p = new PerfUtil("LogImportWizard.run " + getUserInput(element,false).get("file_path"),true);			
		TRCAgent agent;
		bAppend = true; //assume append by default			
		//mon.subTask(getAgentName(element));	
		if(element.getMergedAgent()!=null){
			
			Display.getDefault().syncExec(new Runnable(){
			  public void run(){
			  	TraceMergeUIDialog dialog = new TraceMergeUIDialog(getShell(), LogUIPlugin.getResourceString("STR_EXISTING_DATA_DESC_2"));					
			  	int ret = dialog.open();
			  	if(ret==Window.CANCEL){
			  		return;
			  	}
			  	if(ret==2){
			  		bAppend = true;
			  	}
			  	if(ret==3){
			  		bAppend = false;
			  	}
			   
			  }
			});				
		}
		
		final String project = element.getProject();
		final String monitor = element.getMonitor();
		final String host = element.getHost();
	    
	    try {
	        node = NodeFactory.createNode(host);
	
	        Node lnode = NodeFactory.createNode("localhost");
	
	        localHost = (node.getName().equals(lnode.getName()));
	    } catch (Exception e) {
//	        errorMessage = MonitoringLogUIPlugin.getResourceString("IMPORT_LOG_FILE_HOST_ERROR_");
	        reasonAction = LogUIPlugin.getResourceString("IMPORT_LOG_FILE_HOST_REASON");
	        reasonAction = HyadesUtil.change(reasonAction, "%1", host);
	
	        Status result = new Status(IStatus.ERROR, LogUIPlugin.getPluginId(), 0, reasonAction, e);
	        return result;
	    }
	
	    
	
	    agent = element.getMergedAgent()!=null ? element.getMergedAgent().getAgent():null;
	    	        
	    if (agent == null) { //user didn't select any agent to merge.
	
			IContainer container = PDCoreUtil.createContainer(new Path(project));	
			TRCMonitor trcMonitor = PDCoreUtil.createMonitor(container, monitor);
			final IContainer fContainer = container; 
			refreshView(fContainer);
				
	        String portNb = UIPlugin.getDefault().getPreferenceStore().getString(HyadesConstants.LOCALHOST_PORT);
	        TRCNode trcNode = PDCoreUtil.createNode(trcMonitor, node.getName(), portNb);
			final TRCNode fNode = trcNode;
			
			final TRCMonitor fMonitor = trcMonitor;	
			refreshView(fMonitor);
	
	        process = createProcess(container, trcNode, element);
	
			refreshView(fNode);
				
			try {
				agent = createAgent(process, element,null);
				displayAgent = agent;
			} catch (Exception e) {
				errorMessage = LogUIPlugin.getResourceString("AgentCreationError");
				LogUIPlugin.log(e);
				return new Status(IStatus.ERROR, LogUIPlugin.getPluginId(), 0,
						errorMessage, e);
			}
			if(errorMessage!=null && errorMessage.startsWith(CANCEL)){				
				return new Status(IStatus.CANCEL, LogUIPlugin.getPluginId(), 0,
						"", null);
			}
			
			final TRCProcessProxy fProcess = process;
			
			refreshView(fProcess);
	            
	
	    } else { //user selected an agent to merge
	        process = agent.getAgentProxy().getProcessProxy();
	
	        if (!bAppend) {
	        	//replace: create an new agent and remove the existing one for replace
	        	agent = createAgent(process, element,element.getMergedAgent());
	            //_agent.getDefaultEvents().clear();
	        }
	    }
	
	    final TRCAgent trcAgent = agent;
	    final TRCProcessProxy trcProcess = process;
		trcAgent.getAgentProxy().setActive(true);
//		System.out.println("fViewer="+fViewer);
		if(fViewer!=null)	
			fViewer.getViewer().refresh(trcAgent.getAgentProxy());
	    try{
	        if (localHost) {		 
	        	importFromLocalHost(element, trcAgent);
	            //mon.done();
	        } else {
	        	importFromRemoteHost(element, trcProcess, trcAgent);
	        }
	    }catch(OperationCanceledException e){
	    	return new Status(IStatus.CANCEL,LogUIPlugin.getPluginId(), 0, "", e);
	    }catch(Exception e){	           
	    	e.printStackTrace();
	    	IStatus result = null;
	    	if(errorMessage==null){
				errorMessage = LogUIPlugin
				.getResourceString("IMPORT_LOG_FILE_RAC_ERROR_");		
	    	}
	    	if(reasonAction!=null){	    		
		        if(detailMsg!=null){
		        	result = new MultiStatus(LogUIPlugin.getPluginId(), 0, reasonAction, e);
		        	((MultiStatus)result).add(new Status(IStatus.ERROR, LogUIPlugin.getPluginId(), 0, detailMsg, e));
		        }else{
		        	result = new Status(IStatus.ERROR, LogUIPlugin.getPluginId(), 0, reasonAction, e);
		        }
	    		
	    	}else if(detailMsg!=null){
	    		result = new Status(IStatus.ERROR, LogUIPlugin.getPluginId(), 0, detailMsg, e);
	    	}else if(e.getLocalizedMessage()!=null){
	    		result = new Status(IStatus.ERROR, LogUIPlugin.getPluginId(), 0, e.getLocalizedMessage(), e);
	    	}else{
	    		result = new Status(IStatus.ERROR, LogUIPlugin.getPluginId(), 0, e.toString(), e);
	    	}
			trcAgent.getAgentProxy().setActive(false);
			if(fViewer!=null)	
				fViewer.getViewer().refresh(trcAgent.getAgentProxy());
	        return result;	        	
	    }
		//mon.done();
	    p.stopAndPrintStatus();	
		return new Status(IStatus.OK, LogUIPlugin.getPluginId(), 0, "", null);
	}
	
	protected void importFromLocalHost(LogFileElement element, TRCAgent trcAgent) throws Exception{
			XMLLoader xmlLoader = new XMLLoader(trcAgent);
			final Hashtable input = getUserInput(element, true);
			Parser parser = element.getParser().getParserInstance();			
//			FilterPackageImpl.init();
			if(element.getFilter()!=null && element.getFilter().getQuery()!=null){
				ResourceSet resourceSet = new ResourceSetImpl();
				resourceSet.getResourceFactoryRegistry().getExtensionToFactoryMap().put("adapter", new AdapterResourceFactoryImpl());
				String nameSpace = "org.eclipse.hyades.logging.adapter.config";
				if(element.getParser()!=null && element.getParser().getNamespace()!=null && element.getParser().getNamespace().length()>0){
					nameSpace = element.getParser().getNamespace();
				}
				GLAFilterHelper.addFilterToAdapter(null, resourceSet, input, element.getFilter().getQuery(), nameSpace);
			}
			LocalLogParserLoader parserLoader = new LocalLogParserLoader(
					parser, input);
			parserLoader.setXMLLoader(xmlLoader);
			
			LocalLogImportJob job = new LocalLogImportJob(LogUIPlugin.getResourceString("IMPORT_LOG_FILE_PROGRESS_DESC"), element, parser, xmlLoader, parserLoader, trcAgent);
			job.addJobChangeListener(jobListener);
			jobPool.add(job);			
    }
    
    protected void importFromRemoteHost(LogFileElement element, final TRCProcessProxy trcProcess, final TRCAgent trcAgent) throws Exception{

    	Node node = PDCoreUtil.profileConnect(element.getHost(), String.valueOf(Constants.CTL_PORT_NUM_SERVER));

        if (node == null) {
            //mon.done();
            return;
        }

        //mon.done();
        Hashtable userInput = getUserInput(element, true);
		if(element.getFilter()!=null && element.getFilter().getQuery()!=null){
			ResourceSet resourceSet = new ResourceSetImpl();
			resourceSet.getResourceFactoryRegistry().getExtensionToFactoryMap().put("adapter", new AdapterResourceFactoryImpl());
			String nameSpace = "org.eclipse.hyades.logging.adapter.config";
			if(element.getParser()!=null && element.getParser().getNamespace()!=null && element.getParser().getNamespace().length()>0){
				nameSpace = element.getParser().getNamespace();
			}			
			String tempAdapterPath = GLAFilterHelper.addFilterToAdapter(node, resourceSet, userInput, element.getFilter().getQuery(), nameSpace);
			element.setTempAdapterPath(tempAdapterPath);
		}

        Process process = ProcessFactory.createProcess(node, ParserConstants.REMOTE_LOG_LOADER_RAC_APPLICATION, getParserCommand(element, userInput));
        process.addProcessListener(new ProcessListenerImpl(trcProcess, trcAgent.getAgentProxy()));

        final Agent agent = AgentFactory.createAgent(process, ParserConstants.REMOTE_LOG_LOADER_AGENT_NAME, HyadesConstants.LOG_AGENT_TYPE);
		final TRCAgentProxy aproxy = trcAgent.getAgentProxy();

        agent.setAutoAttach(true);
        
        final Agent errorAgent = AgentFactory.createAgent(process, ParserConstants.REMOTE_LOG_LOADER_ERROR_AGENT_NAME, HyadesConstants.LOG_AGENT_TYPE);
        errorAgent.setAutoAttach(true);

        final Agent statusAgent = AgentFactory.createAgent(process, ParserConstants.REMOTE_LOG_LOADER_STATUS_AGENT_NAME, HyadesConstants.LOG_AGENT_TYPE);
        statusAgent.setAutoAttach(true);

        
        RemoteLogImportJob job = new RemoteLogImportJob(LogUIPlugin.getResourceString("IMPORT_LOG_FILE_PROGRESS_DESC"), element, node, process, errorAgent, trcAgent);
//        RemoteLogImportJob statusJob = new RemoteLogImportJob(LogUIPlugin.getResourceString("IMPORT_LOG_FILE_PROGRESS_DESC"), element, node, process, errorAgent, trcAgent);        
		agent.addAgentListener(new LogAgentListener(job, aproxy, trcProcess));
		
		ErrorAgentListener errorListener = new ErrorAgentListener();
		errorAgent.addAgentListener(errorListener);
        job.setErrorListener(errorListener);        
		statusAgent.addAgentListener(job);
		job.addJobChangeListener(jobListener);
		jobPool.add(job); 
    }
    
    private int getLocalImportsCount(){
    	List elements = getLogFileElements();
    	int count = 0;
    	LogImportJob job = null;
    	if(jobPool.size()==0){
    		return 0;
    	}
    	for(int i=0; i<jobPool.size(); i++){
			job = (LogImportJob)jobPool.get(i);
			if(job!=null && job instanceof LocalLogImportJob){
				++count;
			}
		}
    	return count;
    }
	

}