/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: XMLmethodExitLoader.java,v 1.16 2005/04/21 21:36:10 slavescu Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 * 
 * $Id: XMLmethodExitLoader.java,v 1.16 2005/04/21 21:36:10 slavescu Exp $
 **********************************************************************/
package org.eclipse.hyades.loaders.trace;

import java.util.Iterator;
import org.eclipse.hyades.loaders.trace.TraceUtils.InvocationInfo;
import org.eclipse.hyades.loaders.util.HierarchyContext;
import org.eclipse.hyades.loaders.util.LoadersUtils;
import org.eclipse.hyades.loaders.util.LookupServiceExtensions;
import org.eclipse.hyades.models.trace.TRCFullMethodInvocation;
import org.eclipse.hyades.models.trace.TRCFullTraceObject;
import org.eclipse.hyades.models.trace.TracePackage;


/**
 * @author slavescu
 *
 */
public class XMLmethodExitLoader extends TraceMethodBaseLoader {
    //~ Static fields/initializers -----------------------------------------------------------------

    //  protected static final String TICKET = "ticket";
    //	protected static final String STACK_DEPTH = "stackDepth";
    protected static final String OVERHEAD = "overhead";
    protected static final String CPU_TIME = "cpuTime";
    protected static final String CONSUMED_CPU_TIME = "consumedCpuTime";

    //~ Instance fields ----------------------------------------------------------------------------

    protected double overhead;
    protected double cpuTime;

    //~ Methods ------------------------------------------------------------------------------------

    public void addAttribute(String name, String value) {
        if (!loadToModel) {
            return;
        }

        switch (LoadersUtils.getHashCode(name)) {
        case TraceConstants.OVERHEAD_int:
            overhead = Double.parseDouble(value);

            break;
        case TraceConstants.CPU_TIME_int:
        case TraceConstants.CONSUMED_CPU_TIME_int:
            cpuTime = Double.parseDouble(value);

            break;

        default:
            super.addAttribute(name, value);

            break;
        }
    }

    public void addYourselfInContext() {
        if (!loadToModel) {
            return;
        }
		if(context.isFilterOn())
		{
			if(context.getFilterEngine().isFiltered(TracePackage.eINSTANCE.getTRCMethod_Name(),LoadersUtils.getLookUpKey(methodIdRef)))
			{
				return;
			}
		}

        cs = (CallStackPerThread) LookupServiceExtensions.getInstance().locate(context, CallStackPerThread.class, LoadersUtils.getLookUpKey(threadIdRef));

        if ((cs == null) || cs.isEmpty()) {
            return;
        }

        invocationPool = cs.invocationPool;

        theProcess = getProcess();

        dispatchProcessMode(ProcessSteps.ALL);
    }

    public void initialize(HierarchyContext context, String name) {
        loadToModel = context.isLoadToModel();

        if (!loadToModel) {
            return;
        }

        super.initialize(context, name);
        overhead = 0;
        cpuTime=0;
    }

    protected void processEF(int step) {
        super.processEF(step);

        fullInvocation = ((InvocationInfo) cs.peek()).getMethodInvocation();

        if (fullInvocation != null) {
            theThread = fullInvocation.getThread();
            fullInvocation.setExitTime(createDeltaTime());
            fullInvocation.setOverhead(overhead);
            addInputOutputValues();

            synchronized (REMOTE_INVOCATION_RESOLUTION_LOCK) {
    			updateForwardInvokes(fullInvocation);
    		}

			if(context.getCustomData().containsKey(USE_updateStatisticalInfoOld))
				updateStatisticalInfo();
			else
				updateStatisticalInfoNew();
            invocationPool.release((InvocationInfo) cs.pop());
        } else {
            // ignore method exit if there is no method entry
        }
    }

    protected void processES(int step) {
        super.processES(step);

        //		theThread = getThreadByIdRef(theProcess);
        //		theMethod = fullInvocation.getMethod();
        //		theObject = fullInvocation.getOwningObject();
        //		theClass = theMethod.getDefiningClass();
        updateStatisticalInfoOnly();
    }

    protected void updateStatisticalInfoNew() {
        try {
			double deltaBaseTime1 = 0;
			for (Iterator iter = fullInvocation.getInvokes().iterator(); iter.hasNext();) {
				TRCFullMethodInvocation element = (TRCFullMethodInvocation) iter.next();
				if(element.getExitTime()>0)
					deltaBaseTime1 = deltaBaseTime1+element.getExitTime()-element.getEntryTime()-element.getOverhead();
			}
			deltaCumulativeTime = fullInvocation.getExitTime() - fullInvocation.getEntryTime() - fullInvocation.getOverhead();
            deltaBaseTime = deltaCumulativeTime - deltaBaseTime1;

			invokerMethod = fullInvocation.getMethod();
			invokerObject = (TRCFullTraceObject) fullInvocation.getOwningObject();
			invokerClass = fullInvocation.getMethod().getDefiningClass();

			invokerObjectClass = getExtendedClass(invokerObject,invokerClass);
			
			updateCPUTime();
            //update base and cumulative using deltaBaseTime, invokerObject and invokerClass
			updateTimeStatisticsNew();
			
			if(fullInvocation.getInvokedBy()!=null && fullInvocation.eContainer() != fullInvocation.getInvokedBy().eContainer())
				updateBaseTimeIfRequired(fullInvocation,fullInvocation.getInvokedBy());
            //			p.setCurrentTime(createTimeDouble());
        } catch (Exception e) {
			LoadersUtils.log(e);
        }
    }

    protected void updateStatisticalInfo() {
        try {
            int size = fullInvocation.getInvokes().size();

            if (size > 0) {
                //this "invocation" has children
                lastChildExitTime = ((TRCFullMethodInvocation) fullInvocation.getInvokes().get(size - 1)).getExitTime();
                deltaBaseTime = fullInvocation.getExitTime() - lastChildExitTime - fullInvocation.getOverhead();
            } else {
                //this "invocation" doesn't have children
                deltaBaseTime = fullInvocation.getExitTime() - fullInvocation.getEntryTime() - fullInvocation.getOverhead();
            }

			invokerMethod = fullInvocation.getMethod();
			invokerObject = (TRCFullTraceObject) fullInvocation.getOwningObject();
			invokerClass = fullInvocation.getMethod().getDefiningClass();

			invokerObjectClass = getExtendedClass(invokerObject,invokerClass);
			
			updateCPUTime();
            //update base and cumulative using deltaBaseTime, invokerObject and invokerClass
            updateTimeStatistics();

            //			p.setCurrentTime(createTimeDouble());
        } catch (Exception e) {
			LoadersUtils.log(e);
        }
    }

    /**
	 * 
	 */
	protected void updateCPUTime() {
		if(cpuTime!=0)
		{
			try {
				double correctedCPUTime = cpuTime;//-overhead;
				if(fullInvocation!=null)
					fullInvocation.setCpuTime(correctedCPUTime);
				invokerMethod.setTotalCpuTime(invokerMethod.getTotalCpuTime()+correctedCPUTime);
				invokerClass.setTotalCpuTime(invokerClass.getTotalCpuTime()+correctedCPUTime);
				invokerClass.getPackage().setTotalCpuTime(invokerClass.getPackage().getTotalCpuTime()+correctedCPUTime);
				getProcess().setTotalCpuTime(getProcess().getTotalCpuTime()+correctedCPUTime);
				
			} catch (Exception e) {
				LoadersUtils.log(e);
			}
		}
	}

	protected void updateStatisticalInfoOnly() {
        try {
            InvocationInfo currentInvocation = (InvocationInfo) cs.peek();
            double exitTime = createDeltaTime();

            lastChildExitTime = currentInvocation.getLastChildExitTime();

            if (lastChildExitTime > 0) {
                deltaBaseTime = exitTime - lastChildExitTime - overhead;
            } else {
                deltaBaseTime = exitTime - currentInvocation.getEntryTime() - overhead;
            }

            invokerObject = currentInvocation.getObject();

            invokerClass = currentInvocation.getTheClass();

            invokerObjectClass = currentInvocation.getObjectClass();

            invokerMethod = currentInvocation.getMethod();
            
            updateCPUTime();
            //update base and cumulative time using deltaBaseTime, invokerObject and invokerClass
            updateTimeStatistics();

            //remove current invocation
            invocationPool.release((InvocationInfo) cs.pop());

            if (!cs.isEmpty()) {
                //update current invocation parent 
                InvocationInfo invoker = (InvocationInfo) cs.peek();

                if (invoker != null) {
                    invoker.setLastChildExitTime(exitTime);
                }
            }

            theProcess.setLastEventTime(exitTime);
        } catch (Exception e) {
			LoadersUtils.log(e);
        }
    }
}
