package org.eclipse.hyades.ui.sample.svg.generator;
/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DOMDataRetriever.java,v 1.4 2005/02/16 22:24:05 qiyanli Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

import java.io.Serializable;
import java.util.Hashtable;
import java.util.Vector;

import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.NodeList;

/**
 * Convenience class for accessing DOM elements and attributes. 
 * Methods return the data or null/empty result objects if not 
 * found. Not intended for public use.
 * 
 * @version 1.23.2.2
 */
public class DOMDataRetriever implements Serializable {
		private final String EMPTY_STRING = "";
		
		DOMDataRetriever() {
			super();
		}	
		
		/**
		 * Retrieve the specified attribute values of all the specified elements.
		 * 
		 * @param document the document
		 * @param tagName the element whose attribute value are being collected
		 * @param theAttribute whose vaule is being collected
		 * @return the list of value a vector of the value of the attribute "theAttirbute" of all the element with tag name "tagName" in the document.
		 */
		Vector getAttributeFromElements(Document document, String tagName, String theAttribute) {
			NodeList list = document.getElementsByTagName(tagName);
			int length = list.getLength();
						
			if (!(length > 0)) {
				return null;
			}	
			
			Element element;
			Vector result = new Vector();
						
			for (int i = 0; i < length; i++) {
				element = (Element) list.item(i);
				String attribute = (element.getAttribute(theAttribute)).trim();
				if (!attribute.equals(EMPTY_STRING)) {				
					result.add(attribute);				
				}
			}								
			if (result.isEmpty()) {
				return null;
			}	
			
			result.trimToSize();	
			return result;
		}	
		
		/**
		 * Get the specified attribute of the first specified element.
		 * 
		 * @param document the document
		 * @param tagName the element whose attribute value are being collected
		 * @param theAttribute whose vaule is being collected
		 * @return the value  of the attribute "theAttirbute" of the first element with tag name "tagName" in the document.
		 */
		String getAttributeFromSingleElement(Document document, String tagName, String theAttribute) {
			String result = null;
			NodeList list = document.getElementsByTagName(tagName);
		
			if (list.getLength() == 1) {
				String attribute = (((Element)(list.item(0))).getAttribute(theAttribute)).trim();			
				if (!attribute.equals(EMPTY_STRING)) {					
					result = attribute;
				}	
			} 		
			return result;		
		}	
		
		/**
		 * Count the number of specified elements.
		 * 
		 * @param document the document
		 * @param tagName the tag name of those element to be counted
		 * @return the number of element in the document with tagName "tagName".
		 */
		int getNumberOfElements(Document document, String tagName) {
			NodeList list = document.getElementsByTagName(tagName);
			return (list.getLength());
		}	
			
		/**
		 * Parse the attribues of the first specified element.
		 * 
		 * @param document the document
		 * @param tagName the tag name of those element being counted
		 * @return a hashtable of the attributes and their value of the element "tagName" in the document.
		 */
		Hashtable getAttributesFromSingleElement(Document document, String tagName) {
			Hashtable result = null;
			NodeList list = document.getElementsByTagName(tagName);
			
			if (list.getLength() != 1) {
				return result;
			}	
			
			NamedNodeMap attrs = list.item(0).getAttributes();
			int number = attrs.getLength();
			if (number > 0) {
				result = new Hashtable(number);			
				for (int i = 0; i < number; i++) {
					String name = (((Attr) attrs.item(i)).getName()).trim();
					String value = (((Attr) attrs.item(i)).getValue()).trim();	
					// NullPointerException not possible														
					if (!name.equals(EMPTY_STRING) && !value.equals(EMPTY_STRING)) {
						result.put(name, value);						
					}
				}
				if (result.isEmpty()) {
					return null;
				}		
			}			
			return result;	
		}	
		
		/**
		 * Get a list of the specified attributes of all the specified sub-elements of the first specified element.
		 * 
		 * @param document the document
		 * @param tagName the tag name of the enclosing element 
		 * @param childElementName the tag name of the sub-element whose attribute value is to be collected
		 * @param theAttribute the name of the attribute
		 * @return a list of value of attribute "theAttribute" of all the sub-element with tag name "childElementName" of the first element with tag name "tagName" in the document.
		 */
		Vector getAttributeValuesFromSingleElementChildren(Document document, String tagName, String childElementName, String theAttribute) {
			Vector result = null;
			NodeList list = document.getElementsByTagName(tagName);
			
			
			if (list.getLength() > 0) {			
				Element element = (Element)(list.item(0));
				if (element.getTagName().equals(tagName)) {
					list = element.getElementsByTagName(childElementName);
					int length = list.getLength();			
					if (length > 0) {		
						result = new Vector();
						for (int i = 0; i < length; i++) {
							Element e = (Element)list.item(i);							
							String attribute = (e.getAttribute(theAttribute)).trim();							
							if (!attribute.equals(EMPTY_STRING)) {
								result.add(attribute);									
							}																					
						}
						if (result.isEmpty()) {
							return null;
						}				
						result.trimToSize();	
					}
				}
				
			}				
			return result;					
		}	


		/**
		 * Parse the key and value attributes of a given list of Elements.
		 * 
		 * @param parent the parent element
		 * @param childElementName the name of sub-elements containing the map
		 * @param keyAttribute name of the key attribute
		 * @param valueAttribute name of the value attribute
		 * @return a map in which the keys are the values of the attribute "keyAttribute" in each of the sub-element with
		 * tag name "childElementName" under the parent element. The corresponding values are the values of the attribute "valueAttribute" in the same sub-element.
		 */
		Hashtable getMapInChildElements(Element parent, String childElementName, String keyAttribute, String valueAttribute) {
		    Hashtable result = null;
		    if (parent != null) {
			    NodeList list = parent.getElementsByTagName(childElementName);
				int length = list.getLength();
				if (length > 0) {
					result = new Hashtable(length);												
					for (int i = 0; i < length; i++) {
						Element e = (Element)list.item(i);							
						// NullPointerException not possible
						String key = (e.getAttribute(keyAttribute)).trim();
						String value = (e.getAttribute(valueAttribute)).trim();
						if (!key.equals(EMPTY_STRING) && !value.equals(EMPTY_STRING)) {
							result.put(key, value);
						}								
					}
					if (result.isEmpty()) {
						return null;
					}								
				}
		    }
			return result;
		}


		/**
		 * Get the first of the specified element in the document.
		 * 
		 * @param document the document
		 * @param tagName the tag name of the enclosing element
		 * @return the first element with tag name "tagName" in the document.
		 */
		Element getFirstElement(Document document, String tagName) {
			Element result = null;
			// get the elements with the elementName we want
			NodeList list = document.getElementsByTagName(tagName);
			if (list.getLength() > 0) {			
				// take the first one
				result = (Element)(list.item(0));
			}
			return result;
		}


		/**
		 * Get all the specified sub-elements of the specified element in the document.
		 * 
		 * @param document the document
		 * @param tagName the tag name of the enclosing element
		 * @param childElementName the tag name of the child element
		 * @return a list of elements with tag name "childElementName" under the first element with tag name "tagName" in the document.
		 */
		NodeList getChildrenOfFirstElement(Document document, String tagName, String childElementName) {
		    NodeList result = null;

            Element theElement = getFirstElement(document, tagName);
            						
			// get all the child elements we are looking for			
			if (theElement != null) {
				result = theElement.getElementsByTagName(childElementName);
			}
			return result;
		}

		
		
		/**
		 * Parse the key and value attributes of the specified sub-elements of the specified element.
		 * 
		 * @param document the document
		 * @param tagName the tag name of the enclosing element
		 * @param childElementName the tag name of the child element
		 * @param childAttributeKey name of the key attribute
		 * @param childAttributeValue name of the value attribute
		 * @return a map in which the keys are the values of the attribute "childAttributeKey" in each of the
		 * elements with tag name "childElementName" under the first element with tag name "tagName" in the document.
		 * The corresponding values are the values of the attribute "childAttributeValue" in the same element.
		 */
		Hashtable getAttributeValuesFromChildrenOfFirstElement(Document document, String tagName, String childElementName, String childAttributeKey, String childAttributeValue) {
			Element parent = getFirstElement(document, tagName);
		    return getMapInChildElements(parent, childElementName, childAttributeKey, childAttributeValue);
		}

		
		/**
		 * Get all the n-th specified element in the document.
		 * 
		 * @param document the document
		 * @param tagName the tag name of the enclosing element
		 * @param ordinal the ordinal of the "tagName" element whose sub-elements with tag name "childElementName" are to be collected.
		 * @return a list of elements with tag name "childElementName" under the n-th (ordinal) element with tag name "tagName" in the document.
		 */
		Element getElementWithDocOrderOrdinal(Document document, String tagName, String ordinal) {
            Element result = null;
			int position = Integer.parseInt(ordinal);			
			// get the elements with the elementName we want
			NodeList list = document.getElementsByTagName(tagName);
			int parentLength = list.getLength();
			if (parentLength > position) {			
				// get the one with the doc order ordinal
				result = (Element)(list.item(position));
			}
			return result;
		}


/*
		/**
		 * Get all the specified sub-elements of the n-th specified element in the document.
		 * 
		 * @param document the document
		 * @param tagName the tag name of the enclosing element
		 * @param childElementName the tag name of the child elements to be collected.
		 * @param ordinal the ordinal of the "tagName" element whose sub-elements with tag name "childElementName" are to be collected.
		 * @return a list of elements with tag name "childElementName" under the n-th (ordinal) element with tag name "tagName" in the document.
		 * /
		NodeList getChildrenOfElementWithDocOrderOrdinal(Document document, String tagName, String childElementName, String ordinal) {
            NodeList list = null;
            Element theElement = getElementWithDocOrderOrdinal(document, tagName, ordinal);
				if (theElement != null) {
					list = theElement.getElementsByTagName(childElementName);
				}
			return list;
		}
*/

		/**
		 * Get all the specified sub-elements of the n-th specified element in the document.
		 * 
		 * @param document the document
		 * @param tagName the tag name of the enclosing element
		 * @param childElementName the tag name of the child elements to be collected.
		 * @param childAttributeKey name of the key attribute
		 * @param childAttributeValue name of the value attribute
		 * @param ordinal the ordinal of the "tagName" element whose sub-elements with tag name "childElementName" are to be collected.
		 * @return a map in which the keys are the values of the attribute "childAttributeKey" in each of the
		 * elements with tag name "childElementName" under the the n-th (ordinal) element with tag name "tagName" in the document.
		 * The corresponding values are the values of the attribute "childAttributeValue" in the same element.
		 */
		Hashtable getAttributeValuesFromChildrenOfElementWithDocOrderOrdinal(Document document, String tagName, String childElementName, String childAttributeKey, String childAttributeValue, String ordinal) {
			Element parent = getElementWithDocOrderOrdinal(document, tagName, ordinal);
		    return getMapInChildElements(parent, childElementName, childAttributeKey, childAttributeValue);
		}


		/**
		 * Get the specified element whose specific attribute has a given value.
		 * 
		 * @param document the document
		 * @param tagName the tag name of the enclosing element
		 * @param elementAttribute
		 * @param elementAttributeValue
		 * @return the element E with tag name "tagName" and the "elementAttribute" attribute of E has the 
		 * value "elementAttributeValue".
		 */
		Element getElementWithAttribute(Document document, String tagName, String elementAttribute, String elementAttributeValue) {
            Element result = null;
			
			// get the elements with the elementName we want
			NodeList list = document.getElementsByTagName(tagName);
			if (list.getLength() > 0) {				
				// get the element with the attribute value we are looking for
				for (int i = 0, length = list.getLength(); i< length; i++) {				
					String attr = ((Element)(list.item(i))).getAttribute(elementAttribute);
					if (attr.equals(elementAttributeValue)) {
						result = (Element)(list.item(i));
					}					
					else if (attr.equals(EMPTY_STRING) && length == 1) {
						result = (Element)(list.item(i));	
						break;
					}
				}				
			}
			return result;
		}


/*
		/**
		 * Get all the specified sub-elements of the specified element whose specific attribute has a given value.
		 * 
		 * @param document the document
		 * @param tagName the tag name of the enclosing element
		 * @param elementAttribute
		 * @param elementAttributeValue
		 * @param childElementName the tag name of the child elements to be collected.
		 * @return a list of elements with tag name "childElementName" under the element E with tag name "tagName".
		 * The "elementAttribute" attribute of E must also has the value "elementAttributeValue".
		 * /
		NodeList getChildrenFromElementWithAttribute(Document document, String tagName, String elementAttribute, String elementAttributeValue, String childElementName) {
            NodeList result = null;
			Element theElement = getElementWithAttribute(document, tagName, elementAttribute, elementAttributeValue);
			
			// get all the child elements we are looking for
			if (theElement != null) {
				result = theElement.getElementsByTagName(childElementName);
			}
			return result;
		}
*/		

	
		/**
		 * Get all the specified sub-elements of all the specified element whose specific attribute has a given value.
		 * 
		 * @param document the document
		 * @param tagName the tag name of the enclosing element
		 * @param elementAttribute
		 * @param elementAttributeValue
		 * @param childElementName the tag name of the child elements to be collected.
		 * @param childAttributeKey name of the key attribute
		 * @param childAttributeValue name of the value attribute
		 * @return a map in which the keys are the values of the attribute "childAttributeKey" in each of the
		 * elements with tag name "childElementName" under the element E with tag name "tagName".
		 * The "elementAttribute" attribute of E must also has the value "elementAttributeValue".
		 */
		Hashtable getChildAttributesFromElementWithAttribute(Document document, String tagName, String elementAttribute, String elementAttributeValue, String childElementName, String childAttributeKey, String childAttributeValue) {
		    Element parent = getElementWithAttribute(document, tagName, elementAttribute, elementAttributeValue);
		    return getMapInChildElements(parent, childElementName, childAttributeKey, childAttributeValue);
		}


		/**
		 * Get the value of the specified attribute of the first specified sub-element of the first specified element.
		 *
		 * @param document the document
		 * @param tagName the tag name of the enclosing element
		 * @param childElementName the tag name of the child element
		 * @param childElementAttribute  the attribute name of the child element
		 * @return the value of the attribute "childElementAttribute" of the first sub-element with tag name "childElementName" under the first element with tag name "tagName".
		 */
		String getAttributeValueFromFirstChildOfFirstElement(Document document, String tagName, String childElementName, String childAttributeName) {
			String result = null;			
			// get the elements with the elementName we want
			NodeList list = document.getElementsByTagName(tagName);
			if (list.getLength() > 0) {			
				// take the first one
				Element theElement = (Element)(list.item(0));
				// get all the child elements we are looking for			
				if (theElement != null) {
					list = theElement.getElementsByTagName(childElementName);	
					int length = list.getLength();
					if (length > 0) {					
						// take the first one
						Element childElement = (Element)(list.item(0));
						if (childElement != null) {
							String value = (childElement.getAttribute(childAttributeName)).trim();
							if (!value.equals(EMPTY_STRING)) {
								result = value;
							}	
						}
					}
				}			
			}					
			return result;				
		}
}
