package org.eclipse.hyades.ui.sample.svg.generator;
/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ECMAScriptGenerator.java,v 1.4 2005/02/16 22:24:05 qiyanli Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

import java.util.HashMap;
import java.util.Iterator;
/**
 * <code>ECMAScriptGenerator</code> generates the (ECMA 1.0-compliant) JavaScript functions
 * needed to support browser-based user interaction with <code>GraphicDocumentGenerators</code> 
 * that generate Scalable Vector Graphics (SVG).
 * 
 * @version 1.77.1.21
 * 
 */
public class ECMAScriptGenerator {
	private StringBuffer script;
//	private static final String eol = "\n";
	private static final String eol = "" ;
	
	/** 
	 * Generates browser information variables for use in browser specific code.
	 */
	public String generateClientSnifferVariables(){
		script = new StringBuffer("var browserVersion = parseInt(parent.navigator.appVersion); ");
		script.append("var isNN = parent.navigator.appName.indexOf(\"Netscape\")!= -1; ").append(eol);
		script.append("var isIE = parent.navigator.appName.indexOf(\"Microsoft\")!= -1; ").append(eol);
				
		return script.toString();
	}	
	
	/** 
	 * Generates null browser information variables for use when SVG is transcoded to other formats
	 */
	public String generateNullClientSnifferVariables(){
		script = new StringBuffer("var browserVersion = null;");
		script.append("var isNN = false;").append(eol);
		script.append("var isIE = false; ").append(eol);
				
		return script.toString();
	}	
	
	/** 
	 * Generates global variables for use in HTML functions.
	 */
	public String generateGlobalVariables(){		
		script = new StringBuffer("var currentValues = null; ");
		script.append("var palettes = null; ").append(eol);
		script.append("var preferencePage = null;").append(eol);
		return script.toString();
	}
	
	/** 
	 * Generates an event handler which performs initialization functions.
	 * Typically assigned as the "onload" event handler. 
	 * 
	 * @param palettes the available data set color palettes 
	 * @param type		the graphic type
	 */
	public String generateInitFunction(String userPreferences, HashMap palettes, String currentPalette, String currentFont, String cookieName, String type, int sets, int gridXOffset, int gridYOffset, int xAxisLength, int yAxisLength){
		script = new StringBuffer("function init(e){");
		
		// dynamically set the hover text box attributes (add 3 to each end for spacing)
		script.append("var svgDocument = e.target.ownerDocument;").append(eol);
		script.append("if (svgDocument != null) {").append(eol);
		script.append("	var t = svgDocument.getElementById(\"hoverBoxText\");").append(eol);
		script.append("	var bbox;").append(eol);
		script.append("	if (t != null) {").append(eol);
		script.append("   bbox = t.getBBox();").append(eol);
		script.append("	}").append(eol);
		script.append(" var hoverBoxElement = svgDocument.getElementById(\"hoverBox\");").append(eol);
		script.append(" if (hoverBoxElement != null && bbox != null) {").append(eol);
		script.append("   hoverBoxElement.setAttribute('x', bbox.x - 3);").append(eol);
		script.append("   hoverBoxElement.setAttribute('y', bbox.y - 3);").append(eol);
		script.append("   hoverBoxElement.setAttribute('width', bbox.width + 6);").append(eol);
		script.append("   hoverBoxElement.setAttribute('height', bbox.height + 6);").append(eol);
		script.append(" }").append(eol);
		script.append(" var hoverGroup = svgDocument.getElementById('preferencesHoverText');").append(eol);
		script.append(" if (hoverGroup != null) {").append(eol);
		script.append("  hoverGroup.setAttribute('visibility', 'hidden');").append(eol);
		script.append(" }").append(eol);
		
		
		script.append("  var rectElement = null;").append(eol);
		script.append("  for (var i = 0; i < ").append(sets).append("; i++) {").append(eol);
		script.append("    var number = 0; ").append(eol);
		script.append("    while ((rectElement = svgDocument.getElementById(\"textboxRect\" + i + number)) != null) {").append(eol);
		script.append("    var hoverBoxText = svgDocument.getElementById('textboxText' + i + number);").append(eol);
		script.append("    if (hoverBoxText != null) { ").append(eol);
		script.append("      hoverBoxText.setAttribute('visibility', 'visible');").append(eol);
		script.append("      bbox = hoverBoxText.getBBox();").append(eol);
		script.append("      var originalBoxY = rectElement.getAttribute('y');").append(eol);
		script.append("      var originalBoxX = rectElement.getAttribute('x');").append(eol);
	//	script.append("      if ( (parseFloat(originalBoxY) + bbox.height + 6) > ").append(gridYOffset + yAxisLength).append(") {").append(eol);
	//	script.append("        rectElement.setAttribute('y', (originalBoxY - bbox.height - 6 - 1));").append(eol);
	//	script.append("      }").append(eol);
		script.append("      rectElement.setAttribute('width', bbox.width + 6);").append(eol);
		script.append("      rectElement.setAttribute('height', bbox.height + 6);").append(eol);
		script.append("      rectElement.setAttribute('visibility', 'visible');").append(eol);
		script.append("      hoverBoxText.setAttribute('x', parseFloat(rectElement.getAttribute('x')) + (bbox.width + 6) / 2);").append(eol);
		script.append("      hoverBoxText.setAttribute('y', parseFloat(rectElement.getAttribute('y')) + bbox.height + 3);").append(eol);
		script.append("      hoverBoxText.setAttribute('visibility', 'visible');").append(eol);
		script.append("    }").append(eol);
		script.append("    number++;").append(eol);
		script.append("    }").append(eol);
		script.append("  }").append(eol);
		
		
		script.append("}").append(eol);   // end of if svgDocument != null

		
		if (palettes == null || palettes.isEmpty()) {
			script.append("} ").append(eol);
			return script.toString();
		}			
		
		// create the colour palettes array		
		script.append("palettes = new Array();").append(eol);
		
		
		Iterator keys = (palettes.keySet()).iterator();
		int i = 0;
		while (keys.hasNext()) {
			String name = (String)keys.next();
			String [] codes = (String [])palettes.get(name);
			
			script.append("palettes[");
			script.append(i);
			script.append("] = [");
			// add the colour codes
			for (int j = 0; j < codes.length; j++) {
				script.append("\"");
				script.append(codes[j]);
				script.append("\", ");
			}		
			
			// add the palette name	
			script.append("\"");
			script.append(name);
			script.append("\"");
			script.append("];").append(eol);
			i++;
		}
		
		// get any user preferences previously saved to a cookie
		if (cookieName != null) {
			script.append("currentValues = getPreferences();").append(eol);
		}			
	    script.append("if (currentValues != null) {").append(eol);
		script.append("updateStyles(currentValues, true);}").append(eol);
		
		// OR initialize the current values with the user preferences passed in
		if (userPreferences != null) {
			int numberOfPreferences = IGraphicDocumentStyle.NUMBER_OF_PREFERENCES;
			int lastCommaIndex = 0;
			int nextCommaIndex = 0;
			String userPreference = null;
			script.append("else {currentValues = [\"");
			// first preference
			nextCommaIndex = userPreferences.indexOf(",", lastCommaIndex);			
			userPreference = userPreferences.substring(lastCommaIndex,  nextCommaIndex);			
			script.append(userPreference);
			script.append("\",\"");			
			
			for (int k = 0; k < numberOfPreferences - 2; k++) {
				lastCommaIndex = nextCommaIndex;
				nextCommaIndex = userPreferences.indexOf(",", lastCommaIndex + 1);			
				userPreference = userPreferences.substring(lastCommaIndex + 1, nextCommaIndex);
				script.append(userPreference);
				script.append("\",\"");	
			}
			
			// last preference
			lastCommaIndex = nextCommaIndex;							
			userPreference = userPreferences.substring(lastCommaIndex + 1, userPreferences.length());
			script.append(userPreference);					
			script.append("\"];updateStyles(currentValues, true); }").append(eol);	
				
		}
		else {	
			// OR initialize the current values with the default settings		
			String defaultFont = IGraphicDocumentStyle.BASIC_FONTSTYLE;
			String newDefaultFont = defaultFont.substring(0, defaultFont.indexOf("font-family") + "font-family".length() + 1)
									+ currentFont 
									+ defaultFont.substring(defaultFont.indexOf("; fill"), defaultFont.length());
					
			script.append("else {currentValues = [\"");
			script.append(currentPalette);
			script.append("\",\"");
			script.append(IGraphicDocumentStyle.BASIC_FONTSIZE);
			script.append("\",\"");
			script.append(newDefaultFont);
			script.append("\",\"");
			script.append(IGraphicDocumentStyle.BASIC_SHOW_SHAPES);
			script.append("\",\"");
			// default for show values in pie chart is ON, all others OFF
			if (type == IGraphicTypeConstants.PIE_CHART) {
				script.append("1");
			}
			else {	
				script.append(IGraphicDocumentStyle.BASIC_SHOW_VALUES);
			}
			script.append("\",\"");
			script.append(IGraphicDocumentStyle.BASIC_SHOW_PERCENTS);
			script.append("\"];updateStyles(currentValues, true);}").append(eol);
			
		}
		script.append("} ").append(eol);
		return script.toString();
	}	
	
	/** 
	 * Generates a function which opens a new browser window on the content "url" + "src" 
	 * with the specified "name", "width" and "height". 
	 * 
	 * @param url		the URL to the folder which contains "src"
	 * @param src		the content file
	 * @param name		the name of the new window
	 * @param width	the width of the new window
	 * @param height	the height of the new window
	 */	
	public String generateOpenPreferencePageFunction(String url, String src, String name, String width, String height){
		script = new StringBuffer("function openPreferencePage(){");
		// needed for basic types preference pages if using NS 4.7
		if (src.indexOf(IUserPreferencesConstants.PREFS) == 0) {	
			script.append("if (isNN) {useNetscapePreferencePage();}").append(eol);
			script.append("else {").append(eol);
		}
		script.append("if (preferencePage) {preferencePage = null;}").append(eol);
		script.append("preferencePage=open('");
		script.append(url + src);
		script.append("','");
		script.append(name);
		script.append("', 'width=");
		script.append(width);
		script.append(",height=");
		script.append(height);
		script.append("'); ").append(eol);
		script.append("preferencePage.opener=window;} ").append(eol);
		script.append("} ").append(eol);
		// might want to move the preference page so it isn't blocking the widget, if possible
              
		return script.toString();
	}		
		
	/** 
	 * Generates a function which launches the NS 4.7 preference page.
	 * Extension classes should not use this for custom preference pages. 	
	 * 
	 * @param url		the URL to the folder which contains "src"
	 * @param src		the content file
	 * @param name		the name of the new window
	 */
	public String generateUseNetscapePreferencePage(String url, String src, String name, String pageWidth, String pageHeight){					
		String height = null;
		String ns_src = IUserPreferencesConstants.NS_PREFS;
		String width = IUserPreferencesConstants.NS_PREFS_WIDTH;
			
		if (name.equals(IGraphicTypeConstants.LINE_CHART)) {
			height = IUserPreferencesConstants.NS_LINE_PREFS_HEIGHT;
		} else if (name.equals(IGraphicTypeConstants.AREA_CHART)) {
			height = IUserPreferencesConstants.NS_AREA_PREFS_HEIGHT;
		} else if (name.equals(IGraphicTypeConstants.SCATTER_CHART)) {
			height = IUserPreferencesConstants.NS_SCATTER_PREFS_HEIGHT;			
		} else if (name.equals(IGraphicTypeConstants.BAR_CHART)) {
			height = IUserPreferencesConstants.NS_BAR_PREFS_HEIGHT;
		} else if (name.equals(IGraphicTypeConstants.STACKBAR_CHART)) {
			height = IUserPreferencesConstants.NS_STACKBAR_PREFS_HEIGHT;
		} else if (name.equals(IGraphicTypeConstants.GROUPBAR_CHART)) {
			height = IUserPreferencesConstants.NS_GROUPBAR_PREFS_HEIGHT;
		} else if (name.equals(IGraphicTypeConstants.PIE_CHART)) {
			height = IUserPreferencesConstants.NS_PIE_PREFS_HEIGHT;
		} else if (name.equals(IGraphicTypeConstants.METER)) {
			height = IUserPreferencesConstants.NS_METER_PREFS_HEIGHT;
		} else {
			width = pageWidth;
			height = pageHeight;
		}
		
		script = new StringBuffer("function useNetscapePreferencePage(){");
		script.append("if (preferencePage) {preferencePage = null;}").append(eol);
		script.append("preferencePage=open('");
		script.append(url + ns_src);
		script.append("','");
		script.append(name);
		script.append("','");
		script.append("width=");
		script.append(width);
		script.append(",");
		script.append("height=");
		script.append(height);
		script.append(",top=0,left=0,screenx=0,screeny=0,toolbar=no,scrollbar=no'); ").append(eol);
		script.append("preferencePage.prefType='");
		script.append(name);
		script.append("'; preferencePage.opener=window;");
		script.append("}").append(eol);
		
		return script.toString();
	}			
			
		
	/** 
	 * Generates a function which returns the set of available palettes. 	
	 */
	public String generateGetPalettesFunction(){
		script = new StringBuffer("function getPalettes(){").append(eol);
		script.append("return palettes;").append(eol);
		script.append("} ").append(eol);
		
		return script.toString();
	}			
		
	/** 
	 * Generates a function which returns an array of the current preference values			
	 */
	public String generateGetCurrentValuesFunction(){
		script = new StringBuffer("var current = new Array();").append(eol);
		// assemble the values
		script.append("function getCurrentValues() {").append(eol);
		script.append("for (var i = 0; i < 6; i++) {current[i] = currentValues[i];}").append(eol);
		// find the first colour of the current palette
		script.append("for (var i = 0; i < palettes.length; i++) {").append(eol);
		script.append("if (palettes[i][palettes[i].length - 1] == current[0]) {").append(eol);
		script.append(" current[6] = palettes[i][0];}}").append(eol);
		script.append("return current;").append(eol);
		script.append("} ").append(eol);
		
		return script.toString();
	}
		
	/** 
	 * Generates an event handler which toggles the visibility attribute of the group/element 
	 * with the specified "id". 
	 * 
	 * @param type the graphic type
	 */
	public String generateToggleVisibilityFunction(int sets, String type){
		script = new StringBuffer("var values=1, percents=1;var textboxState = new Array();");
		script.append("for (var i = 0; i < ").append(sets).append("; i++) {textboxState[i] = 0;}").append(eol);
		
		if (type == IGraphicTypeConstants.PIE_CHART) {
			// set all object visibility to the specified value
			script.append("function toggleVisibility(id) {").append(eol);
			script.append("  setVisibility('tickline', 'hidden');").append(eol);
			script.append("  if (isVisible('bothValues')) {").append(eol);
			script.append("    setVisibility('bothValues', 'hidden');").append(eol);
			script.append("  } else if (isVisible('percentValues')) {").append(eol);
			script.append("	   setVisibility('percentValues', 'hidden');").append(eol);
			script.append("  } else if (isVisible('exactValues')) {").append(eol);
			script.append("	   setVisibility('exactValues', 'hidden');").append(eol);
			script.append("  } else {").append(eol);
			script.append("    setVisibility('tickline', 'visible');").append(eol);
			script.append("    if ((values == 1 && percents == 1) || (values == 0 && percents == 0)) { ").append(eol);
			script.append("      setVisibility('bothValues', 'visible');").append(eol);
			script.append("    } else if (values == 1) {").append(eol);
			script.append("	     setVisibility('exactValues', 'visible');").append(eol);
			script.append("    } else if (percents == 1) {").append(eol);
			script.append("	     setVisibility('percentValues', 'visible');").append(eol);
			script.append("    }").append(eol);
			script.append("  }").append(eol);
			script.append("}").append(eol);
		}
		else if ((type == IGraphicTypeConstants.LINE_CHART) || (type == IGraphicTypeConstants.AREA_CHART) || (type == IGraphicTypeConstants.SCATTER_CHART)) {		
			script.append("var noShapes = true;").append(eol);
			script.append("function changeAllTextBoxes(state) {").append(eol);
			script.append("  for (var i = 0; i < ").append(sets).append("; i++) {");
			script.append("    var boxId = \"textbox\" + i;").append(eol);
			script.append("    if (document.getElementById(boxId) != null) {").append(eol);
			script.append("      if (isVisible('dataColour'+i)){").append(eol);
			//
			script.append("        setVisibility(boxId, state);").append(eol);
			script.append("        var number = 0;").append(eol);
			script.append("        while (svgDocument.getElementById('textboxText' + i + number) != null) {").append(eol);
			script.append("          setVisibility('textboxText' + i + number, state);").append(eol);
			script.append("          setVisibility('textboxRect' + i + number, state);").append(eol);
			script.append("          number++;").append(eol);
			script.append("        }").append(eol);
			//
			script.append("      }").append(eol);
			script.append("      if (state == 'hidden') {textboxState[i] = 0;}").append(eol);
			script.append("      else if (state == 'visible') {textboxState[i] = 1;}").append(eol);
			script.append("    }").append(eol);
			script.append("  }").append(eol);
			script.append("}").append(eol);
			script.append("function toggleVisibility(id) {").append(eol);
			script.append("  if (id.substring(0,10) == 'dataColour') {").append(eol);
			script.append("	   var lineId = id.substring(10,id.length);").append(eol);
			script.append("    if (isVisible(\"dataColour\" + lineId) ) {").append(eol);
			script.append(" 	 setVisibility(\"dataColour\" + lineId, 'hidden');").append(eol);
			script.append("		 setVisibility(\"textbox\" + lineId, 'hidden');").append(eol);
			// start
			script.append("      var number = 0;").append(eol);
			script.append("      while (svgDocument.getElementById('textboxText' + lineId + number) != null) {").append(eol);
			script.append("        setVisibility('textboxText' + lineId + number, 'hidden');").append(eol);
			script.append("        setVisibility('textboxRect' + lineId + number, 'hidden');").append(eol);
			script.append("        number++;").append(eol);
			script.append("      }").append(eol);
			// end
			script.append("	   } else if (!isVisible(\"dataColour\" + lineId)) {").append(eol);
			script.append("	     setVisibility(\"dataColour\" + lineId, 'visible');").append(eol);
			script.append("		 var state = textboxState[lineId] == 0 ? 'hidden' : 'visible';	").append(eol);
			script.append("		 setVisibility(\"textbox\" + lineId, state);").append(eol);
			//start
			script.append("      var number = 0;").append(eol);
			script.append("      while (svgDocument.getElementById('textboxText' + lineId + number) != null) {").append(eol);
			script.append("        setVisibility('textboxText' + lineId + number, state);").append(eol);
			script.append("        setVisibility('textboxRect' + lineId + number, state);").append(eol);
			script.append("        number++;").append(eol);
			script.append("      }").append(eol);
			// end
			
			script.append("	   }").append(eol);
			// turn on/off the shapes on a line
			script.append("    var group = document.getElementById(id);").append(eol);
			script.append("    if (group != null) {").append(eol);
			script.append("      var useElementGroup = group.getElementsByTagName('use');").append(eol);
			script.append("      if (useElementGroup != null) {").append(eol);
			script.append("        for (var j = 0; j < useElementGroup.length; j++) {").append(eol);
			if (type == IGraphicTypeConstants.SCATTER_CHART) {
				script.append("          useElementGroup.item(j).setAttribute('visibility', (isVisible(id) ? 'visible':'hidden') );").append(eol);
			}else{
				script.append("          useElementGroup.item(j).setAttribute('visibility', (isVisible(id) && noShapes == false ? 'visible':'hidden') );").append(eol);
			}
			script.append("        }").append(eol);
			script.append("      }").append(eol);
			script.append("    }").append(eol);
			script.append("  } else if (id.substring(0,7) == 'textbox') {").append(eol);
			script.append("    if (isVisible(\"dataColour\" + id.substring(7,id.length))) {").append(eol);
			script.append("      changeAllTextBoxes(isVisible(id) ? 'hidden':'visible');").append(eol);
			script.append("    }").append(eol);
			script.append("  }").append(eol);
			script.append("}").append(eol);
		}
		else if (type == IGraphicTypeConstants.STACKBAR_CHART) {					
			script.append("function toggleVisibility(id) {").append(eol);
			script.append("  setVisibility('exactValues', isVisible('exactValues')?'hidden':'visible');").append(eol);
			script.append("  for (var i = 0; i < ").append(sets).append("; i++) {").append(eol);  // turn on/off all boxes
			script.append("    var textBoxSet = document.getElementById(\"textbox\" + i);").append(eol);
			script.append("    var state = isVisible('exactValues')?'visible':'hidden';").append(eol);
			script.append("    if (textBoxSet != null) { ").append(eol);
			script.append("      textBoxSet.setAttribute('visibility', isVisible('exactValues')?'visible':'hidden');").append(eol);
			script.append("      var number = 0;").append(eol);
			script.append("      while (svgDocument.getElementById('textboxText' + i + number) != null) {").append(eol);
			script.append("        setVisibility('textboxText' + i + number, state);").append(eol);
			script.append("        setVisibility('textboxRect' + i + number, state);").append(eol);
			script.append("        number++;").append(eol);
			script.append("      }").append(eol);
			script.append("	   }").append(eol);
			script.append("	 }").append(eol);
			script.append("}").append(eol);
		}
		else {				
			script.append("function toggleVisibility(id) {").append(eol);
			script.append(" if(isVisible(id)){setVisibility(id,'hidden');}").append(eol);
			script.append(" else{setVisibility(id,'visible');}").append(eol);
			script.append("}").append(eol);
		}		
		
		// check if an object is visible
		script.append("function isVisible(id) {").append(eol);
		script.append(" if (document.getElementById(id).getAttribute(\"visibility\") == \"visible\") {return true;}").append(eol);
		script.append("} ").append(eol);
		
		// set a particular object visibility
		script.append("function setVisibility(id, value) {").append(eol);
		script.append(" document.getElementById(id).setAttribute(\"visibility\",value);").append(eol);
		script.append("} ").append(eol);
			
		return script.toString();
	}
		
	/**
	 * Generates helper functions used for XY-style charts.
	 */	
	public String generateXYChartHelperFunctions() {
		script = new StringBuffer("function resizeAndRepositionTextBox(textBox, newX, newY, newWidth, newHeight) {");
		script.append(" textBox.setAttribute('x', newX);").append(eol);
		script.append(" textBox.setAttribute('y', newY);").append(eol);
		script.append("	textBox.setAttribute('width', newWidth);").append(eol);
		script.append("	textBox.setAttribute('height', newHeight);").append(eol);
		script.append("} ").append(eol);
		
		script.append("function resizeTextBox(textBox, newWidth, newHeight) {").append(eol);
		script.append("	textBox.setAttribute('width', newWidth);").append(eol);
		script.append("	textBox.setAttribute('height', newHeight);").append(eol);
		script.append("} ").append(eol);
		
		script.append("function repositionText(text, newX, newY) {").append(eol);
		script.append("	text.setAttribute('x', newX);").append(eol);
		script.append("	text.setAttribute('y', newY);").append(eol);
		script.append("} ").append(eol);
				
		return script.toString();
	}
					
	/**
	 * Generates helper functions used for pie charts.
	 */				
	public String generatePieChartHelperFunctions() {
		script = new StringBuffer("");
		script.append("function changeTextColour(bothElement, percentTextElement, exactTextElement, palette) {").append(eol);
		script.append("  for (var i = 0, j = palette.length - 1; i < percentTextElement.length; i++, j = palette.length - 1) {").append(eol);
		script.append("    if (((i+1) % j) == 0) {j = palette.length - 2;} else {j=((i+1)%j)-1;}").append(eol);
		script.append("    bothElement.item(i).setAttribute('style', \"fill:\" + palette[j]);").append(eol);
		script.append("	   percentTextElement.item(i).setAttribute('style', \"fill:\" + palette[j]);").append(eol);
		script.append("	   exactTextElement.item(i).setAttribute('style', \"fill:\" + palette[j]);").append(eol);
		script.append("  }").append(eol);
		script.append("} ").append(eol);
			
		return script.toString();
	}	
	
	/** 
	 * Generates a function which resets the graphic user preference styles
	 * to their former values. 
	 */
	public String generateResetStylesFunction(){		
		script = new StringBuffer("function resetStyles(){");
		script.append("updateStyles(currentValues, true);").append(eol);
		script.append("} ").append(eol);
		
		return script.toString();
	}	
	
	/** 
	 * Generates a function which updates the graphic legend with user preference styles.
	 *  	
	 * @param sets 		the number of data sets displayed by the graphic legend
	 * @param type 		the graphic type
	 */
	public String generateUpdateLegendStylesFunction(int sets, String type){		
		script = new StringBuffer();
		script.append("function updateLegendStyles(newStyles, preview) {").append(eol);

		script.append("var fontSize=parseInt(newStyles[1]);").append(eol);
		// legend title
		script.append("var group = document.getElementById(\"legendTitle\");").append(eol);
		script.append("if (group != null){").append(eol);
		script.append(" group.setAttribute(\"style\",\"font-size:\" + (fontSize + 2) + \"pt; \" + newStyles[2]);}").append(eol);
		// legend data
		script.append("var group = document.getElementById(\"legendData\");").append(eol);
		script.append("if (group != null){").append(eol);
		script.append(" group.setAttribute(\"style\",\"font-size:\" + fontSize + \"pt; \" + newStyles[2]);}").append(eol);
		// only line chart has shapes		
		if ((type == IGraphicTypeConstants.LINE_CHART) || (type == IGraphicTypeConstants.AREA_CHART) || (type == IGraphicTypeConstants.SCATTER_CHART)) {		
			script.append("for (var i = 0; i < " + sets + "; i++){").append(eol);
			script.append("  var squareshape = document.getElementById('legendsquareshapeWithoutLine' + i);").append(eol);
			script.append("  var shape = document.getElementById('shape' + i);").append(eol);
			script.append("  shape.setAttribute('visibility', (newStyles[3] == '0' ? 'hidden' : 'visible'));").append(eol);
			script.append("  squareshape.setAttribute('visibility', (newStyles[3] == '0' ? 'visible' : 'hidden'));").append(eol);
			script.append("}").append(eol);
		}
		script.append("}").append(eol);
		return script.toString();
		
	}	
	
	/** 
	 * Generates a function which updates the graphic with user preference styles.
	 *  	
	 * @param sets 		the number of data sets displayed by the graphic
	 * @param type 		the graphic type
	 */
	public String generateUpdateStylesFunction(int sets, String cookieName, String type, boolean hasLegend, String updatedUserPrefsVarName, int gridXOffset, int gridYOffset, int xAxisLength, int yAxisLength){		
		script = new StringBuffer();
		script.append("function getTextElement(myGroup) { return myGroup.getElementsByTagName('text'); }").append(eol);
		// newStyles is an array and looks like: {palette name, font size, font text, shape, showValues, showPercents} 
		script.append("function updateStyles(newStyles, preview) {").append(eol);
		script.append("if (newStyles == null || newStyles.length < 6) {return;}").append(eol);
		// get the chosen colour palette
		script.append("var paletteName=newStyles[0];").append(eol);
		// use the first one if we get messed up here
		script.append("var palette= palettes[0];").append(eol);
		script.append("for (var i = 0; i < palettes.length; i++) {").append(eol);
		script.append("if (palettes[i][palettes[i].length-1] == paletteName) {").append(eol);
		script.append(" palette = palettes[i];}}").append(eol);
		// update the color groups with the new palette, wrapping if not enough colours		
		script.append("for (var i=0, j=palette.length-1; i<" + sets + ";i++, j=palette.length-1) {").append(eol);
		script.append("  if (((i+1) % j) == 0) {j = palette.length - 2;} else {j=((i+1)%j)-1;}").append(eol);
		script.append("  var group = document.getElementById(\"dataColour\" + i);").append(eol);
		script.append("  if (group != null){").append(eol);
		script.append("    group.setAttribute(\"style\",\"stroke:\" + palette[j] + \"; fill:\" + palette[j]);}").append(eol);
		script.append("  group = document.getElementById(\"legendsquareshapeWithoutLine\" + i);").append(eol);
		script.append("  if (group != null){").append(eol);
		script.append("    group.setAttribute(\"style\",\"fill:\" + palette[j]);}").append(eol);
		
		if ((type == IGraphicTypeConstants.LINE_CHART) || (type == IGraphicTypeConstants.AREA_CHART) || (type == IGraphicTypeConstants.SCATTER_CHART)) {		
			script.append("  group = document.getElementById(\"shape\" + i);").append(eol);
			script.append("  if (group != null){").append(eol);
			script.append("    group.setAttribute(\"style\",\"fill:\" + palette[j]);}").append(eol);
		}
		
		script.append("  group = document.getElementById(\"pieColour\" + i);").append(eol);
		script.append("  if (group != null){").append(eol);
		script.append("    group.setAttribute(\"style\", \"stroke:\" + \"black\" + \"; fill: \" + palette[j]);}").append(eol);
		script.append("  group = document.getElementById(\"percentValues\" + i);").append(eol);
		script.append("  if (group != null){").append(eol);
		script.append("    group.setAttribute(\"style\", \"stroke:\" + \"black\" + \"; fill: \" + palette[j]);}").append(eol);
		script.append("  group = document.getElementById(\"bothValues\" + i);").append(eol);
		script.append("  if (group != null){").append(eol);
		script.append("    group.setAttribute(\"style\", \"stroke:\" + palette[j] + \"; fill: \" + palette[j]);}").append(eol);
		script.append("  group = document.getElementById(\"line\" + i);").append(eol);
		script.append("  if (group != null){").append(eol);
		script.append("    group.setAttribute(\"style\", \"fill: \" + palette[j]);}").append(eol);
		script.append("}").append(eol);
				
		// update the font group with the new font attributes, e.g., "size:11pt; weight:normal;
		// family:Arial, sans-serif; stroke:#000000; "
		
		script.append("var fontSize=parseInt(newStyles[1]);").append(eol);
		// chart title
		script.append("var group = document.getElementById(\"graphTitle\");").append(eol);
		script.append("var graphTitleStyle = 'font-weight:bold; ' + newStyles[2].substring(newStyles[2].indexOf('font-style'), newStyles[2].length);").append(eol);
		script.append("if (group != null){").append(eol);
		script.append(" group.setAttribute(\"style\",\"font-size:\" + (fontSize+3) + \"pt; \" + graphTitleStyle);}").append(eol);
		// timestamp
		script.append("var group = document.getElementById(\"timeStamp\");").append(eol);
		script.append("if (group != null){").append(eol);
		script.append(" group.setAttribute(\"style\",\"font-size:\" + fontSize + \"pt; \");").append(eol);
		script.append(" group.setAttribute(\"y\", fontSize + 5);}").append(eol);
				
		if (type != IGraphicTypeConstants.METER) {
			// legend specific
			if (hasLegend && type != IGraphicTypeConstants.BAR_CHART) {
				script.append("updateLegendStyles(newStyles, preview);").append(eol);
			}
			
			if (type != IGraphicTypeConstants.PIE_CHART) {
				// x-axis data
				script.append("var group = document.getElementById(\"dataValues0\");").append(eol);
				script.append("if (group != null){").append(eol);
				script.append(" group.setAttribute(\"style\",\"font-size:\" + fontSize + \"pt; \" + newStyles[2]);}").append(eol);
				// y-axis data
				script.append("var group = document.getElementById(\"dataValues1\");").append(eol);
				script.append("if (group != null){").append(eol);
				script.append(" group.setAttribute(\"style\",\"font-size:\" + fontSize + \"pt; \" + newStyles[2]);}").append(eol);
					
				// resize textboxes and text inside textboxes
				script.append("for (var i = 0; i < ").append(sets).append("; i++) {").append(eol);
				script.append("var number = 0; var rectElement = null; var textElement = null;").append(eol);
				script.append("while ((rectElement = document.getElementById(\"textboxRect\" + i + number)) != null) {").append(eol);
				script.append("  textElement = document.getElementById(\"textboxText\" + i + number);").append(eol);
				//
				script.append("    if (textElement != null) {").append(eol);
				script.append("      textElement.setAttribute('style', 'font-size:' + fontSize + 'pt');").append(eol);
				script.append("      textElement.setAttribute('visibility', 'visible');").append(eol);
				script.append("      rectElement.setAttribute('visibility', 'visible');").append(eol);
				script.append("      var bbox = textElement.getBBox(); ").append(eol);
				script.append("      var originalBoxY = rectElement.getAttribute('y');").append(eol);
				script.append("      var originalBoxX = rectElement.getAttribute('x');");
				script.append("      if ( (parseFloat(originalBoxY) + bbox.height + 6) > ").append(gridYOffset + yAxisLength).append(") {");
				script.append("        rectElement.setAttribute('y', (originalBoxY - bbox.height - 6));").append(eol);
				script.append("      }").append(eol);
				script.append("      rectElement.setAttribute('width', bbox.width + 6);").append(eol);
				script.append("      rectElement.setAttribute('height', bbox.height + 6);").append(eol);
				script.append("      textElement.setAttribute('x', parseFloat(rectElement.getAttribute('x')) + (bbox.width + 6) / 2.0);").append(eol);
				script.append("      textElement.setAttribute('y', parseFloat(rectElement.getAttribute('y')) + bbox.height + 3);").append(eol);
				script.append("      if (newStyles[4] == '0' || !isVisible('dataColour' + i)) {").append(eol);
				script.append("        textElement.setAttribute('visibility', 'hidden');").append(eol);
				script.append("        rectElement.setAttribute('visibility', 'hidden');").append(eol);
				script.append("      }").append(eol);
				script.append("    }").append(eol);
				script.append("    ++number;").append(eol);
				//			
				script.append("  }").append(eol);  // end: for 
				script.append("}").append(eol);  // end for 
			}			
			else {	
				// percentages
				script.append("var group = document.getElementById(\"percentValues\");").append(eol);
				script.append("if (group != null){").append(eol);
				script.append(" group.setAttribute(\"style\",\"font-size:\" + fontSize + \"pt; \");}").append(eol);
				
				// both values 
				script.append("var bothGroup = document.getElementById(\"bothValues\");").append(eol);
				script.append("if (bothGroup != null){").append(eol);
				script.append(" bothGroup.setAttribute(\"style\",\"font-size:\" + fontSize + \"pt; \");}").append(eol);
			}
			
			// x-axis label
			script.append("var group = document.getElementById(\"labelsValues0\");").append(eol);
			script.append("if (group != null){").append(eol);
			script.append(" group.setAttribute(\"style\",\"text-anchor:middle; font-size:\" + (fontSize+2) + \"pt; \" + newStyles[2]);}").append(eol);
			// y-axis label
			script.append("var group = document.getElementById(\"labelsValues1\");").append(eol);
			script.append("if (group != null){").append(eol);
			script.append(" group.setAttribute(\"style\",\"font-size:\" + (fontSize+2) + \"pt; \" + newStyles[2]);}").append(eol);
			// exact values
			script.append("var group = document.getElementById(\"exactValues\");").append(eol);
			script.append("if (group != null){").append(eol);
			script.append(" group.setAttribute(\"style\",\"font-size:\" + fontSize + \"pt; \");}").append(eol);
			
		}		
		else {
			// minmaxRate
			script.append("var group = document.getElementById(\"minmaxRate\");").append(eol);
			script.append("if (group != null){").append(eol);
			script.append(" group.setAttribute(\"style\",\"font-size:\" + (fontSize) + \"pt; \");}").append(eol);
			// rateDefinition	
			script.append("var group = document.getElementById(\"rateDefinition\");").append(eol);
			script.append("if (group != null){").append(eol);
			script.append(" group.setAttribute(\"style\",\"font-size:\" + (fontSize) + \"pt; \");}").append(eol);
			// critDiv	
			script.append("var group = document.getElementById(\"critDiv\");").append(eol);
			script.append("if (group != null){").append(eol);
			script.append(" group.setAttribute(\"style\",\"font-size:\" + (fontSize) + \"pt; \");}").append(eol);
			// meter label
			script.append("var group = document.getElementById(\"meterLabel\");").append(eol);
			script.append("if (group != null){ ").append(eol);
			script.append("   var tmpText = group.getElementsByTagName('text');").append(eol);
			script.append("   for (var i = 0; i < tmpText.length; i++) {").append(eol);
			script.append("     tmpText.item(i).setAttribute('style', 'font-size: ' + (fontSize) + 'pt; ');").append(eol);
			script.append("   }").append(eol);
			script.append("}").append(eol);
		}	
		
		// Show shapes
		if ((type == IGraphicTypeConstants.LINE_CHART) || (type == IGraphicTypeConstants.AREA_CHART) || (type == IGraphicTypeConstants.SCATTER_CHART))  {
			script.append("for (var i = 0; i < " + sets + "; i++){").append(eol);
			script.append("  var group = document.getElementById('dataColour' + i);").append(eol);
			script.append("  if (group != null) {").append(eol);
			script.append("    var useElementGroup = group.getElementsByTagName('use');").append(eol);
			script.append("    if (useElementGroup != null) {").append(eol);
			script.append("      for (var j = 0; j < useElementGroup.length; j++) {").append(eol);
			if (type == IGraphicTypeConstants.SCATTER_CHART) {
				script.append("        useElementGroup.item(j).setAttribute('visibility', (newStyles[3] == '1' && isVisible('dataColour'+i) ? 'visible' : 'visible') );").append(eol);
			}else{
				script.append("        useElementGroup.item(j).setAttribute('visibility', (newStyles[3] == '1' && isVisible('dataColour'+i) ? 'visible' : 'hidden') );").append(eol);
			}
			script.append("      }").append(eol);
			script.append("    }").append(eol);
			script.append("  }").append(eol);
			script.append("  noShapes = newStyles[3] == '0' ? true : false;").append(eol);
			script.append("}").append(eol);
		}
		
		// show/hide values 
		if (type == IGraphicTypeConstants.PIE_CHART) {	
			script.append("var bothGroup=document.getElementById('bothValues');var perGroup=document.getElementById('percentValues');var eGroup=document.getElementById('exactValues');").append(eol);
			script.append("changeTextColour(getTextElement(bothGroup), getTextElement(perGroup), getTextElement(eGroup), palette);").append(eol);
			script.append("values = (newStyles[4] == '0' ? 0 : 1);").append(eol);
			script.append("percents = (newStyles[5] == '0' ? 0 : 1);").append(eol);
			script.append("setVisibility('tickline', (values==0 && percents==0 ? 'hidden':'visible'));").append(eol);
			script.append("eGroup.setAttribute('visibility', (values==1 && percents==0 ? 'visible':'hidden'));").append(eol);
			script.append("perGroup.setAttribute('visibility', (values==0 && percents==1 ? 'visible':'hidden'));").append(eol);
			script.append("bothGroup.setAttribute('visibility', (values==1 && percents==1 ? 'visible':'hidden'));").append(eol);
		}
		else if ((type == IGraphicTypeConstants.LINE_CHART) || (type == IGraphicTypeConstants.AREA_CHART) || (type == IGraphicTypeConstants.SCATTER_CHART)) {										
			script.append("for (var i = 0; i < ").append(sets).append("; i++) {").append(eol);
			script.append("  if (document.getElementById(\"textbox\" + i) != null) {").append(eol);
			script.append("    textboxState[i] = parseInt(newStyles[4]);").append(eol);
			script.append("    if (isVisible('dataColour'+i)) {").append(eol);
			script.append("      setVisibility(\"textbox\"+i, newStyles[4]=='0' ? 'hidden':'visible');").append(eol);
			script.append("      var number = 0; ").append(eol);
			script.append("      while ((hoverBoxElement = svgDocument.getElementById(\"textboxRect\" + i + number)) != null) {").append(eol);
			script.append("        var hoverBoxText = svgDocument.getElementById('textboxText' + i + number);").append(eol);
			script.append("        if (hoverBoxText != null) {").append(eol);
			script.append("          hoverBoxText.setAttribute('visibility', (parseInt(newStyles[4]) == 0 ? 'hidden':'visible') );").append(eol);
			script.append("          hoverBoxElement.setAttribute('visibility', (parseInt(newStyles[4]) == 0 ? 'hidden':'visible') );").append(eol);
			script.append("        }").append(eol);
			script.append("        number++;").append(eol);
			script.append("      }").append(eol);
			script.append("    }").append(eol);
			script.append("	 }").append(eol);
			script.append("}").append(eol);
		}
		else if (type == IGraphicTypeConstants.STACKBAR_CHART) {		
			script.append("var showAllState = newStyles[4]=='0' ? 'hidden':'visible';").append(eol);
			script.append("var eGroup = document.getElementById('exactValues');").append(eol);
			script.append("if (eGroup!=null) {eGroup.setAttribute('visibility',showAllState);}").append(eol);
			script.append("for (var i = 0; i < ").append(sets).append("; i++) {").append(eol);
			script.append("  if (document.getElementById(\"textbox\" + i) != null) {").append(eol);
			script.append("    setVisibility(\"textbox\"+i, showAllState);").append(eol);
			script.append("      var number = 0;").append(eol);
			script.append("      while (svgDocument.getElementById('textboxText' + i + number) != null) {").append(eol);
			script.append("        setVisibility('textboxText' + i + number, showAllState);").append(eol);
			script.append("        setVisibility('textboxRect' + i + number, showAllState);").append(eol);
			script.append("        number++;").append(eol);
			script.append("      }").append(eol);
			script.append("  }").append(eol);
			script.append("}").append(eol);
		}
		else if (type != IGraphicTypeConstants.METER) {			
			script.append("var eGroup = document.getElementById(\"exactValues\");").append(eol);
			script.append("if (eGroup != null){").append(eol);
 			script.append("eGroup.setAttribute('visibility', (newStyles[4]=='0'? 'hidden':'visible') );").append(eol);
 	        script.append("}").append(eol);
		}
	
		// update current values if this is not just a preview
		script.append("if (!preview) {for(var i=0; i<6; i++) {currentValues[i] = newStyles[i];}").append(eol);
		if (cookieName != null) {
			script.append("storePreferences(currentValues);").append(eol);
		}
		
		// export the updated preferences, if a variable name is made available
		if (updatedUserPrefsVarName != null) {			
			script.append("	parent.").append(eol);
			script.append(updatedUserPrefsVarName).append(eol);
			script.append(" = getUserPreferences();").append(eol);			
		}	
		script.append("} }").append(eol);
		
		return script.toString();
	}	
	
	/** 
	 * Generates a function which retrieves user preference styles from cookie
	 *  	
	 * @param name 		the name of the cookie
	 */
	public String generateGetPreferencesFunction(String name) {
		String cookie = name;
		
		script = new StringBuffer("function getPreferences() {");
		script.append("var value = get_Cookie(\"");
		script.append(cookie);
		script.append("\");").append(eol);
		script.append("if (value != null) {").append(eol);
		script.append("return value.split(\"&\");").append(eol);
		script.append("} else {").append(eol);
		script.append("return null;}").append(eol);
		script.append("}").append(eol);
			
		return script.toString();
	}
		
	/** 
	 * Generates a function which stores user preference styles in cookie
	 *  	
	 * @param name 		the name of the cookie
	 */
	public String generateStorePreferencesFunction(String name) {
		String cookie = name;
		
		script = new StringBuffer("function storePreferences(storevalues) {");	
		script.append("var value = '';").append(eol);
		script.append("for (var i = 0; i < storevalues.length -1; i++) {").append(eol);
		script.append("  value += storevalues[i] + \"&\";").append(eol);
		script.append("}").append(eol);
		script.append("value += storevalues[storevalues.length-1];").append(eol);
		script.append("set_Cookie('");
		script.append(cookie);
		script.append("', value, expires_date, '/');").append(eol);	
		script.append("}").append(eol);
	
		return script.toString();
	}
	
	/** 
	 * Generates a function to return the current user preferences. The
	 * return value should be treated as a blob.   		
	 */
	public String generateGetUserPreferencesFunction() {
		
		script = new StringBuffer("function getUserPreferences()");	
		script.append("{var value = '';").append(eol);
		script.append("for (var i = 0; i < currentValues.length -1; i++) {").append(eol);
		script.append("  value += currentValues[i] + \",\";").append(eol);
		script.append("}").append(eol);
		script.append("value += currentValues[currentValues.length-1];").append(eol);
		script.append("return value;").append(eol);	
		script.append("}").append(eol);
	
		return script.toString();
	}
	/** 
	 * Generates a function which allows the object attributes to
	 * be returned as a string for debugging purposes, for example, in
	 * an "alert".  
	 */
	public String generateInspectFunction(){
		script = new StringBuffer("function inspect(obj)");
		script.append("{var output=\"\";").append(eol);
        script.append("if(obj==null){").append(eol);
        script.append("return null;} ").append(eol);
        script.append("else{").append(eol);
        script.append("for(var i in obj){").append(eol);
        script.append(" output += i + \" : \" + obj[i] + \"\\n\";}").append(eol);
        script.append("return output;}").append(eol);
        script.append("} ").append(eol);
		return script.toString();
	}
	
	/** 
	 * Generates a function for reading and writing cookies.  
	 */
	public String generateCookieFunctions(){ 
		// note that escape() has been deprecated in ECMAScript v3 and
		// may not work in IE 5.5+ and Netscape 6/Mozilla
		script = new StringBuffer("var today= new Date();");
		// set expiry date for one year from current date	
 		script.append("var expires_date= new Date(today.getTime() + (365 * 24 * 60 * 60 * 1000));").append(eol);
		
		script.append("function get_Cookie(name){").append(eol);
		// can't access browser DOM in NN - have to use window function calls 
		// hasSVGCookieSupport and get_/set_NNCookie have to be defined in embedding HTML
		script.append("if (isNN && hasSVGCookieSupport) {").append(eol);
		script.append(" return get_NNCookie(name);").append(eol);
		script.append("} else if (isIE && top.document.cookie != null) {").append(eol);
		script.append("	var start = top.document.cookie.indexOf(name+\"=\");").append(eol);
		script.append("	var len = start+name.length+1;").append(eol);
		script.append("	if ((!start) && (name != top.document.cookie.substring(0,name.length))) return null;").append(eol);
		script.append("	if (start == -1) return null;").append(eol);
		script.append("	var end = top.document.cookie.indexOf(\";\",len);").append(eol);
		script.append("	if (end == -1) end = top.document.cookie.length;").append(eol);
		script.append("	return unescape(top.document.cookie.substring(len,end));").append(eol);
		script.append("} else {").append(eol);
		script.append(" return null;").append(eol);
		script.append("}").append(eol);
		script.append("} ").append(eol);
			
		script.append("function set_Cookie(name,value,expires,path,domain,secure){").append(eol);
		// can't access browser DOM in NN - have to use window function calls 
		script.append("if (isNN && hasSVGCookieSupport) {").append(eol);
		script.append(" set_NNCookie(name,value,expires,path,domain,secure);").append(eol);
		script.append("} else if (isIE) { ").append(eol);
		script.append("top.document.cookie = name + \"=\" +escape(value) + ").append(eol);
		script.append("((expires) ? \";expires=\" + expires.toGMTString() : \"\") +").append(eol);
		script.append("((path) ? \";path=\" + path : \"\") + ").append(eol);
		script.append("((domain) ? \";domain=\" + domain : \"\") + ").append(eol);
		script.append("((secure) ? \";secure\" : \"\");" ).append(eol);
		script.append("}").append(eol);
		script.append("}").append(eol);
    	
		return script.toString();
	}
	/** 
	 * Generates a function which returns the Changed Opacity
	 *	
 	 */
	public String generateChangeOpacity()
	{
		script = new StringBuffer("function ChangeOpacity(id,NewOpacity){").append(eol);;
		script.append(" var group = 'fill-opacity:'+ NewOpacity + '; stroke: black; stroke-width: 0;';").append(eol);;
		script.append(" if (group != null) {").append(eol);;
		script.append(" }").append(eol);;
		script.append(" document.getElementById(id).setAttribute(\"style\", group);").append(eol);;
		script.append("} ").append(eol);;
		return script.toString();
	}

		//Praful
		//This four java script function can be reduce by 1. Will have to pass four arg to the funcrion	
		// 	Text id/rect id/group id/mouseoverORmouseout
		//Also this will make more general java script function for flyover
		/** 
	 * Generates a function which returns the fly over box for the legent text
	 *	
 	 */
	public String generateFunctionFlyOver()
	{
		script = new StringBuffer("function FlyOver(id){");
		script.append("var flyOverTest = svgDocument.getElementById(\"flyOverText\" + id);").append(eol);
		script.append("flyOverTest.setAttribute('visibility', 'visible');").append(eol);
		script.append("var bbox;").append(eol);
		script.append("	if (flyOverTest != null)").append(eol);
		script.append(" {").append(eol);
		script.append("   bbox = flyOverTest.getBBox();").append(eol);
		script.append("	}").append(eol);
		script.append(" var flyOverBoxElement = svgDocument.getElementById(\"flyOverBox\"  + id);").append(eol);
		script.append(" if (flyOverBoxElement != null && bbox != null)").append(eol);
		script.append(" {").append(eol);		
		script.append("   flyOverBoxElement.setAttribute('x', bbox.x - 3);").append(eol);
		script.append("   flyOverBoxElement.setAttribute('y', bbox.y - 3);").append(eol);
		script.append("   flyOverBoxElement.setAttribute('width', bbox.width + 6);").append(eol);
		script.append("   flyOverBoxElement.setAttribute('height', bbox.height + 6);").append(eol);
		script.append(" }").append(eol);
		script.append(" var flyOverGroup = svgDocument.getElementById(\"flyOverGroup\" + id);").append(eol);
		script.append(" if (flyOverGroup != null)").append(eol);
		script.append(" {").append(eol);		
		script.append("  flyOverGroup.setAttribute('visibility', 'visible');").append(eol);
		script.append(" }").append(eol);
		script.append(" }").append(eol);
		
		
		script.append(("function FlyOut(id){"));
		script.append("var flyOverTest = svgDocument.getElementById(\"flyOverText\" + id);").append(eol);
		script.append("flyOverTest.setAttribute('visibility', 'hidden');").append(eol);		
		script.append("var bbox;").append(eol);
		script.append("	if (flyOverTest != null)").append(eol);
		script.append(" {").append(eol);
		script.append("   bbox = flyOverTest.getBBox();").append(eol);
		script.append("	}").append(eol);
		script.append(" var flyOverBoxElement = svgDocument.getElementById(\"flyOverBox\"  + id);").append(eol);
		script.append(" if (flyOverBoxElement != null && bbox != null)").append(eol);
		script.append(" {").append(eol);		
		script.append("   flyOverBoxElement.setAttribute('x', bbox.x - 3);").append(eol);
		script.append("   flyOverBoxElement.setAttribute('y', bbox.y - 3);").append(eol);
		script.append("   flyOverBoxElement.setAttribute('width', bbox.width + 6);").append(eol);
		script.append("   flyOverBoxElement.setAttribute('height', bbox.height + 6);").append(eol);
		script.append(" }").append(eol);
		script.append(" var flyOverGroup = svgDocument.getElementById(\"flyOverGroup\" + id);").append(eol);
		script.append(" if (flyOverGroup != null)").append(eol);
		script.append(" {").append(eol);		
		script.append("  flyOverGroup.setAttribute('visibility', 'hidden');").append(eol);
		script.append(" }").append(eol);
		script.append(" }").append(eol);
		
		return script.toString();
	}	

		/** 
	 * Generates a function which returns the fly over box for the legent text
	 *	
 	 */
	public String generateFunctionPrefFlyOver()
	{
		script = new StringBuffer("function PrefFlyOver(){");
		script.append("var flyOverText = svgDocument.getElementById(\"hoverBoxText\");").append(eol);
		script.append("flyOverText.setAttribute('visibility', 'visible');").append(eol);
		script.append("var bbox;").append(eol);
		script.append("	if (flyOverText != null)").append(eol);
		script.append(" {").append(eol);
		script.append("   bbox = flyOverText.getBBox();").append(eol);
		script.append("	}").append(eol);
		script.append(" var flyOverBoxElement = svgDocument.getElementById(\"hoverBox\");").append(eol);
		script.append(" if (flyOverBoxElement != null && bbox != null)").append(eol);
		script.append(" {").append(eol);		
		script.append("   flyOverBoxElement.setAttribute('x', bbox.x - 3);").append(eol);
		script.append("   flyOverBoxElement.setAttribute('y', bbox.y - 3);").append(eol);
		script.append("   flyOverBoxElement.setAttribute('width', bbox.width + 6);").append(eol);
		script.append("   flyOverBoxElement.setAttribute('height', bbox.height + 6);").append(eol);
		script.append(" }").append(eol);
		script.append(" var flyOverGroup = svgDocument.getElementById(\"preferencesHoverText\");").append(eol);
		script.append(" if (flyOverGroup != null)").append(eol);
		script.append(" {").append(eol);		
		script.append("  flyOverGroup.setAttribute('visibility', 'visible');").append(eol);
		script.append(" }").append(eol);
		script.append(" }").append(eol);
		
		
		script.append(("function PrefFlyOut(){"));
		script.append("var flyOverText = svgDocument.getElementById(\"hoverBoxText\");").append(eol);
		script.append("flyOverText.setAttribute('visibility', 'hidden');").append(eol);		
		script.append("var bbox;").append(eol);
		script.append("	if (flyOverText != null)").append(eol);
		script.append(" {").append(eol);
		script.append("   bbox = flyOverText.getBBox();").append(eol);
		script.append("	}").append(eol);
		script.append(" var flyOverBoxElement = svgDocument.getElementById(\"hoverBox\");").append(eol);
		script.append(" if (flyOverBoxElement != null && bbox != null)").append(eol);
		script.append(" {").append(eol);		
		script.append("   flyOverBoxElement.setAttribute('x', bbox.x - 3);").append(eol);
		script.append("   flyOverBoxElement.setAttribute('y', bbox.y - 3);").append(eol);
		script.append("   flyOverBoxElement.setAttribute('width', bbox.width + 6);").append(eol);
		script.append("   flyOverBoxElement.setAttribute('height', bbox.height + 6);").append(eol);
		script.append(" }").append(eol);
		script.append(" var flyOverGroup = svgDocument.getElementById(\"preferencesHoverText\");").append(eol);
		script.append(" if (flyOverGroup != null)").append(eol);
		script.append(" {").append(eol);		
		script.append("  flyOverGroup.setAttribute('visibility', 'hidden');").append(eol);
		script.append(" }").append(eol);
		script.append(" }").append(eol);
		
		return script.toString();
	}	
//Praful

}
