package org.eclipse.hyades.ui.sample.svg.generator;
/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: IGraphicDocumentGenerator.java,v 1.4 2005/02/16 22:24:05 qiyanli Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
 
import org.w3c.dom.Document;
import java.io.IOException;
/**
 * <code>IGraphicDocumentGenerator</code> interface provides users with access 
 * to objects which generate graphic visualizations of (potentially) dynamic input data.
 * These objects can generate customized visualizations based on a set of configurable 
 * property values.
 *  
 * @version 1.40.1.6
 */
public interface IGraphicDocumentGenerator {
				
	/**
	 * Returns the type identifier for the graphic generated by this object.
	 * 
	 * @return String  a type identifier	
	 * @see IGraphicTypeConstants 
	 */
	String getGraphicType(); 
	
	/**
	 * Generates and returns a graphic <code>Document</code> object based on input data.
	 * Use this method when no other customization is needed.	 
	 * 
	 * @param data the <code>Document</code> containing the data
	 * @return Document the generated graphic object
	 */
	Document generateGraphicDocument(Document data) throws DataInputProcessingException;
						
	/**
	 * Generates and returns a graphic <code>Document</code> object based on input data	 
	 * 
	 * @param properties customization values 
	 * @return Document the generated graphic object	 
	 */
	Document generateGraphicDocument(GraphicDocumentProperties properties) throws DataInputProcessingException;
		
	/**
	 * Generates a graphic <code>Document</code> object based on the input data and
	 * returns it XML-serialized to a <code>String</code>. 
	 * 
	 * @param properties customization values 
	 * @return String the XML-serialized form of the generated object 
	 */
	String generateAndSerializeToString(GraphicDocumentProperties properties) throws DataInputProcessingException, IOException;
		
	/**
	 * Generates a graphic <code>Document</code> object by based on the input data and then 
	 * XML-serializes it to the specified filename on the local machine where this object
	 * is executing.
	 * 
	 * @param properties customization values 
	 * @param filename the location to where the generated object is to be XML-serialized.
	 * @return boolean <code>true</code>, if the operation is successful
	 */
	boolean generateAndSerializeToFile(GraphicDocumentProperties properties, String filename) throws DataInputProcessingException, IOException;

	/**
	 * XML-serializes the generated <code>Document</code> object to the specified location
	 * on the physical machine where this object is executing.
	 * 
	 * @param generatedDocument the generated graphic
	 * @param filename The location to which the object is serialized.
	 * @return boolean <code>true</code>, if the operation is successful.
	 */
	boolean serializeGeneratedDocumentToFile(Document generatedDocument, String uri) throws IOException;
	
	/**
	 * XML-serializes and returns the generated <code>Document</code> object as a <code>String</code>.
	 * 
	 * @param generatedDocument the generated graphic
	 * @return String the XML-serialized form of the <code>Document</code>
	 */	
	String serializeGeneratedDocumentToString(Document generatedDocument) throws IOException;
		
}

