package org.eclipse.hyades.ui.sample.svg.generator;
/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SVGMeter.java,v 1.4 2005/02/16 22:24:05 qiyanli Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.Hashtable;

import org.w3c.dom.CDATASection;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * <code>SVGMeter</code> generates a meter graphic DOM using Scalable Vector Graphics (SVG).
 * 
 * @version 1.50.1.13
 */
public class SVGMeter extends SVGDocumentGenerator implements IGraphicDocumentStyle, IDataInputConstants, IDataInputProcessingExceptionCodes, Serializable {
	private final MeterDataRetriever dataRetriever;
	
	/**
	 *  Sole constructor.
	 */
	public SVGMeter() {
		super();
		graphicType = IGraphicTypeConstants.METER;		
		dataRetriever = new MeterDataRetriever();
	}
		
	/**
     * @see IGraphicDocumentGenerator#generateGraphicDocument(GraphicDocumentProperties)
     */       
	public Document generateGraphicDocument(GraphicDocumentProperties input) throws DataInputProcessingException {			
		GraphicAttributes graphicAttrs = createGraphicAttributes(input);			
		Document newDataDocument = graphicAttrs.getDataDocument();		
				
		// make sure there is some data, or exit
		if (newDataDocument == null) {
			short code = NO_DATA_DOCUMENT;				
			String element = EMPTY_STRING;								
			DataInputProcessingException e = new DataInputProcessingException(code, element);
			e.setWidgetType(graphicType);
			throw e;
		}		


		DataRange dataRange = null;
		SegmentMarker[] markers = null;
		DataSet dataSet = null;

		// The data range
		try {
			dataRange = dataRetriever.getDataRange(newDataDocument, "");
		} catch (DataInputProcessingException e) {
			e.setWidgetType(graphicType);
			//System.out.println(e.getWidgetType() + " " + e.getElement() + " " + e.getMessage());
			throw e;				
		}					

		if (dataRange == null) {
			throw new DataInputProcessingException("", NO_RANGE_FOUND, DATARANGE, getGraphicType());
		}

		if (dataRange.getType() != DataRange.CONTINUUM) {
			throw new DataInputProcessingException("", UNSUPPORTED_DATA_RANGE_TYPE, DATARANGE, getGraphicType());
		}

		markers = dataRange.getSegmentMarkers();

		// The dataset
		dataSet = dataRetriever.getDatasetWithPosition(newDataDocument, "0");
		

		Document generatedDocument = createSvgDocument(newDataDocument.getImplementation(), Short.toString(graphicAttrs.getGraphicWidth()), Short.toString(graphicAttrs.getGraphicHeight()));
		Element svgRoot = generatedDocument.getDocumentElement();
		registerEventHandler(svgRoot, "onload", "init(evt)");	
		addDefinitions(generatedDocument, graphicAttrs);					
		addJavaScriptFunctions(generatedDocument, graphicAttrs, 0, 0, 0, 0, 0);
 		
 		// graphic outline 
 		if (!graphicAttrs.isOutlineSuppressed()) {
 			addOutline(generatedDocument, graphicAttrs);
 		}	 		
 		 		
		// timestamp
		if (!graphicAttrs.isTimestampSuppressed()) {		
			try {
				String timestamp = dataRetriever.getTimestamp(newDataDocument);
				Date date = GraphicAttributes.parseISO8601Date(timestamp);
	        	addTimeStamp(generatedDocument, graphicAttrs, (date == null) ? timestamp : graphicAttrs.formatDate(date));
			} catch (DataInputProcessingException e) {
				e.setWidgetType(graphicType);
				//System.out.println(e.getWidgetType() + " " + e.getElement() + " " + e.getMessage());
				throw e;
			}
		}	


 		// title bar
 		if (!graphicAttrs.isTitlebarSuppressed()) {	 		
	   		addTitleBar(generatedDocument, graphicAttrs);
 		}
 		
		// preference icon
		if(!graphicAttrs.isUserPreferencesSuppressed() && graphicAttrs.getSvgFormatOnly())
		{
			addPreferenceIcon(generatedDocument, graphicAttrs);
		}
 				
		// if bad data, exit
	   	try {
			Hashtable labels = dataRetriever.getLabels(newDataDocument);		
			DataPoint[] dps = dataSet.getDataPoints();
			DataPoint dp = null;
			if ((dps != null) && (dps.length > 0)) {
				dp = dps[0];
			}
			if (dp.getType() != DataPoint.HOLE) {
				addDial(generatedDocument, graphicAttrs, labels, markers, dp);
			}
		} catch (DataInputProcessingException e) {			
			e.setWidgetType(graphicType);			
			//System.out.println(e.getWidgetType() + " " + e.getElement() + " " + e.getMessage());	
			throw e;		
		}		
		
		
        return generatedDocument;
	}    
		
	// add style element with default style sheet rules for the meter
	private void addDefaultInternalStylesheet(Document generatedDocument, Element parent) {
		StringBuffer styleRules = new StringBuffer(COMMON_STYLE);
		styleRules.append(" ");
		styleRules.append(METER_STYLE);
		
		CDATASection newCDATASection = generatedDocument.createCDATASection(styleRules.toString());
		
		Element newElement = generatedDocument.createElement("style");
		newElement.setAttribute("id", "defaultStyleRules");
		newElement.setAttribute("type", "text/css");
		newElement.appendChild(newCDATASection);
		parent.appendChild(newElement);
	}	
	
	// add defs element with meter definitions 
	protected Element addDefinitions(Document generatedDocument, GraphicAttributes attrs) {
		Element defsElement = super.addDefinitions(generatedDocument, attrs);	
        addDefaultInternalStylesheet(generatedDocument, defsElement);		
	
		// id: tickline		
		Element gElement_tick = generatedDocument.createElement("g");
		gElement_tick.setAttribute("id", "tickline");
		
		Element tickElement = generatedDocument.createElement("polyline");
		tickElement.setAttribute("class", "grid");
		tickElement.setAttribute("points", "0 0 5 0");
		gElement_tick.appendChild(tickElement);
		defsElement.appendChild(gElement_tick);
		
		return defsElement;		
	}	
	
	// add the dial
	//private void addDial(Document generatedDocument, GraphicAttributes attrs, Hashtable titles, String [] divisions, String needle)  throws DataInputProcessingException {
	private void addDial(Document generatedDocument, GraphicAttributes attrs, Hashtable titles, SegmentMarker[] segmentMarkers, DataPoint needle)  throws DataInputProcessingException {
		Element parent = generatedDocument.getDocumentElement();		
		Element newElement, gElement;
		double x, y;
	
		// centre points for everything
		// NumberFormatException not possible here
		int gWidth = attrs.getGraphicWidth();
		int gHeight = attrs.getGraphicHeight();
		double cx = gWidth / 2;
		double cy = gHeight / 2 + 5;		
		
		// get the smaller value of graphicHeight/graphicWidth
		int baseLength = gHeight >= gWidth ? gWidth : gHeight; 
		
		// outer rim
		newElement = generatedDocument.createElement("circle");
		newElement.setAttribute("class", "st1 outerRim");
		newElement.setAttribute("cx", Double.toString(cx));
		newElement.setAttribute("cy", Double.toString(cy));
		newElement.setAttribute("r", Double.toString(baseLength * 0.360));		
		parent.appendChild(newElement);

		// inner rim
		newElement = generatedDocument.createElement("circle");
		newElement.setAttribute("class", "st1 innerRim");
		newElement.setAttribute("cx", Double.toString(cx));
		newElement.setAttribute("cy", Double.toString(cy));
		newElement.setAttribute("r", Double.toString(baseLength * 0.352));			
		parent.appendChild(newElement);

		// meter background
		newElement = generatedDocument.createElement("circle");
		newElement.setAttribute("class", "st1 speedometerBg");
		newElement.setAttribute("cx", Double.toString(cx));
		newElement.setAttribute("cy", Double.toString(cy));
		newElement.setAttribute("r", Double.toString(baseLength * 0.344));
		newElement.setAttribute("fill", "#ffffff");		
		parent.appendChild(newElement);

		// actual rate box
		double boxWidth = baseLength * 0.262;
		double boxYOffset = baseLength * 0.1;
		short boxHeight = 14;
		newElement = generatedDocument.createElement("rect");
		newElement.setAttribute("class", "st1 actualRate");
		newElement.setAttribute("width", Double.toString(boxWidth));
		newElement.setAttribute("height", Short.toString(boxHeight));
		newElement.setAttribute("x", Double.toString(cx - (boxWidth/2)));
		newElement.setAttribute("y", Double.toString(cy + boxYOffset));		
		parent.appendChild(newElement);

		// actual rate		
		String actualRate = (String)titles.get(ACTUAL_RATE);	
		if (actualRate != null) {
			// group for font changes			
			gElement = generatedDocument.createElement("g");
			gElement.setAttribute("id","minmaxRate");
			gElement.setAttribute("class", "minmaxRate anchorAtMiddle");		
			parent.appendChild(gElement);				
			addLabel(generatedDocument, attrs, gElement, actualRate, null, cx, (cy + boxYOffset + boxHeight - 3), 0);					
		}
		
		// rate definition label
		String unitLabel = (String)titles.get(RATE_DEFINITION);
		// group for font changes		
		if (unitLabel != null && !unitLabel.equals(EMPTY_STRING)) {			
			gElement = generatedDocument.createElement("g");
			gElement.setAttribute("id", "rateDefinition");
			gElement.setAttribute("class", "rateDef anchorAtMiddle");
			parent.appendChild(gElement);				
			addLabel(generatedDocument, attrs, gElement, unitLabel, null, cx, (cy + 14), 0);				
		}

		//Draw the meter sections - divisions should have exactly 3 elements	
		if (segmentMarkers.length < 2) {
			//throwException(MAXIMUM_NUMBER_OF_SEGMENT_MARKERS_EXCEEDED, RANGE);
			short code = MISSING_SEGMENT_MARKERS_OR_DATAPOINT_VALUES;				
			String element = DATARANGE;				
			DataInputProcessingException e = new DataInputProcessingException(code, element);
			throw e;	
		}	

		if (segmentMarkers.length > 4) {
			//throwException(MAXIMUM_NUMBER_OF_SEGMENT_MARKERS_EXCEEDED, RANGE);
			short code = MAXIMUM_NUMBER_OF_SEGMENT_MARKERS_EXCEEDED;				
			String element = DATARANGE;				
			DataInputProcessingException e = new DataInputProcessingException(code, element);
			throw e;	
		}	
		
		// NO need to sort, already sorted
/*
		// need to sort these markers
		double [] sorter = new double[divisions.length];
		for (int i = 0; i < divisions.length; i++){
			if (divisions[i] != null) {
				// NumberFormatException not possible here
				sorter[i] = Double.parseDouble(divisions[i]);
			}	
		}		
		if (sorter.length > 1 ){
			Arrays.sort(sorter);
		}
*/
		
		SegmentMarker minMarker = null;
		SegmentMarker safeMarker = null;
		SegmentMarker criticalMarker = null;
		SegmentMarker maxMarker = null;		

		if (segmentMarkers.length == 4) {
			minMarker = segmentMarkers[0];
			criticalMarker = segmentMarkers[1];
			safeMarker = segmentMarkers[2];
			maxMarker = segmentMarkers[3];		
		} else if (segmentMarkers.length == 3) {
			minMarker = segmentMarkers[0];
			criticalMarker = segmentMarkers[1];			
			maxMarker = segmentMarkers[2];	
		} else {
			minMarker = segmentMarkers[0];
			maxMarker = segmentMarkers[1];
		}
		
		double totalUnits = maxMarker.getValue() - minMarker.getValue();

		// determine the labels
		String minLabel = getSegmentLabel(attrs, minMarker);
		String criticalLabel = getSegmentLabel(attrs, criticalMarker);
		String safeLabel = getSegmentLabel(attrs, safeMarker);
		String maxLabel = getSegmentLabel(attrs, maxMarker);

		// calculate the number of units between the different ranges				
		double unitsCriticalRange = 0;
		double unitsSemiCriticalRange = 0;
		double unitsSafeRange = totalUnits;
		double radians = 0.01745329252;

		if (criticalMarker != null) {
			unitsCriticalRange = criticalMarker.getValue() - minMarker.getValue();
			unitsSemiCriticalRange = - criticalMarker.getValue();
			if (safeMarker != null) {
				unitsSemiCriticalRange = safeMarker.getValue() - criticalMarker.getValue();
				unitsSafeRange = maxMarker.getValue() - safeMarker.getValue();
			}
		}

		double safeDegrees = 0;
		double semiDegrees = 0;
		double critDegrees = 0;
		if (unitsSafeRange > 0) {
			safeDegrees = unitsSafeRange / totalUnits * 180 * radians;
		}
		if (unitsSemiCriticalRange > 0) {
			semiDegrees = unitsSemiCriticalRange / totalUnits * 180 * radians;
		}
		if (unitsCriticalRange > 0) {
			critDegrees = unitsCriticalRange / totalUnits * 180 * radians;
		}

		// safe range slice		
		double radius = baseLength * 0.25;	
		String path = EMPTY_STRING;
		String translateString = "translate(" + cx + "," + cy + ")";
		if (safeDegrees > 0) {			
			x = Math.cos(safeDegrees-0.000001) * radius;
			y = Math.sin(safeDegrees-0.000001) * radius;		
			path =
				"M "
				+ radius
				+ " 0 A "
				+ radius
				+ " "
				+ radius
				+ " 0 0 0 "
				+ x
				+ " -"
				+ y
				+ " L 0 0 Z";
			newElement = generatedDocument.createElement("path");
			newElement.setAttribute("class", "st1 safeRange");
			newElement.setAttribute("transform", translateString);		
			newElement.setAttribute("d", path);
			parent.appendChild(newElement);
		}
				
		// semi-critical range slice
		if (semiDegrees > 0) {
			x = Math.cos(semiDegrees) * radius;
			y = Math.sin(semiDegrees) * radius;
			path =
				"M "
				+ radius
				+ " 0 A "
				+ radius
				+ " "
				+ radius
				+ " 0 0 0 "
				+ x
				+ " -"
				+ y
				+ " L 0 0 Z";
			newElement = generatedDocument.createElement("path");
			newElement.setAttribute("class", "st1 semiCritical");
			newElement.setAttribute("transform", translateString + " rotate(-" + safeDegrees / radians + ")");	
			newElement.setAttribute("d", path);
			parent.appendChild(newElement);
		}
		// critical range slice
		if (critDegrees > 0) {
			x = Math.cos(critDegrees) * radius;
			y = Math.sin(critDegrees) * radius;
			path =
				"M "
				+ radius
				+ " 0 A "
				+ radius
				+ " "
				+ radius
				+ " 0 0 0 "
				+ x
				+ " -"
				+ y
				+ " L 0 0 Z";		
			newElement = generatedDocument.createElement("path");
			newElement.setAttribute("class", "st1 critical");
			newElement.setAttribute("transform", translateString + " rotate(-" + (180 - (critDegrees / radians)) + ")");		
			newElement.setAttribute("d", path);
			parent.appendChild(newElement);
		}
		
		Element meterLabelElement = generatedDocument.createElement("g");
		meterLabelElement.setAttribute("id", "meterLabel");
		parent.appendChild(meterLabelElement);
		
		// min rate label 
		x = cx - radius - 2;
		y = cy + 2;	
		addLabel(generatedDocument, attrs, meterLabelElement, minLabel, "minmaxRate anchorAtEnd", x, y, 0); 

		// max rate label
		x = cx + radius + 1;
		y = cy + 2;
		addLabel(generatedDocument, attrs, meterLabelElement, maxLabel, "minmaxRate anchorAtStart", x, y, 0); 
		
		// range of the meter
		double totalRange = maxMarker.getValue() - minMarker.getValue();

		// determine critical marker position and then draw it
		if (criticalMarker != null) {
			ArrayList criticalMarkerInfo = calculateMarkerPosition(criticalMarker.getValue()-minMarker.getValue(), totalRange, radius);
			String criticalMarkerX = (String)criticalMarkerInfo.get(0);
			String criticalMarkerY = (String)criticalMarkerInfo.get(1);
			String criticalMarkerStyle = (String)criticalMarkerInfo.get(2);
			addTranslationLabel(generatedDocument, meterLabelElement, criticalMarkerX, criticalMarkerY, criticalLabel, translateString, criticalMarkerStyle);
		}
		// determine safe marker position and then draw it
		if (safeMarker != null) {
			ArrayList safeMarkerInfo = calculateMarkerPosition(safeMarker.getValue()-minMarker.getValue(), totalRange, radius);
			String safeMarkerX = (String)safeMarkerInfo.get(0);
			String safeMarkerY = (String)safeMarkerInfo.get(1);
			String safeMarkerStyle = (String)safeMarkerInfo.get(2);		
			addTranslationLabel(generatedDocument, meterLabelElement, safeMarkerX, safeMarkerY, safeLabel, translateString, safeMarkerStyle);
		}
			
		// draw 19 ticks within the outline of the meter
		double angle = 9;
		String rotateString = EMPTY_STRING;
		for (int i = 0; i < 19; i++) {
			double rad = angle * Math.PI / 180.0;
			double xpos = (Math.cos(-1 * rad)) * (radius - 5);
			double ypos = (Math.sin(-1 * rad)) * (radius - 5);
			rotateString = "rotate(-" + angle + "," + xpos + "," + ypos + ")";
			// xlink the tick line 
			newElement = generatedDocument.createElement("use");
			newElement.setAttribute("xlink:href", "#tickline");
			newElement.setAttribute("x", Double.toString(xpos));
			newElement.setAttribute("y", Double.toString(ypos));  
			newElement.setAttribute("transform", translateString + rotateString);	  
			parent.appendChild(newElement);
			
			angle += 9;
		}	
		
		// needle
		// NumberFormatException not possible here
		if (needle != null) {
			double needleValue = needle.getValue2();
			if ((needleValue >= minMarker.getValue()) && (needleValue <= maxMarker.getValue())) {
				double needleAngle = (needleValue - minMarker.getValue()) / (maxMarker.getValue() - minMarker.getValue())* 180 + 180;
				newElement = generatedDocument.createElement("rect");		
				newElement.setAttribute("class", "st1 needle");
				newElement.setAttribute("width", Double.toString(radius - 6));
				newElement.setAttribute("height", Double.toString(1.3));
				newElement.setAttribute("x", Double.toString(cx));
				newElement.setAttribute("y", Double.toString(cy));		
				newElement.setAttribute("transform", "rotate(" + needleAngle + "," + Double.toString(cx) +  "," + Double.toString(cy) + ")");				
				parent.appendChild(newElement);
				// needle outer circle
				newElement = generatedDocument.createElement("circle");
				newElement.setAttribute("class", "st1 outerCircle");
				newElement.setAttribute("cx", Double.toString(cx));
				newElement.setAttribute("cy", Double.toString(cy));		
				newElement.setAttribute("r", Double.toString(2.5));
				parent.appendChild(newElement);
				// needle inner circle
				newElement = generatedDocument.createElement("circle");
				newElement.setAttribute("class", "st1 innerCircle");
				newElement.setAttribute("cx", Double.toString(cx));
				newElement.setAttribute("cy", Double.toString(cy));		
				newElement.setAttribute("r", Double.toString(1.7));
				parent.appendChild(newElement);

		        // needle center
    		    newElement = generatedDocument.createElement("circle");
				newElement.setAttribute("class", "st1 center");
				newElement.setAttribute("cx", Double.toString(cx));
				newElement.setAttribute("cy", Double.toString(cy));		
				newElement.setAttribute("r", Double.toString(0.5));
				parent.appendChild(newElement);
			}
		}
	}


	private String getSegmentLabel(GraphicAttributes attrs, SegmentMarker marker) {
		if (marker != null) {
			String label = marker.getLabel();
			return (label != null) ? label : attrs.formatNumber(marker.getValue());
		}
		return "";
	}
	
	// add marker labels	
	private void addTranslationLabel(Document generatedDocument, Element parent, String x, String y, String label, String translateString, String styleSheetClass) {
		Element newElement;
		newElement = generatedDocument.createElement("text");
		newElement.setAttribute("class", "critDiv " + styleSheetClass);
		newElement.setAttribute("x", x);
		newElement.setAttribute("y", y);
		newElement.setAttribute("transform", translateString);	
		newElement.appendChild(generatedDocument.createTextNode(label));
		parent.appendChild(newElement);
	}
	
	// Get marker information
	// @return an ArrayList with the following elements
	//  - the first element is the marker degree value
	//  - the second element is the marker x position
	//  - the third element is the marker y position
	//  - the fourth element is the marker anchor style
	ArrayList calculateMarkerPosition(double marker, double totalRange, double radius) {
		double markerDeg = (totalRange - marker) / totalRange * 180.0;
		double markerRad = markerDeg * Math.PI / 180.0;
		double markerX = (Math.cos(-1 * markerRad)) * (radius + 3);
		double markerY = (Math.sin(-1 * markerRad)) * (radius + 3);		
		
		String markerStyle = EMPTY_STRING;
		if (markerDeg < 90) 
			markerStyle = "anchorAtStart";
		else if (markerDeg == 90)
			markerStyle = "anchorAtMiddle";
		else if (markerDeg > 90) 
			markerStyle = "anchorAtEnd";

		ArrayList markerPosInfo = new ArrayList();
		markerPosInfo.add(Double.toString(markerX));
		markerPosInfo.add(Double.toString(markerY));
		markerPosInfo.add(new String(markerStyle));
		
		return markerPosInfo;	
	}
	
	/**
	 * Retrieves all configuration attributes from the data retriever
	 */
	private void getConfigAttributes(GraphicAttributes attrs) throws DataInputProcessingException {
		Hashtable configProperties = dataRetriever.getConfigurationAttributes(attrs.getConfigDocument(), attrs.getDataDocument());	
		if (configProperties == null) {
			return;
		}	
		String width = (String)configProperties.get(WIDTH);
		String height = (String)configProperties.get(HEIGHT);
		//String fontFamily = (String)configProperties.get(FONT);
		String title = (String)configProperties.get(TITLE);
		String timestampPrefix = (String)configProperties.get(TIMESTAMP_PREFIX);
		
		if (width != null) {
			attrs.setGraphicWidth(width);
		}	
		
		if (height != null) {
			attrs.setGraphicHeight(height);
		}	
		
		//if (fontFamily != null) {
		//	attrs.setGraphicFont(fontFamily);			
		//}			
		
		if (title != null) {
			attrs.setGraphicTitle(title);			
		}
			
		if (timestampPrefix != null) {
			attrs.setGraphicTimestampPrefix(timestampPrefix);			
		}	
		
		/* Test 

		System.out.println("From the input documents Graphic Color Palette: " + attrs.getGraphicColorPalette());
		System.out.println("From the input documents Graphic Color Palette Name: " + attrs.getGraphicColorPaletteName());
		System.out.println("From the input documents Graphic Font: " + attrs.getGraphicFont());
		System.out.println("From the input documents Graphic Height: " + attrs.getGraphicHeight());
		System.out.println("From the input documents Graphic Legend Title: " + attrs.getGraphicLegendTitle());
		System.out.println("From the input documents Graphic Timestamp Prefix: " + attrs.getGraphicTimestampPrefix());
		System.out.println("From the input documents Graphic Title: " + attrs.getGraphicTitle());
		System.out.println("From the input documents Graphic Type: " + attrs.getGraphicType());
		System.out.println("From the input documents Graphic Width: " + attrs.getGraphicWidth());
		System.out.println("From the input documents Preferences Icon Text: " + attrs.getPreferencesIconText());
		System.out.println("From the input documents Preferences Page: " + attrs.getPreferencesPage());
		System.out.println("From the input documents Preferences Page Height: " + attrs.getPreferencesPageHeight());
		System.out.println("From the input documents Preferences Page Width: " + attrs.getPreferencesPageWidth());
		/* Test */	
		
	}
	
	/**
	 * Initialize the meter chart attributes
	 */
	private GraphicAttributes createGraphicAttributes(GraphicDocumentProperties input) throws DataInputProcessingException {
		GraphicAttributes graphicAttrs = new GraphicAttributes();
			
		// set up the defaults
		graphicAttrs.setGraphicWidth(METER_DEFAULT_WIDTH);
		graphicAttrs.setPreferencesPage(IUserPreferencesConstants.METER_PREFS);
		graphicAttrs.setPreferencesPageWidth(IUserPreferencesConstants.PREFS_WIDTH);
		graphicAttrs.setPreferencesPageHeight(IUserPreferencesConstants.METER_PREFS_HEIGHT);
		
		// get the input documents
		graphicAttrs.setConfigDocument(input.getConfigDocument());
		graphicAttrs.setDataDocument(input.getDataDocument());
		
		// (re)set any properties from the input documents		
		getConfigAttributes(graphicAttrs);
		
		// (re)set any properties from the input bean
		graphicAttrs.setAttributesFrom(input);		
				
		return graphicAttrs;
	}	
}

