package org.eclipse.hyades.ui.sample.svg.generator;
/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SVGStackBarChart.java,v 1.4 2005/02/16 22:24:05 qiyanli Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

import java.io.Serializable;
import java.util.Date;

import org.w3c.dom.CDATASection;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
/**
 * <code>SVGStackBarChart</code> generates a stacked bar chart graphic DOM using Scalable Vector Graphics (SVG).
 * 
 * @version 1.66.1.22
 */
public class SVGStackBarChart extends SVGXYChart implements IGraphicDocumentStyle, IDataInputProcessingExceptionCodes, Serializable {
	/**
	 * Sole constructor.
	 */
	public SVGStackBarChart() {
		super();
		graphicType = IGraphicTypeConstants.STACKBAR_CHART;		
	}
	
	/**
	 * @see IGraphicDocumentGenerator#generateGraphicDocument(GraphicDocumentProperties)
	 */	
	public Document generateGraphicDocument(GraphicDocumentProperties input) throws DataInputProcessingException {	
		GraphicAttributes graphicAttrs = createGraphicAttributes(input);
		Document newDataDocument = graphicAttrs.getDataDocument();		
		
		String rangeMaxValue = null;
		int numberOfDatasets = 0;
		int gridXOffset = 60;
		int halfGridXOffset = gridXOffset/2;
		int gridYOffset = 50;
	
		// The axes
		try {
			xAxis = dataRetriever.getDataRange(newDataDocument, "S");
		} catch (DataInputProcessingException e) {
			e.setWidgetType(getGraphicType());
			//System.out.println(e.getWidgetType() + " " + e.getElement() + " " + e.getMessage());
			throw e;				
		}					

		try {
			yAxis = dataRetriever.getDataRange(newDataDocument, "W");
		} catch (DataInputProcessingException e) {
			e.setWidgetType(getGraphicType());
			//System.out.println(e.getWidgetType() + " " + e.getElement() + " " + e.getMessage());
			throw e;				
		}

		if ((xAxis == null) || (yAxis == null)) {
			throw new DataInputProcessingException("", NO_RANGE_FOUND, DATARANGE, getGraphicType());
		}
		if ((xAxis.getType() != DataRange.CATEGORIZATION) || (yAxis.getType() != DataRange.CONTINUUM)) {
			throw new DataInputProcessingException("", UNSUPPORTED_DATA_RANGE_TYPE, DATARANGE, getGraphicType());
		}

		xMarkers = xAxis.getSegmentMarkers();
		yMarkers = yAxis.getSegmentMarkers();

		if (yMarkers != null && yMarkers.length > 0) { 
			rangeMaxValue = (String)yMarkers[yMarkers.length - 1].getValueString();
		}
		
		// The datasets
		numberOfDatasets = dataRetriever.getNumberOfDatasets(newDataDocument);
		dataSets = new DataSet[numberOfDatasets];
		for(int i=0; i < numberOfDatasets; i++) {
            dataSets[i] = dataRetriever.getDatasetWithPosition(newDataDocument, Integer.toString(i));
		}
		//get the Fly over text for the legend item.				
		legendFlyOverLabels = getLegendFlyOverLabels(dataSets);		
		//get the max size of Ymarker Width.		
		double maxLengthOfYmarkerWidth = getMaxLengthOfYmarker(yMarkers, graphicAttrs);
	    double maxAllowableYMarkerWidth = graphicAttrs.getGraphicWidth() / 4;
	    if (maxLengthOfYmarkerWidth > maxAllowableYMarkerWidth) {
			maxLengthOfYmarkerWidth = maxAllowableYMarkerWidth;
		}
		
		//get the max size of Legent Lable Width.		
		double maxLengthOfLegendLableWidth = getMaxLengthOfLegendLabel(graphicAttrs, dataSets);
	    double maxAlloableLegendWidth = graphicAttrs.getGraphicWidth() / 3;
   		if (maxLengthOfLegendLableWidth > maxAlloableLegendWidth) {
			maxLengthOfLegendLableWidth = maxAlloableLegendWidth;
		}

		// Start calculations
        if (graphicAttrs.isLegendSuppressed()) {
			maxLengthOfLegendLableWidth = 0;
        }
		int xAxisLength = (int)(graphicAttrs.getGraphicWidth() - (maxLengthOfLegendLableWidth + maxLengthOfYmarkerWidth + gridXOffset));
   		int xLabelRowCount = getNumberOfAxisLabels(graphicAttrs, xAxisLength, xAxis);									
		int yAxisLength = (int)(graphicAttrs.getGraphicHeight() - (gridYOffset * 2.5) - (xLabelRowCount * 10));			


		Document generatedDocument = createSvgDocument(newDataDocument.getImplementation(), Short.toString(graphicAttrs.getGraphicWidth()), Short.toString(graphicAttrs.getGraphicHeight()));
		Element svgRoot = generatedDocument.getDocumentElement();
		registerEventHandler(svgRoot, "onload", "init(evt)");	
		addDefinitions(generatedDocument, graphicAttrs, numberOfDatasets);					
		addJavaScriptFunctions(generatedDocument, graphicAttrs, numberOfDatasets, gridXOffset, gridYOffset, xAxisLength, yAxisLength);
		
		// graphic outline 
 		if (!graphicAttrs.isOutlineSuppressed()) {
 			addOutline(generatedDocument, graphicAttrs);
 		}	 		
 		 	
 	   // timestamp	 		
		if (!graphicAttrs.isTimestampSuppressed()) {		 	   
		   try {
				String timestamp = dataRetriever.getTimestamp(newDataDocument);
				Date date = GraphicAttributes.parseISO8601Date(timestamp);
	        	addTimeStamp(generatedDocument, graphicAttrs, (date == null) ? timestamp : graphicAttrs.formatDate(date));
		} catch (DataInputProcessingException e) {
				e.setWidgetType(graphicType);
				//System.out.println(e.getWidgetType() + " " + e.getElement() + " " + e.getMessage());
				throw e;
			}
		}
		      
	   // title bar
 		if (!graphicAttrs.isTitlebarSuppressed()) {	 		
	   		addTitleBar(generatedDocument, graphicAttrs);
 		}
 		// need to adjust graphic vertical position
 		else {
 			gridYOffset = super.resetGridOffsetForCenter(graphicAttrs.getGraphicHeight(), yAxisLength, 3.0);	
 		}	

		// preference icon
		if(!graphicAttrs.isUserPreferencesSuppressed() && graphicAttrs.getSvgFormatOnly())
		{
			addPreferenceIcon(generatedDocument, graphicAttrs);
		}
		
		// the actual x offset
		gridXOffset = (halfGridXOffset) + (int)maxLengthOfYmarkerWidth;
		 							
		// work out where to put the data horizontally
		evenlyDistributeSegmentMarkers(xAxis);

		// border, tick marks etc.
        addGrid(generatedDocument, svgRoot, xAxisLength, yAxisLength, gridXOffset, gridYOffset, xAxis, yAxis);

		// x and y axis labels
        addAxisLabels(generatedDocument, graphicAttrs, svgRoot, xAxisLength, yAxisLength, gridXOffset, gridYOffset, xAxis, yAxis);
                
        // axis titles
        addTitles(generatedDocument, graphicAttrs, svgRoot, xAxisLength, yAxisLength, gridXOffset, gridYOffset, xAxis, yAxis, xLabelRowCount, rangeMaxValue);

       // data lines and area
		Element gTextBoxElement = generatedDocument.createElement("g");   
        // data bars
        addDataSets(generatedDocument, graphicAttrs, svgRoot, xAxisLength, yAxisLength, gridXOffset, gridYOffset, xAxis, yAxis, dataSets); 

       // legend
        if (!graphicAttrs.isLegendSuppressed()) {      		
	  		try {
	       		String [] legendLabels = dataRetriever.getLegendLabels(newDataDocument);		       
				addLegend(generatedDocument, graphicAttrs, gTextBoxElement, legendLabels, legendFlyOverLabels, (int)maxLengthOfYmarkerWidth + xAxisLength + (halfGridXOffset), gridYOffset); 																
	  		} catch (DataInputProcessingException e) {
	  			e.setWidgetType(graphicType);
	  			//System.out.println(e.getWidgetType() + " " + e.getElement() + " " + e.getMessage());	  			
	  			throw e;
	  		} 		
    	}
		svgRoot.appendChild(gTextBoxElement);
    	return generatedDocument;
	}
	
	// Go through each x position and then draw the total value of each bar
//	private void addAllBarValues(Document generatedDocument, GraphicAttributes attrs, Element parent, double yUnits, int xTicks, String [] barTotals, int xAxisLength, int yAxisLength, int gridXOffset, int gridYOffset)	{
	private void addAllBarValues(Document generatedDocument, GraphicAttributes attrs, Element gValueElement, 
									int xAxisLength, int yAxisLength, DataRange xAxis, DataRange yAxis, 
									double barWidth, double[] accumulatedYValues) {

		SegmentMarker[] xsms = xAxis.getSegmentMarkers();
		double yMax = yAxis.getMaxValue();
		double yMin = yAxis.getMinValue();
		for (int i = 0; i < xsms.length; i++) {	
			double yValue = accumulatedYValues[i];
			if (yValue < yMin) continue;
			if (yValue > yMax) continue;
			double y = yAxisLength * (yMax - yValue) / (yMax - yMin);
			double x = (xsms[i].getPosition() * xAxisLength);
			String label = attrs.formatNumber(yValue);
			double ypos = (yValue >= 0) ? (y-5) : (y+12);
			addLabel(generatedDocument, attrs, gValueElement, label, "dataValues, anchorAtMiddle", x, ypos, 0);
		}
	}


	/*
	 * 	Draw bars for each barset
	 *  @param generatedDocument the document
	 *  @param oneBarSetElement the parent element for bar charts of the same dataset
	 *  @param gToogleElement the parent element for the textbox
	 *  @param barsetNumber the dataset number
	 *  @param xAxisLength the width of the chart
	 *  @param yAxisLength the height of the chart
	 *  @param xAxis the x axis
	 *  @param yAxis the y axis
	 *  @param barWidth the width of the bar
	 *  @param accumulatedYValue the array of latest Y value of each bar. The values are updated upon return.
	 *  @param dataSet the dataset
	 */
	private void addBarSet(Document generatedDocument, GraphicAttributes attrs, Element oneBarSetElement, Element gToggleElement, int barsetNumber, double xAxisLength,
							 int yAxisLength, DataRange xAxis, DataRange yAxis, double barWidth, double[] accumulatedYValues, DataSet dataSet) throws DataInputProcessingException {

		SegmentMarker[] xsms = xAxis.getSegmentMarkers();
		double yMax = yAxis.getMaxValue();
		double yMin = yAxis.getMinValue();
		int boxcount = 0;
		for (int i = 0; i < xsms.length; i++) {	
			DataPoint dp = dataSet.findDataPoint(xsms[i].getValue());
			
			if (dp == null) continue;
			if (dp.getType() == DataPoint.HOLE) continue;
			double dataValue = dp.getValue2();
			
			double yValue = dataValue + accumulatedYValues[i];
			double lastYValue = accumulatedYValues[i];
			accumulatedYValues[i] = yValue;
			
			if (lastYValue < yMin) lastYValue = yMin;
			if (lastYValue > yMax) lastYValue = yMax;

			if (yValue < yMin) yValue = yMin;
			if (yValue > yMax) yValue = yMax;
			double y = yAxisLength * (yMax - yValue) / (yMax - yMin);
			double y0 = yAxisLength * (yMax - lastYValue) / (yMax - yMin);
			double x = (xsms[i].getPosition() * xAxisLength) - (barWidth / 2);

			double upperY = (y < y0) ? y : y0;
			double height = (y < y0) ? (y0-y) : (y-y0);
			
			if (height <= 0) continue;

			// draw the bar
			Element newElement = generatedDocument.createElement("rect");		
			newElement.setAttribute("x", Double.toString(x));
			newElement.setAttribute("y", Double.toString(upperY));
			newElement.setAttribute("width", Double.toString(barWidth));
			newElement.setAttribute("height", Double.toString(height));

			oneBarSetElement.appendChild(newElement);
			
			// add the text box
			String label = dp.getDisplayLabel(attrs);
			double boxXPos = x + barWidth + 2.0;
			double boxYPos = upperY;
			double boxWidth = 16 + (label.length() - 1) * 6.0;
			double boxHeight = 18;
			addEachTextBox(generatedDocument, attrs, gToggleElement, boxXPos, boxYPos, boxWidth, boxHeight, label, barsetNumber, boxcount++);
		}
	}



	// add bars
	private void addDataSets(Document generatedDocument, GraphicAttributes attrs, Element parent, 
								int xAxisLength, int yAxisLength, int gridXOffset, int gridYOffset, 
								DataRange xAxis, DataRange yAxis, DataSet[] dataSets) 
							throws DataInputProcessingException {

		int numberOfSegments = xAxis.getSegmentMarkers().length;
		double barWidth = (numberOfSegments > 0) ? (xAxisLength / numberOfSegments / 2) : 0;
		double[] accumulatedYValues = new double[numberOfSegments];
		
		// initialize the array
		for(int i=0; i < numberOfSegments; i++) {
			accumulatedYValues[i] = 0;
		}

		// the top element for visibility toggle of text
		Element gValueElement = generatedDocument.createElement("g");
		String stylesheetClass = "dataValues anchorAtMiddle";		
		gValueElement.setAttribute("id","exactValues");
		gValueElement.setAttribute("transform", "translate(" + gridXOffset + "," + gridYOffset + ")");
		gValueElement.setAttribute("visibility", "hidden");
		gValueElement.setAttribute("onclick","toggleVisibility(\"exactValues\")");
		parent.appendChild(gValueElement);

		for(int i=0; i < dataSets.length; i++) {
			Element oneBarSetElement = generatedDocument.createElement("g");
			oneBarSetElement.setAttribute("id", "dataColour" + i);
			oneBarSetElement.setAttribute("class", "shape" + i);
			oneBarSetElement.setAttribute("transform", "translate(" + gridXOffset + "," + gridYOffset + ")");
			oneBarSetElement.setAttribute("onclick","toggleVisibility(\"exactValues\")");
			parent.appendChild(oneBarSetElement);	

			Element gToggleElement = generatedDocument.createElement("g");
			String boxStylesheetClass = "dataValues anchorAtMiddle";		
			gToggleElement.setAttribute("id","textbox" + i);
			gValueElement.appendChild(gToggleElement);
		
			addBarSet(generatedDocument, attrs, oneBarSetElement, gToggleElement, i, xAxisLength, yAxisLength, xAxis, yAxis,
			          barWidth, accumulatedYValues, dataSets[i]);
		}

		addAllBarValues(generatedDocument, attrs, gValueElement, xAxisLength, yAxisLength, xAxis, yAxis, barWidth, accumulatedYValues);
	}

	// specific to stack bar chart
	private void addDefaultInternalStylesheet(Document generatedDocument, GraphicAttributes attrs, Element parent, int sets) {
		StringBuffer styleRules = new StringBuffer(COMMON_STYLE);
		styleRules.append(XYCHART_STYLE);				
		String [] palette = (String [])(attrs.getGraphicColorPalette()).get(attrs.getGraphicColorPaletteName());
		// palette has limited number of colours, so loop if more sets
		int paletteLength = palette.length;		
		boolean hasLegend = !attrs.isLegendSuppressed();
		for (int i = 0, j = 0; i < sets; i++) {
		  j = (i + 1) % paletteLength;
		  j = (j == 0) ? paletteLength - 1 : j-1;			  			  
		  styleRules.append(" .dataset" + i + "{stroke-width:0.75pt; stroke:" + palette[j] + ";}"); 
		  if (hasLegend) {
		  	styleRules.append(" .shape" + i + "{stroke:none; fill:" + palette[j] + ";}");
		  }
		}
		
		CDATASection newCDATASection = generatedDocument.createCDATASection(styleRules.toString());
		
		Element newElement = generatedDocument.createElement("style");
		newElement.setAttribute("id", "defaultStyleRules");
		newElement.setAttribute("type", "text/css");
		newElement.appendChild(newCDATASection);
		parent.appendChild(newElement);
	}	
	
	// specific to stack bar chart
	protected void addDefinitions(Document generatedDocument, GraphicAttributes attrs, int numberOfDatasets) {
		Element defsElement = super.addDefinitions(generatedDocument, attrs);
       
        // default internal style rules
		addDefaultInternalStylesheet(generatedDocument, attrs, defsElement, numberOfDatasets);			
	}	
		
	// add total value over each bar
	private void addExactValue(Document generatedDocument, GraphicAttributes attrs, Element parent, double x, double y, String value) {
		addLabel(generatedDocument, attrs, parent, value, "dataValues, anchorAtMiddle", x, y, 0);
	}
	
	
	/**
	 * Initialize the stacked bar chart attributes
	 */
	private GraphicAttributes createGraphicAttributes(GraphicDocumentProperties input) throws DataInputProcessingException {
		GraphicAttributes graphicAttrs = new GraphicAttributes();
			
		// set up the defaults
		graphicAttrs.setGraphicWidth(STACKBAR_DEFAULT_WIDTH);
		graphicAttrs.setPreferencesPage(IUserPreferencesConstants.STACKBAR_PREFS);
		graphicAttrs.setPreferencesPageWidth(IUserPreferencesConstants.PREFS_WIDTH);
		graphicAttrs.setPreferencesPageHeight(IUserPreferencesConstants.STACKBAR_PREFS_HEIGHT);
		
		// get the input documents
		graphicAttrs.setConfigDocument(input.getConfigDocument());
		graphicAttrs.setDataDocument(input.getDataDocument());
		
		// (re)set any properties from the input documents		
		getConfigAttributes(graphicAttrs);
		
		// (re)set any properties from the input bean
		graphicAttrs.setAttributesFrom(input);		
				
		return graphicAttrs;
	}	
}
