/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: IAssociationMappingRegistry.java,v 1.4 2005/02/16 22:24:04 qiyanli Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.ui.extension;

import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.ui.IPropertyListener;

import org.eclipse.hyades.ui.util.IDisposable;

/**
 * 
 * <p>This interface is not intended to be implemented by clients.  Instances can 
 * be obtained by 
 * {@link org.eclipse.hyades.ui.HyadesUI#createAssociationMappingRegistry()}.
 * 
 * @author marcelop
 * @author popescu
 * @since 0.0.1
 */
public interface IAssociationMappingRegistry
extends IDisposable
{
	/**
	 * Property id used to notify listeners that the content of a registry
	 * has changed.
	 */
	public static final int PROP_CONTENTS = 0x01;
	
	/**
	 * Initializes this registry.
	 * 
	 * <p>One of the <code>initialize</code> methods must be invoked at the beginning of 
	 * the life cycle of the registry.  Subsequents invocations of any of those methods 
	 * don't produce any effect in the registry.
	 * 
	 * @param objectCategory The unique identifier of the category of object for which
	 * the mappings refers to.  An example of <code>objectCategory</code> is the
	 * file extension of the resource in which the object is saved.  This value is used to 
	 * persist the registry information in the preference store.
	 * 
	 * @param pluginId The id of the plugin that declares the association descriptors of this
	 * registry's mappings.  This value is used to persist the registry information in the preference 
	 * store.
	 * 
	 * @param filter The association description filter to be used by this registry
	 * 
	 * @param preferenceStore.  A preference store that allows a this association mapping registry 
	 * to store and retrieve user preferences.  If <code>null</code>, no preference information is
	 * persisted.
	 * 
	 * @throws IllegalArgumentException if <code>objectCategory</code> is <code>null</code>.
	 */
	public void initialize(String objectCategory, IAssociationDescriptorFilter filter, IPreferenceStore preferenceStore)
	throws IllegalArgumentException;
	
	/**
	 * Returns the association descriptor filter to be used with this
	 * mapping.
	 * @param filter
	 */
	public IAssociationDescriptorFilter getFilter();

	/**
	 * Returns the category of object that the mappings in this registry
	 * are referring to.
	 * @param type
	 */
	public String getObjectCategory();
	
	/**
	 * Returns the preference store used by this association mapping registry.
	 * @return IPreferenceStore
	 */
	public IPreferenceStore getPreferenceStore();
	
	/**
	 * Retrieves a mapping for a given extension point returning an existing
	 * mapping or creating a new one.
	 * 
	 * <p>The association mapping contains the association descriptors that applies
	 * to the category of object defined in this registry.
	 * 
	 * @param extensionPoint  The name of the extension point that is used to defined the
	 * association descriptor.  An example of <code>extensionPoint</code> is the plugin 
	 * extension point name of the editors that are used with this registry's category of 
	 * object.  This value is used to persist the mapping information in the preference 
	 * store.  The extensionPoint should be unique in this registry. 
	 * 
	 * @param pluginId The id of the plugin that declares the association descriptors of the
	 * mapping to be registered.  If this parameter is <code>null</code> then no information
	 * is retrieved from the plugin registry.
	 * 
	 * @return IAssociationMapping the association mapping.
	 * @throws IllegalArgumentException if <code>extensionPoint</code> is 
	 * <code>null</code>.
	 */
	public IAssociationMapping retrieveAssociationMapping(String extensionPoint, String pluginId)
	throws IllegalArgumentException;; 
	
	/**
	 * Retrieves a mapping for a given extension point returning null if no
	 * mapping is available.
	 * 
	 * <p>The association mapping contains the association descriptors that applies
	 * to the category of object defined in this registry.
	 * 
	 * @param extensionPoint  The name of the extension point that is used to defined the
	 * association descriptor.  An example of <code>extensionPoint</code> is the plugin 
	 * extension point name of the editors that are used with this registry's category of 
	 * object.
	 * 
	 * @return IAssociationMapping the association mapping
	 */
	public IAssociationMapping getAssociationMapping(String extensionPoint);
	
	/**
	 * Returns the association mappings of this registry.
	 * @return IAssociationMapping[]
	 */
	public IAssociationMapping[] getAssociationMappings();
	
	/**
	 * Takes a snapshot of the current state of this registry.  The snapshot
	 * is kept in memory.
	 * 
	 * <p>A new checkpoint overwrites a previous one.
	 * 
	 * @return <code>true</code> if the snapshot was taken or <code>false</code> 
	 * otherwise.
	 * @see #rollback()
	 * @see #commit()
	 */
	public boolean checkpoint();
	
	/**
	 * Clears a previous defined checkpoint.  This method neither changes the
	 * current state of the registry nor affects the preference settings.
	 * @see #checkpoint()
	 */
	public void clearCheckpoint();
	
	/**
	 * Commits the current state of this registry.  The consequences of invoking this
	 * method are:</OL>
	 * <LI>If {@link #getPreferenceStore()} is not <code>null</code>, then the state 
	 * is persisted in the preferences.</LI>
	 * <LI>The committed state is set as the current checkpoint.  This means that by invoking 
	 * {@link #rollback()} without any invocation of {@link #checkpoint()} returns
	 * the state of the registry to the committed state.</LI>
	 * </OL>
	 * @return <code>true</code> if the registry changes were committed or 
	 * <code>false</code> otherwise.
	 * @see #checkpoint()
	 * @see #rollback()
	 */
	public boolean commit();

	/**
	 * Rolls this registry back to the state of the last checkpoint.
	 * @return <code>true</code> if the registry was rolled back or <code>false</code> 
	 * otherwise.
	 * @see #checkpoint()
	 * @see #commit()
	 */
	public boolean rollback();
	
	/**
	 * Adds a listener that is notified whenever a commit of this registry is 
	 * storing one or more changes.  In other words there aren't notifications
	 * for commits that are performed on unchanged registries.
	 * @param propertyListener
	 * @see #commit()
	 */
	public void addCommitChangeListener(IPropertyListener propertyListener);

	/**
	 * Removes a commit listener.
	 * @param propertyListener
	 * @see #addCommitChangeListener(IPropertyListener)
	 * @see #commit()
	 */
	public void removeCommitChangeListener(IPropertyListener propertyListener);	
}
