/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: LabelProviderPicker.java,v 1.4 2005/02/16 22:24:05 qiyanli Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.provider;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.jface.util.ListenerList;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ILabelProviderListener;
import org.eclipse.swt.graphics.Image;

import org.eclipse.hyades.ui.util.ILabelAndDescriptionProvider;

/**
 * This label provider is a type of label provider decorator that allows clients to 
 * select a provider to handle the requests.
 * 
 * <p>Listeners added or removed to this label provider are automatically added or
 * removed to all the providers that were added.
 * 
 * @author marcelop
 * @since 0.0.1
 */
public class LabelProviderPicker 
implements ILabelAndDescriptionProvider
{
	private ListenerList listenerList;
	private List providers;
	private int currentProvider;

	/**
	 * Constructor for LabelProviderPicker
	 */
	public LabelProviderPicker()
	{
		super();
		listenerList = new ListenerList();
		providers = new ArrayList();
		currentProvider = -1;
	}

	/**
	 * @see org.eclipse.jface.viewers.IBaseLabelProvider#dispose()
	 */
	public void dispose()
	{
		currentProvider = -1;
		for (Iterator i = providers.iterator(); i.hasNext();)
		{
			ILabelProvider treeContentProvider = (ILabelProvider)i.next();
			treeContentProvider.dispose();
		}
		providers.clear();
	}

	/**
	 * Adds a provider.  The provider are kept in the
	 * order they are added.
	 * 
	 * <p>This method sets the specified provider as the current if no other
	 * provider is already defined as current.
	 * 
	 * @param provider
	 */
	public void addProvider(ILabelProvider provider)
	{
		if(providers.add(provider))
		{
			if(currentProvider < 0)
				currentProvider = providers.size()-1;
				
			Object[] listeners = listenerList.getListeners();
			for(int i = 0, maxi = listeners.length; i < maxi; i++)
				provider.addListener((ILabelProviderListener)listeners[i]);
		}		
	}
	
	/**
	 * Removes all the providers.
	 */
	public void clearProviders()
	{
		Object[] listeners = listenerList.getListeners();
		for (Iterator i = providers.iterator(); i.hasNext();)
		{
			ILabelProvider provider = (ILabelProvider)i.next();
			for(int j = 0, maxj = listeners.length; j < maxj; j++)
				provider.addListener((ILabelProviderListener)listeners[j]);
		}

		currentProvider = -1;
		listenerList.clear();
		providers.clear();
	}
	
	/**
	 * Removes the provider associated with a given index.  The first 
	 * provider added to this picker has index 0.
	 * 
	 * <p>This method removes from the label provider identified by the index
	 * all the listeners.
	 * 
	 * @param index
	 */
	public void removeProvider(int index)
	{
		if(index == currentProvider)
			currentProvider = -1;

		else if(index < currentProvider)
			currentProvider--;
			
		ILabelProvider provider = (ILabelProvider)providers.remove(index);
		if(provider != null)
		{
			Object[] listeners = listenerList.getListeners();
			for(int i = 0, maxi = listeners.length; i < maxi; i++)
				provider.removeListener((ILabelProviderListener)listeners[i]);			
		}
	}
	
	/**
	 * Sets the current provider.  If the index is not in used the current
	 * selection is kept,
	 * @param index
	 * @return <code>true</code> if the current provider was set or <code>false</code> 
	 * otherwise.
	 */
	public boolean setCurrentProvider(int index)
	{
		if((index < 0) || (index >= providers.size()))
			return false;
		if(index == currentProvider)
			return true;
			
		currentProvider = index;
		return true;
	}
	
	/**
	 * Returns the current provider
	 * @return ILabelProvider
	 */
	public ILabelProvider getCurrentProvider()
	{
		if(currentProvider < 0)
			return null;
		return (ILabelProvider)providers.get(currentProvider);
	}
	
	/**
	 * Returns the index of the current provider.
	 * @return int
	 */
	public int getCurrentProviderIndex()
	{
		return currentProvider;
	}
	
	/**
	 * @see org.eclipse.jface.viewers.IBaseLabelProvider#addListener(org.eclipse.jface.viewers.ILabelProviderListener)
	 */
	public void addListener(ILabelProviderListener listener)
	{
		listenerList.add(listener);
		for(Iterator i = providers.iterator(); i.hasNext();)
			((ILabelProvider)i.next()).addListener(listener);
	}

	/**
	 * @see org.eclipse.jface.viewers.IBaseLabelProvider#removeListener(org.eclipse.jface.viewers.ILabelProviderListener)
	 */
	public void removeListener(ILabelProviderListener listener)
	{
		listenerList.remove(listener);
		for(Iterator i = providers.iterator(); i.hasNext();)
			((ILabelProvider)i.next()).removeListener(listener);
	}

	/**
	 * @see org.eclipse.hyades.ui.util.ILabelAndDescriptionProvider#getDescription(java.lang.Object)
	 */
	public String getDescription(Object element)
	{
		if(getCurrentProviderIndex() < 0)
			return null;

		ILabelProvider labelProvider = getCurrentProvider();
		if(labelProvider instanceof ILabelAndDescriptionProvider)
			return ((ILabelAndDescriptionProvider)labelProvider).getDescription(element);
			
		return null;
	}

	/**
	 * @see org.eclipse.jface.viewers.ILabelProvider#getImage(java.lang.Object)
	 */
	public Image getImage(Object element)
	{
		if(getCurrentProviderIndex() < 0)
			return null;

		return getCurrentProvider().getImage(element);
	}

	/**
	 * @see org.eclipse.jface.viewers.ILabelProvider#getText(java.lang.Object)
	 */
	public String getText(Object element)
	{
		if(getCurrentProviderIndex() < 0)
			return null;

		return getCurrentProvider().getText(element);
	}

	/**
	 * @see org.eclipse.jface.viewers.IBaseLabelProvider#isLabelProperty(java.lang.Object, java.lang.String)
	 */
	public boolean isLabelProperty(Object element, String property)
	{
		if(getCurrentProviderIndex() < 0)
			return false;

		return getCurrentProvider().isLabelProperty(element, property);
	}
}
