/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ResourceChangeUpdaterProvider.java,v 1.6 2005/02/16 22:24:05 qiyanli Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.provider;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;

import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.util.IDisposable;

/**
 * @author marcelop
 * @since 
 */
public class ResourceChangeUpdaterProvider 
implements IResourceChangeListener, IDisposable
{
	public static class UIUpdaterProvider
	extends ResourceChangeUpdaterProvider implements Runnable
	{
		private IResourceDelta delta;
		
		/**
		 * @see java.lang.Runnable#run()
		 */
		public void run()
		{
			super.doProcessDelta(delta);
			delta = null;
		}

		/**
		 * @see org.eclipse.hyades.ui.internal.provider.ResourceChangeUpdaterProvider#processDelta(org.eclipse.core.resources.IResourceDelta)
		 */
		protected void doProcessDelta(IResourceDelta _delta)
		{
			this.delta = _delta;
			org.eclipse.swt.widgets.Display.getDefault().syncExec(this);
		}
	}
	
	private IResourceChangeUpdater resourceChangeUpdater;

	/**
	 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
	 */
	public void dispose()
	{
		resourceChangeUpdater = null;
	}
	
	/**
	 * Sets the resource change updater.
	 * @param resourceChangeUpdater
	 */
	public void setResourceChangeUpdater(IResourceChangeUpdater resourceChangeUpdater)
	{
		this.resourceChangeUpdater = resourceChangeUpdater;
	}
	
	/**
	 * Returns the resource change updater.
	 */
	public IResourceChangeUpdater getResourceChangeUpdater()
	{
		return resourceChangeUpdater;
	}

	/**
	 * @see org.eclipse.core.resources.IResourceChangeListener#resourceChanged(org.eclipse.core.resources.IResourceChangeEvent)
	 */
	public void resourceChanged(IResourceChangeEvent event)
	{
		if((getResourceChangeUpdater() == null) || (!getResourceChangeUpdater().isActive()))
			return;
			
		doProcessDelta(event.getDelta());
	}
	
	/**
	 * Utility method that allows subclasses to wrap the entire work of processing
	 * the resource delta and updater notification.
	 * @param delta
	 */
	protected void doProcessDelta(IResourceDelta delta)
	{
		getResourceChangeUpdater().started();
		try
		{
			processDelta(delta);
		}
		catch(Exception e)
		{
			HyadesUIPlugin.logError(e);
		}
		finally
		{
			getResourceChangeUpdater().ended();	
		}		
	}
	
	/**
	 * Process the resource delta for a resource change.
	 * @param delta
	 */
	protected void processDelta(IResourceDelta delta)
	{
		IResourceChangeUpdater resChangeUpdater = getResourceChangeUpdater();
		if((resChangeUpdater == null) || (!getResourceChangeUpdater().isActive()))
			return;
			
		// Get the affected resource
		IResource resource = delta.getResource();

		// If any children have changed type, just do a full refresh of this parent,
		// since a simple update on such children won't work, 
		// and trying to map the change to a remove and add is too dicey.
		// The case is: folder A renamed to existing file B, answering yes to overwrite B.
		IResourceDelta[] affectedChildren =	delta.getAffectedChildren(IResourceDelta.CHANGED);
		for (int i = 0; i < affectedChildren.length; i++)
		{
			int flags = affectedChildren[i].getFlags();
			if((flags & IResourceDelta.TYPE) != 0)
			{
				if(resChangeUpdater.updateChildrenType(resource))
					return;
			}
			else if((flags & IResourceDelta.CONTENT) != 0)
			{
				if(resChangeUpdater.updateContent(resource, affectedChildren[i].getResource()))
					return;
			}
		}

		// Check the flags for changes the Navigator cares about.
		// See ResourceLabelProvider for the aspects it cares about.
		// Notice we don't care about F_CONTENT or F_MARKERS currently.
		int changeFlags = delta.getFlags();
		if ((changeFlags & IResourceDelta.SYNC)	!= 0)
		{
			if(resChangeUpdater.updateProperties(resource))
				return;
		}
		
		//- if the change is open/close just refresh the resource
		if((changeFlags & IResourceDelta.OPEN) != 0) {
			resChangeUpdater.refreshContent(resource);
		}

		// Replacing a resource may affect its label and its children
		if ((changeFlags & IResourceDelta.REPLACED) != 0)
		{
			if(resChangeUpdater.replaced(resource))
				return;
		}

		// Handle changed children .
		for (int i = 0; i < affectedChildren.length; i++)
		{
			processDelta(affectedChildren[i]);
		}

		// Process removals before additions, to avoid multiple equal elements in the viewer.

		// Handle removed children. Issue one update for all removals.
		affectedChildren = delta.getAffectedChildren(IResourceDelta.REMOVED);
		if (affectedChildren.length > 0)
		{
			IResource[] affected = new IResource[affectedChildren.length];
			for (int i = 0; i < affectedChildren.length; i++)
				affected[i] = affectedChildren[i].getResource();
			
			if(resChangeUpdater.remove(resource, affected))
				return;				
		}

		// Handle added children. Issue one update for all insertions.
		affectedChildren = delta.getAffectedChildren(IResourceDelta.ADDED);
		if (affectedChildren.length > 0)
		{
			IResource[] affected = new IResource[affectedChildren.length];
			for (int i = 0; i < affectedChildren.length; i++)
				affected[i] = affectedChildren[i].getResource();
			if(resChangeUpdater.add(resource, affected))
				return;
		}
		
		// Handle added children .
		for (int i = 0; i < affectedChildren.length; i++)
		{
			processDelta(affectedChildren[i]);
		}
		
	}	
}
