/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: FileSystemStructureProvider.java,v 1.4 2005/02/16 22:24:05 qiyanli Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.wizard.exampleproject;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.ui.wizards.datatransfer.IImportStructureProvider;

/**
 * @author marcelop
 * @since 0.0.1
 */
public class FileSystemStructureProvider 
implements IImportStructureProvider
{
	private String pathPrefixToRemove;
	
	/**
	 * The final path will start after the pathPrefixToRemove.
	 * Examples
	 * 	path = "c:/temp/test/file.txt"
	 *  pathPrefixToRemove = "test"
	 *  final path = "file.txt"
	 */
	public FileSystemStructureProvider(String pathPrefixToRemove)
	{
		super();
		this.pathPrefixToRemove = pathPrefixToRemove;
	}
	
	/* (non-Javadoc)
	 * Method declared on IImportStructureProvider
	 */
	public List getChildren(Object element)
	{
		File folder = (File) element;
		String[] children = folder.list();
		int childrenLength = children == null ? 0 : children.length;
		List result = new ArrayList(childrenLength);

		for (int i = 0; i < childrenLength; i++)
		{
			File file = new File(folder, children[i]);
			String path = getFullPath(file);
			if("".equals(path))
				continue;
				
			result.add(new File(folder, children[i]));
		}

		return result;
	}
	
	/* (non-Javadoc)
	 * Method declared on IImportStructureProvider
	 */
	public InputStream getContents(Object element)
	{
		try
		{
			return new FileInputStream((File) element);
		}
		catch (FileNotFoundException e)
		{
			return null;
		}
	}
	
	/* (non-Javadoc)
	 * Method declared on IImportStructureProvider
	 */
	public String getFullPath(Object element)
	{
		File file = (File)element;
		String path = file.getPath();
		if(file.isDirectory() && (!path.endsWith(File.separator)))
			path = path + File.separator;
			
		return removePrefix(path);
	}
	
	/* (non-Javadoc)
	 * Method declared on IImportStructureProvider
	 */
	public String getLabel(Object element)
	{
		//Get the name - if it is empty then return the path as it is a file root
		File file = (File) element;
		
		String name = null;
		if(file.isDirectory())
			name = getFullPath(file);
	
		if("".equals(name))
			return "";
		
		name = file.getName();
		if(name.length() == 0)
		{
			return file.getPath();
		}
		else
		{			
			return name;
		}
	}
	
	/* (non-Javadoc)
	 * Method declared on IImportStructureProvider
	 */
	public boolean isFolder(Object element)
	{
		return ((File) element).isDirectory();
	}
	
	private String removePrefix(String path)
	{
		if(pathPrefixToRemove != null)
		{			
			int index = path.indexOf(pathPrefixToRemove);
			if(index >= 0)
				path = path.substring(index + pathPrefixToRemove.length());
		}

		return path;		
	}
}
