/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TraceLogger.java,v 1.4 2005/02/16 22:20:27 qiyanli Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.execution.trace;

import java.util.HashMap;

import org.eclipse.hyades.execution.trace.util.ProfilerImpl;
import org.eclipse.hyades.execution.trace.util.RecordClassDef;
import org.eclipse.hyades.execution.trace.util.RecordMethodAccess;
import org.eclipse.hyades.execution.trace.util.RecordMethodDef;
import org.eclipse.hyades.execution.trace.util.RecordObjAlloc;
import org.eclipse.hyades.execution.trace.util.RecordParameterDef;
import org.eclipse.hyades.execution.trace.util.RecordThreadStart;
import org.eclipse.hyades.execution.trace.util.RecordVariable;
import org.eclipse.hyades.execution.trace.util.Utilities;


/**
 * This is the logger for distributed tracing.
 * 
 * @author Richard Duggan, Qiyan Li
 */
public class TraceLogger extends ProfilerImpl {

    /**
     * the name of the distributed trace agent
     */
    private static final String DISTRIBUTED_TRACE_PROFILER = "Distributed Trace Profiler";
    
    /**
     * the type of the distributed trace agent
     */
    private static final String TYPE_PROFILER = "Profiler";
    
    /**
     * the singleton instance of the profiler
     */
    private static TraceLogger _profiler = null;

    /**
     * the data structure holding the collection of method defitions
     */
    private HashMap methodCollection;

    /**
     * the data structure holding the collection of class defitions
     */
    private HashMap classCollection;

    /**
     * the data structure holding the collection of object definitions
     */
    private HashMap objectCollection;

    /**
     * the data structure holding the collection of parameter definitions
     */
    private HashMap parameterCollection;

    /**
     * the data structure holding the collection of thread definitions
     */
    private HashMap threadCollection;


    /**
     * Creates an instance of the logger.
     */
    private TraceLogger() {

        super(DISTRIBUTED_TRACE_PROFILER, TYPE_PROFILER);
        methodCollection = new HashMap();
        classCollection = new HashMap();
        objectCollection = new HashMap();
        parameterCollection = new HashMap();
        threadCollection = new HashMap();
    }


    /**
     * Get the singleton instance of this profiler.
     */
    public static TraceLogger getInstance() {

        synchronized (TraceLogger.class) {
            if (_profiler == null) {
                _profiler = new TraceLogger();
            }
        }
        return _profiler;
    }


    /**
     * Invoked by the logging agent mechanism to indicate a remote client is discontinuing
     * data collection.
     */
    public void monitorInactive() {

        super.monitorInactive();

        /* Clear up the state from the previous trace. */
        classCollection.clear();
        methodCollection.clear();
        objectCollection.clear();
        parameterCollection.clear();
        threadCollection.clear();
    }


    /**
     * Logs a particular event using the information provided in <code>self</code>
     * and <code>partner</code>, which may or may not be running on a remote entity.
     * 
     * @param eventType the type of event
     * @param self      the event to be logged
     * @param partner   the parent of the self event
     */
    public void logCorrelator(String eventType, TraceCorrelator self, TraceCorrelator partner) {

        /* Saves the thread definition if necessary. */
        Thread threadKey = Thread.currentThread();
        RecordThreadStart threadStart = (RecordThreadStart) threadCollection.get(threadKey);
        if (threadStart == null) {
            threadStart = new RecordThreadStart();
            threadStart.setTime(Utilities.getCurrentTimeStamp());
            threadCollection.put(threadKey, threadStart);
            write(threadStart.toString());
        }
        int threadIdRef = threadStart.getThreadId();

        /* Saves the class definition if necessary. */
        String className = self.getClazz().getName();
        RecordClassDef classDef = (RecordClassDef) classCollection.get(className);
        if (classDef == null) {
            classDef = new RecordClassDef(className);
            classDef.setTime(Utilities.getCurrentTimeStamp());
            classDef.setThreadIdRef(threadIdRef);
            classCollection.put(className, classDef);
            write(classDef.toString());
        }

        /* Saves the object definition if necessary. */
        int objRef = self.getObject();
        RecordObjAlloc objAlloc = (RecordObjAlloc) objectCollection.get(new Integer(objRef));
        if (objAlloc == null) {
            objAlloc = new RecordObjAlloc();
            objAlloc.setTime(Utilities.getCurrentTimeStamp());
            objAlloc.setClassIdRef(classDef.getClassId());
            objAlloc.setThreadIdRef(threadIdRef);
            objectCollection.put(new Integer(objRef), objAlloc);
            write(objAlloc.toString());
        }

        /* Saves the method definition if necessary. */
        String methodName = self.getInvokedMethod().toString();
        String methodSignature = Utilities.getInvokedMethodSignature(self.getParmClasses(),
            self.getReturnClass() == null ? null : Utilities.getJniNotation(self.getReturnClass()));
        String methodKey = className + "#" + methodName + methodSignature;
        RecordMethodDef methodDef = (RecordMethodDef) methodCollection.get(methodKey);
        if (methodDef == null) {
            methodDef = new RecordMethodDef(methodName, methodSignature, classDef.getClassId());
            methodCollection.put(methodKey, methodDef);
            write(methodDef.toString());
        }

        /* If any of the three input parameters is null, or if the parameter lists are of different lengths,
           the parameter list is assumed to be empty. */
        String[] parameterNames = self.getParmNames();
        Class[] parameterClasses = self.getParmClasses();
        Object[] parameterValues = self.getParmValues();
        int listLength = (parameterNames == null || parameterClasses == null || parameterValues == null ||
            parameterNames.length != parameterClasses.length ||
            parameterClasses.length != parameterValues.length ? 0 : parameterNames.length);

        /* Saves the parameter definitions if necessary. */
        int[] parameterIds = new int[listLength];
        for (int i = 0; i < listLength; i++) {
            String parameterKey = Integer.toString(methodDef.getMethodId()) + "@" + Integer.toString(i);
            RecordParameterDef parameterDef = (RecordParameterDef) parameterCollection.get(parameterKey);
            if (parameterDef == null) {
                parameterDef = new RecordParameterDef(parameterClasses[i], methodDef.getMethodId(), i);
                parameterDef.setName(parameterNames[i]);
                parameterCollection.put(parameterKey, parameterDef);
                write(parameterDef.toString());
            }
            parameterIds[i] = parameterDef.getParameterId();
        }

        /* Saves the event itself. */
        RecordMethodAccess methodAccess =
            new RecordMethodAccess(eventType, methodDef.getMethodId(), objAlloc.getObjId());
        methodAccess.setTime(Utilities.getCurrentTimeStamp());
        methodAccess.setSequenceCounter(self.getOperationCounter());
        methodAccess.setTicket(Long.toString(self.getApplicationCounter()));
        methodAccess.setParameterList(parameterIds, parameterClasses, parameterValues);
        methodAccess.setThreadIdRef(threadIdRef);

        /* Set the location information for the local correlator. */
        self.setAgentIdRef(getAgentId());
        self.setProcessIdRef(getProcessId());
        self.setNodeIdRef(getNodeId());
        self.setThreadIdRef(threadIdRef);

        /* Saves the partner information if the invocation is remote. */
        if (partner != null && partner.getAgentIdRef() != null
            && !self.getAgentIdRef().equals(partner.getAgentIdRef())
            && (eventType == RecordMethodAccess.METHOD_RECEIVE ||
            eventType == RecordMethodAccess.METHOD_RETURN)) {
            methodAccess.setRemoteContext(partner.getNodeIdRef(), partner.getProcessIdRef(),
                partner.getAgentIdRef(), partner.getThreadIdRef(), partner.getApplicationCounter(),
                partner.getOperationCounter());
        }


        /* Saves the return value if the event is an exit event, and the return type is not null. */
        if (eventType == RecordMethodAccess.METHOD_EXIT && self.getReturnClass() != null
            && self.getReturnClass() != void.class) {
            methodAccess.setReturnValue(
                new RecordVariable(RecordVariable.RETURN_VALUE, 0, self.getReturnClass(), self.getReturnValue()));
        }
        write(methodAccess.toString());
    }
}
