/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: RecorderPlugin.java,v 1.9 2005/04/22 14:52:43 bjiang Exp $
 * 
 * 
 * Contributors: 
 * IBM Corporation - initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.execution.recorder;

import java.net.URL;
import java.util.ResourceBundle;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.hyades.test.ui.TestUI;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.ui.internal.util.StringUtil;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.plugin.AbstractUIPlugin;


/**
 * This plugin contains the Hyades infrastructure for trace recording.  It contains all the extendible object and interfaces
 * necessary to deploy a recorder on the Agent Controller 
 * @author Ernest Jessee
 */
public class RecorderPlugin extends AbstractUIPlugin
{
	
	/**
	 * the default instance 
	 */
	private static RecorderPlugin plugin;
	
	/**
	 * the plugin's ID as shown in plugin.xml
	 */
	public static final String ID = "org.eclipse.hyades.execution.recorder"; //$NON-NLS-1$
	private ResourceBundle resourceBundle;
	
	/**
	 * Stores a preference in the plugins preference store with the specified key and value 
	 * @param String key
	 * @param String value
	 */
	public static void storePreference(String key, String value)
	{
		getDefault().getPreferenceStore().setValue(key,value);
	}
	
	/**
	 * Returns the plugin's resource bundle,
	 */
	private ResourceBundle getResourceBundle()
	{
		if(resourceBundle == null)
		{
		
			try
			{
				resourceBundle = Platform.getResourceBundle(Platform.getBundle(ID));
			}
			catch (Exception e) {
				resourceBundle = null;
			}
		}
		
		return resourceBundle;
	}
	
	/**
	 * acquires a reference to a plugin preference specified by "key"
	 * @param String key
	 * @return String
	 */
	public static String getPreference(String key)
	{
		return getDefault().getPreferenceStore().getString(key);
	}	
	/**
	 * returns the plugins install URL
	 * @return URL
	 */
	public URL getInstallURL()
	{
		return getBundle().getEntry("/");
	}
	
	/**
	 * returns a resource String from this plugin's bundle
	 * @param String key
	 * @return String
	 */
	public static String getResourceString(String key)	
	{
		return plugin.getResourceBundle().getString(key);
	}
	
	
	/**
	 * The constructor.
	 */
	public RecorderPlugin() 
	{
		super();
		plugin = this;
				
	}

	/**
	 * Returns the default instance.
	 */
	public static RecorderPlugin getDefault() 
	{
		return plugin;
	}
	
	
	/**
	 * return the port number specified for the Agent Controller (RAC) as used by Hyades Test.  
	 * The value returned by this method is specified using Window->Preferences->Test
	 * @return String
	 */
	public String getRAServerPort()
	{
		return TestUIPlugin.getInstance().getPreferenceStore().getString(TestUI.LOCALHOST_PORT);
	}
	
	/**
	 * returns the host specified for the Agent Controller as used by Hyades Test
	 * The value returend by this method is specified using Window->Preferences->Test
	 * @return String
	 */
	public String getRAServerHost()
	{
		return TestUIPlugin.getInstance().getPreferenceStore().getString(TestUI.DEFAULT_HOST_NAME);
	}
	
	/**
	 * This method causes a dialog box to be displayed warning the user of an exception.
	 * @param Exception e1 - the exception thrown 
	 * @param consequenceDescription - a description of the consequence (i.e. Recording session aborted)
	 * @param cause - a description of the cause. (i.e. Another recorder is active or Agent Controller not available on host/port)
	 * @param dlgTitle - the Title to be displayed on the error dialog
	 */
	public static void reportExceptionToUser(final Exception e1,
										     final String consequenceDescription,
										     final String cause,
										     final String dlgTitle)
	{
		Display.getDefault().asyncExec
		(
			new Runnable()
			{
				public void run()
				{
					Status errorStatus = new Status(IStatus.ERROR,RecorderPlugin.ID,IStatus.ERROR,cause,e1);
					new ErrorDialog(Display.getDefault().getActiveShell(),dlgTitle,consequenceDescription,errorStatus,IStatus.ERROR).open();
				
	
				}
	
			}
		);
	}

	public String getString(String key, String[]args)	
	{
		
		String ret = RecorderPlugin.getResourceString(key);
		for (int i = args.length; i > 0; i--)
			ret = StringUtil.replace(ret, ("%"+i), args[i-1]);
		
		return ret;
	}	

}
