/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: Recorder.java,v 1.7 2005/04/20 16:31:53 jnevicos Exp $
 * 
 * Contributors: 
 * IBM Corporation - initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.internal.execution.recorder;

import org.eclipse.hyades.internal.execution.recorder.local.IRecorderDataProcessor;
import org.eclipse.hyades.internal.execution.recorder.local.RecorderClient;
import org.eclipse.hyades.internal.execution.recorder.remote.RecorderEnvironmentAdapter;
import org.eclipse.hyades.internal.execution.recorder.remote.RecorderExecutableObjectAdapter;

/**
 * Contains all information and/or instantiated objects related to a recorder
 * This object is managed by RecorderFactory and is automatically registred with the factory upon intial creation.
 * Attempting to instantiate an object with a given ID more than once will result in an exception
 * @author Ernest Jessee
 */
public class Recorder
{	
	/**
	 * the recorder's ID.
	 */
	protected String id = null;

	/**
	 * environment Adapter which may append the execution environment's classpath, etc.
	 */
	protected RecorderEnvironmentAdapter environmentAdapter= null;

	/**
	 * the IRecorderDataProcessor which handles incoming data from the Agent Controller
	 * @see IRecorderDataProcessor
	 */
	protected IRecorderDataProcessor dataProcessor= null;
	
	
	/**
	 * This string holds the configuration parameters which will be passed to the recorder's agent object.  This string
	 * should be a semi-colon delimited collection of comma delimited name-value pairs (i.e. host,localhost;port,450)
	 */
	private String configParams = null;
	
	/**
	 * the adapter for the recorder's agent object
	 * @see RecorderExecutableObjectAdapter
	 */
	private RecorderExecutableObjectAdapter executableObjectAdapter = null;
	
	/**
	 * the ID of the application adapter that will be used with this recorder. (set at runtime)
	 */
	private String applicationAdapterID=null;

	
	/**
	 * The ID of the scriptgenerator to be used with this recorder (set at runtime)
	 */
	private String scriptGeneratorID = null;

	/**
	 * The path for the resulting trace file (set at runtime)
	 */
	private String recordingPath=null;;
	
	
	/**
	 * The path to the test to be generated after this recorder completes (set at runtime)
	 */
	private String testPath=null;

	
	/**
	 * The RecorderClient object which will be used to launch this recorder and which may be used to send control messages to the recorder
	 * agent object during recording.
	 * @see RecorderClient
	 */
	private RecorderClient client = null;
	
	
	/**
	 * indicates whether or not this recorder is "active"
	 */
	private boolean active=false;

	/**
	 * This contructor should only be called one time per recorder per eclipse session.  Upon construction, the recorder is automatically registered
	 * with the RecorderFactory.  Subsequent calls to this contructor will result in a RecorderException.
	 * 
	 * @param String recorderID
	 * @param IRecorderDataProcessor dataProcessor
	 * @param RecorderEnvironmentAdapter agentEnvironmentAdapter
	 * @param RecorderExecutableObjectAdapter objectAdapter	 
	 * @throws RecorderException
	 */
	public Recorder(String recorderID, 
					IRecorderDataProcessor dataProcessor, 
					RecorderEnvironmentAdapter agentEnvironmentAdapter,
					RecorderExecutableObjectAdapter objectAdapter)throws RecorderException
	{
		
		//check to see if this recorder has already been registred with the factory.
		if(RecorderFactory.getInstance().getRecorderWithID(recorderID)!=null)
			throw new RecorderException("recorder with ID \""+recorderID+"\" already exists");
		
		
		this.id = recorderID;
		this.dataProcessor = dataProcessor;
		this.environmentAdapter = agentEnvironmentAdapter;
		this.executableObjectAdapter = objectAdapter;
		RecorderFactory.getInstance().addRecorder(this);
		
	}
	
	

	/**
	 * returns the recorders environmen adapter
	 * @see RecorderEnvironmentAdapter
	 * @return
	 */	
	public RecorderEnvironmentAdapter getAgentEnvironmentAdapter()
	{
		return environmentAdapter;
	}

	/**
	 * returns the recorders IRecorderDataProcessor
	 * @see IRecorderDataProcessor
	 */	
	public IRecorderDataProcessor getDataProcessor()
	{
		return dataProcessor;
	}

		
	
	/**
	 * returns the recorders ID. recorder ID's must be unique.
	 */
	public String getId()
	{
		return id;
	}
	
	/**
	 * called to initiate a recording with this recording.  Before this method is called, all valid information for this 
	 * recording session must be set.  For example.
	 * 		recorder.setConfigParams("port,450");
	 * 		recorder.setRecordingPath("/test/recordings/foo.rec");
	 * 		recorder.setScriptgenID("org.eclipse.hyades.execution.testgen.TestGenGenerator.HTTPGenerator");
	 * 		recorder.setTestPath("/test/tests/bar.testsuite");
	 * 		recorder.setApplicationAdapterID("org.eclipse.hyades.execution.recorder.RecorderApplication.InternetExplorer");
	 *  	recorder.Record();
	 * @throws RecorderException
	 */
	public void Record()throws RecorderException
	{		
		try
		{
			setActive(true);
			client = new RecorderClient();
			client.run(this);
		}
		catch (Exception e)
		{
			throw new RecorderException(e.getMessage());
		}
		
	}
	

	/**
	 * gets the recorders configuration parameter string
	 */
	public String getConfigParams()
	{
		return configParams;
	}

	/**
	 * Sets the recoders configuration parameter String which will be passed to the recorder's agent object.  This string
	 * should be a semi-colon delimited collection of comma delimited name-value pairs (i.e. host,localhost;port,450)
	 */
	public void setConfigParams(String string)
	{
		configParams = string;
	}

	/**
	 * returns a reference to the recorders RecorderExecutableObjectAdapter
	 * @return RecorderExecutableObjectAdapter
	 */
	public RecorderExecutableObjectAdapter getExecutableObjectAdapter()
	{
		return executableObjectAdapter;
	}
	
	/**
	 * returns the ID of the recorder application adapter to be used with this recorder.
	 */ 
	public String getApplicationAdapterID()
	{
		return applicationAdapterID;
	}

	/**
	 * sets the ID of the application adapter that will be used with this recorder. (set at runtime)
	 * @param String appAdapterID
	 * 
	 */
	public void setApplicationAdapterID(String appAdapterID)
	{
		applicationAdapterID = appAdapterID;
	}

	/**
	 * returns the path for the resulting trace file (set at runtime)
	 */
	public String getRecordingPath()
	{
		return recordingPath;
	}

	/**
	 * returns the ID of the registred script generator which is to be used with this recorder for the current recording session.
	 * This parameter is set per recording at runtime.
	 * @return
	 */
	public String getScriptgenID()
	{
		return scriptGeneratorID;
	}

	/**
	 * sets the ID of the script generator to be used for this recording session.
	 * @param String scriptGenID
	 */
	public void setScriptgenID(String scriptGenID)
	{
		scriptGeneratorID = scriptGenID;
	}

	/**
	 * sets the path for the trace file generated by the current recording session.  
	 * this value is set per recording at runtime.
	 * @param Strig recPath
	 */
	public void setRecordingPath(String recPath)
	{
		recordingPath = recPath;
	}

	/**
	 * returns the path for the test which will be generated at the conclusion of this recording 
	 * @return String
	 */
	public String getTestPath()
	{
		return testPath;
	}

	/**
	 *sets teh path for the test which will be generated at the conclusion of the current recording session.
	 *this value is set per recording session at runtime
	 * @param string
	 */
	public void setTestPath(String string)
	{
		testPath = string;
	}

	/**
	 * called to stop a recorder.  Calling this method will result in termination of the recording and closing of any application that
	 * was started by the recorder framework.
	 */
	public void stopRecorder()
	{
		client.stopRecording();
	}

	
	/**
	 * returns true if this recorder is currently active
	 * @return boolean
	 */
	public boolean isActive()
	{
		return active;
	}

	/**
	 * called to modify the active status of this recorder
	 * @param boolean b;
	 */
	public void setActive(boolean isActive)
	{
		active = isActive;
		RecorderFactory.getInstance().updateActiveState(this, isActive);
	}

	public RecorderClient getClient()
	{
		return client;
	}
}

