/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: RecorderAppAdapterFactory.java,v 1.10 2005/05/11 05:48:01 jnevicos Exp $
 * 
 * Contributors: 
 * IBM Corporation - initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.internal.execution.recorder.local.appadapters;

import java.util.HashMap;
import java.util.StringTokenizer;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.Platform;

/**
 * Factory which manages Recorder Application Adapters which are comprised of multiple objects which work together 
 * to communicate and function using the Agent Controller
 * @author Ernest Jessee
 */
public class RecorderAppAdapterFactory
{
	/**
	 * ID of the extension point associated with this factory.  At first call, the factory will query the system metadata and 
	 * load all extension point objects with this ID.
	 */
	private String extensionPointID = "org.eclipse.hyades.execution.recorder.RecorderApplication"; //$NON-NLS-1$
	
	/**
	 * the default intstance of this factory	
	 */
	private static RecorderAppAdapterFactory instance = null;
	
	/**
	 * Hashmap for referencing adapters by id.
	 */
	private HashMap elementsByID = new HashMap();
	
	/**
	 * Hashmap for referencing adapter clients (IRecorderApplicationAdapterClient) by ID
	 */
	private HashMap clients = new HashMap();
		
		
	/**
	 * private constructor (singleton)
	 */
	private RecorderAppAdapterFactory(){}
	
	/**
	 * returns default instance of the RecorderAppAdapterFactory
	 * @return RecorderAppAdapterFactory
	 * 
	 */
	public static RecorderAppAdapterFactory getInstance()
	{
		if(instance == null)
		{
			instance = new RecorderAppAdapterFactory();
			instance.loadMetdata();
			
		}
		
		return instance;
	}
	
	
	/**
	 * called at initial instantiation to load the xml data from various plugin.xml files.
	 */
	private void loadMetdata()
	{
		IConfigurationElement[] adapterElements = Platform.getExtensionRegistry().getConfigurationElementsFor(extensionPointID);
		for(int i=0; i<adapterElements.length; i++)
		{
			IConfigurationElement element = adapterElements[i];
			elementsByID.put(element.getAttribute("id"),element); //$NON-NLS-1$
		}
		
	}
	
	/**
	 * returns the name of the registered adapter with the specified ID
	 * @param String adapterID
	 * @return String
	 */
	public String getAdapterName(String adapterID)
	{
		return ((IConfigurationElement)elementsByID.get(adapterID)).getAttribute("name"); //$NON-NLS-1$
	}
	
	/**
	 * returns the full path of the agent object (IRecorderApplicationAdapter) for the Recorder Application Adapter with the specified ID
	 */
	public String getAgentClasspath(String adapterID)
	{
		if (adapterID.equalsIgnoreCase("-1"))
			return adapterID;
		else
			return ((IConfigurationElement)elementsByID.get(adapterID)).getAttribute("agentclass"); //$NON-NLS-1$
	}
	
	/**
	 * returns a String array with supported operating systems, or null if none specified
	 */
	public String[] getSupportedOS(String adapterID)
	{
		if (adapterID.equalsIgnoreCase("-1"))
			return null;
		else
		{
			String oses = ((IConfigurationElement)elementsByID.get(adapterID)).getAttribute("supportedOS"); //$NON-NLS-1$
			if (oses == null)
				return null;
			StringTokenizer tokn = new StringTokenizer(oses, ",");
			int nTokn = tokn.countTokens();
			String[] osArray = new String[nTokn];
			for (int i = 0; i < nTokn; i++)
				osArray[i] = tokn.nextToken();
			
			return osArray;
		}
	}
	
	/**
	 * returns the client object for the adapter with the specified ID
	 * @param String adapterID
	 * @return IRecorderApplicationAdapter
	 */
	public IRecorderApplicationAdapterClient getClient(String adapterID) throws InstantiationException, IllegalAccessException, ClassNotFoundException, CoreException
	{
		Object obj = clients.get(adapterID);
		if(obj!=null)
			return (IRecorderApplicationAdapterClient)obj;
		else
		{		
			if (adapterID.equalsIgnoreCase("-1"))
				return null;
						
				IConfigurationElement appAdapter = ((IConfigurationElement)elementsByID.get(adapterID));
				IRecorderApplicationAdapterClient client = (IRecorderApplicationAdapterClient) appAdapter.createExecutableExtension("clientclass");			
				clients.put(adapterID,client);
				return client;

			
		}
	}
	
	
	/**
	 * returns a String Array of id's corresponding to all registered application adapters
	 * @return String[]
	 */
	public String[] getAdapterIDs()
	{
		Object[] array = elementsByID.keySet().toArray();
		String retArray[] = new String[array.length];
		for(int i=0; i<array.length; i++)
		{
			retArray[i]=(String)array[i];
		}
		return retArray;
	}
	
	
	/**
	 * returns an application id by index
	 * @param int index
	 * @return String 
	 */
	public String getAdapterIDByIndex(int index)
	{
		Object[] values = elementsByID.values().toArray();
		
		if(index>values.length-1)
			return null;
		
		IConfigurationElement element = (IConfigurationElement)values[index];
			
		return element.getAttribute("id"); //$NON-NLS-1$
	}
	
	/**
	 * returns the IConfigurationElement for an appliation adapter with the specified ID
	 * @param String ID
	 * @return IConfigurationElement
	 */
	public IConfigurationElement getAdapterConfigElement(String ID)
	{
		return (IConfigurationElement)elementsByID.get(ID);
	}
	
	
	

}
