/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TestSuiteEditorExtension.java,v 1.5 2005/02/16 22:22:05 qiyanli Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.editor.extension;

import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.test.ui.editor.form.DefaultTestContextOverviewContribution;
import org.eclipse.hyades.test.ui.editor.form.base.IHasEStructuralFeature;
import org.eclipse.hyades.test.ui.editor.form.TestCasesForm;
import org.eclipse.hyades.test.ui.editor.form.TestContextOverview;
import org.eclipse.hyades.test.ui.editor.form.util.EditorForm;
import org.eclipse.hyades.test.ui.editor.form.util.WidgetFactory;
import org.eclipse.hyades.ui.editor.IHyadesEditorPart;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;

/**
 * Hyades default test suite editor extension that is suitable for all the
 * test suite objects.
 * 
 * <p>This class is used whenever it is not possible neither to identify nor
 * to locate the appropriate extension editor.
 * <p>This class might be extended by any user who wants to create his/her own 
 * test context editor.
 * A user might want to override this class. In that case, a simple way is to use a code like the following one :
 * <pre>
 *	public class MyTestSuiteEditor extends TestSuiteEditorExtension {
 *		
 *		public MyTestSuiteEditor() {
 *			super();
 *			
 *			MyTestCasesForm testCasesForm = new MyTestCasesForm(this, getWidgetFactory());		
 *			MyTestComponentsForm compForm = new MyTestComponentsForm(this, getWidgetFactory());
 *			
 *			TestContextOverviewContribution contributor = new MyOverviewContributor();
 *			contributor.setTestCasesForm(testCasesForm);
 *			contributor.setStubsForm(compForm);
 *			TestContextOverview overview = new TestContextOverview(this, getWidgetFactory(), contributor);
 *			
 *	
 *			EditorForm[] forms = {
 *				overview,
 *				testCasesForm,
 *				compForm
 *			};
 *			setForms(forms);
 *		}
 *		
 *	}
 * </pre>
 * This codes creates a test context editor which contains three forms : an overview, a test cases form and a test components form.
 * The overview might be customized thanks to a contributor (@see org.eclipse.hyades.test.ui.editor.form.TestContextOverviewContribution).
 * In this example, the two other forms are user-defined.
 * 
 * @author marcelop, bcormier
 * @since 3.0.0
 */
public class TestSuiteEditorExtension 
extends BaseEditorExtension
{
	public static final String EXTENSION_ID = "org.eclipse.hyades.test.ui.editor.extension.TestSuite"; //$NON-NLS-1$
	
	private WidgetFactory widgetFactory = new WidgetFactory();
	private EditorForm[] forms = {
					new TestContextOverview(this, widgetFactory, new DefaultTestContextOverviewContribution()),
					new TestCasesForm(this, widgetFactory)
			};
	
	/**
	 * Disposes any resource used by this object. There are no guarantees that 
	 * this method is invoked by this object's container or by the Hyades framework.
	 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
	 */
	public void dispose()
	{
		for (int i=0; forms!=null && i<forms.length; i++) {
			if (forms[i]!= null) {
				forms[i].dispose();
				forms[i] = null;
			}
		}
		super.dispose();
	}

	/**
	 * Returns the test suite that is manipulated by this editor
	 * extension.
	 * @return TPFTestSuite 
	 */
	public TPFTestSuite getTestSuite()
	{
		return (TPFTestSuite)getHyadesEditorPart().getEditorObject();
	}
	
	/**
	 * Creates the pages of this test context editor extension.  Each page should be added 
	 * to the editor using one of the <code>addPage(...)</code> methods.
	 * @see org.eclipse.hyades.ui.editor.IEditorExtension#createPages()
	 */
	final public void createPages()
	{
		IHyadesEditorPart hyadesEditorPart = getHyadesEditorPart();
		
		for (int i=0; forms!=null && i<forms.length; i++) {
			hyadesEditorPart.addPage(forms[i].createControl());
			hyadesEditorPart.setPageText(i, forms[i].getHeadingText()); 
		}
			
	}
		
	/**
	 * Refreshes this editor extension.
	 * @param data the data that might be used to refresh this editor extension
	 * @see org.eclipse.hyades.ui.util.IRefreshable#refreshContent(java.lang.Object)
	 */
	public void refreshContent(Object data)
	{
		for (int i=0; forms!=null && i<forms.length; i++) {
			if (forms[i]!= null) {
				forms[i].load();
			}
		}
	}

	/**
	 * Returns the current selection for this editor extension.
	 * @return the current selection
	 * @see org.eclipse.hyades.ui.editor.IEditorExtension#getSelection()
	 */
	public IStructuredSelection getSelection()
	{
		IStructuredSelection selection = null;
		int index = getHyadesEditorPart().getActivePage();
		if (index>-1 && forms!=null && index<forms.length)
			selection = (IStructuredSelection)forms[index].getCurrentSelection();
		if (selection == null)
		    selection = StructuredSelection.EMPTY;
		return selection;
	}
		
	/**
	 * Notifies this editor extension that the page with the given index has been
	 * activated. This method is called when the user selects a different tab.
	 * @param index
	 * @return boolean indicating that the Hyades Editor Part selection listeners 
	 * should be notified.
	 * @see org.eclipse.hyades.ui.editor.IEditorExtension#pageActivated(int)
	 */
	public boolean pageActivated(int index)
	{
		if (index>-1 && forms!=null && index<forms.length)
			return forms[index].activated();
		return false;
	}

	/**
	 * Sets the current selection for this selection provider.
	 * @param selection the new selection
	 * @see org.eclipse.hyades.ui.editor.IEditorExtension#setSelection(org.eclipse.jface.viewers.IStructuredSelection)
	 */
	public void setSelection(IStructuredSelection structuredSelection)
	{
		if(structuredSelection.size() == 1)
		{
			Object object = structuredSelection.getFirstElement();
			if(object == getTestSuite())
			{
				// don't do anything.
			}

			boolean notFound = true;
			for (int i=0; notFound && forms!=null && i<forms.length; i++) {
			    if (forms[i] instanceof IHasEStructuralFeature) {
					Class cl = ((IHasEStructuralFeature)forms[i]).getStructuralFeature().getEType().getInstanceClass();
					if (cl.isAssignableFrom(object.getClass())) {
						notFound = false;
						forms[i].selectReveal(structuredSelection);
						getHyadesEditorPart().setActivePage(i);
					}
			    }
			}
		}
	}

	/**
	 * @return The forms which are part of this test context editor.
	 */
	public EditorForm[] getForms() {
		return forms;
	}

	/**
	 * @return The widget factory which is being used  within this test context editor.
	 */
	public WidgetFactory getWidgetFactory() {
		return widgetFactory;
	}
	
	/**
	 * This method registers the forms which belong to the test context editor.
	 * When overriding this test context editor, the user should call this method in the constructor.
	 * @param forms The array of forms.
	 */
	public void setForms(EditorForm[] forms) {
		this.forms = forms;
	}


}