/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ExtensibleEObjectTreeSection.java,v 1.3 2005/02/16 22:22:05 qiyanli Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.editor.form.util;

import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.hyades.test.ui.adapter.TestWorkbenchAdapter;
import org.eclipse.hyades.test.ui.editor.form.util.EObjectContainerContentProvider;
import org.eclipse.hyades.test.ui.editor.form.util.EditorForm;
import org.eclipse.hyades.test.ui.internal.editor.form.util.TreeSection;
import org.eclipse.hyades.ui.internal.action.CollapseAllAction;
import org.eclipse.hyades.ui.internal.provider.WorkbenchAdapterLabelProvider;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.actions.SelectionListenerAction;
import org.eclipse.ui.part.DrillDownAdapter;

/**
 * This class provides a EObject tree view section with configurable buttons.
 * The sole constructor of this class takes all button actions in an array
 * which will be used to created the buttons and will be invoked when the button is selected.
 * 
 * @author bjiang
 * @since	1.3
 */
public class ExtensibleEObjectTreeSection extends TreeSection {
	
	private IAction[] buttonActions;
	private EStructuralFeature eStructuralFeature;
	private IContentProvider treeContentProvider;
	private ILabelProvider treeLabelProvider;
	protected DrillDownAdapter drillDownAdapter;
	protected CollapseAllAction collapseAllAction;
	private boolean editable;

	/**
	 * Constructor for ExtendedEObjectTreeSection
	 * @param editorForm the form which this section belongs to.
	 * @param eStructuralFeature the feature of of the editing EObject that this tree section displays/edits.
	 * @param theButtonActions actions of all buttons provided by this section. The buttons will be created based off these actions.
	 */
	public ExtensibleEObjectTreeSection(EditorForm editorForm, EStructuralFeature eStructuralFeature, IAction[] buttonActions)
	{
		super(editorForm);
		this.eStructuralFeature = eStructuralFeature;
		this.buttonActions = buttonActions;
		String[] buttonLabels = new String[buttonActions.length];
		for(int i = 0; i < buttonActions.length; i++)
		{
			buttonLabels[i] = buttonActions[i].getText();
		}

		super.setButtonLabels(buttonLabels);		
	}
	
	/**
	 * @see org.eclipse.update.ui.forms.internal.FormSection#dispose()
	 */
	public void dispose()
	{
		eStructuralFeature = null;
		
		for(int i = 0; i < buttonActions.length; i++)
		{
			if(buttonActions[i] != null)
				buttonActions[i] = null;
		}

		super.dispose();
	}	
	
	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.base.BaseTreeSection#adjustClient(org.eclipse.swt.widgets.Composite)
	 */
	protected void adjustClient(Composite parent)
	{
		getTreeViewer().setContentProvider(getContentProvider());
		getTreeViewer().setLabelProvider(getLabelProvider());
		getTreeViewer().setAutoExpandLevel(0);
		
		for(int i = 0; i < buttonActions.length; i++)
		{
			if(buttonActions[i] != null)
			{
				getTreePart().setButtonEnabled(i, buttonActions[i].isEnabled());
			}
		}
	}
	
	/**
	 * Sets the content provider of the tree viewer of this section.
	 * 
	 * @param contentProvider
	 */
	public void setContentProvider(IContentProvider contentProvider)
	{
	    this.treeContentProvider = contentProvider;
	}
	
	/**
	 * Returns this section's content provider. Default provider used is
	 * {@link org.eclipse.hyades.test.ui.editor.form.util.EObjectContainerContentProvider}
	 * if none is set so far.
	 * 
	 * @return IContentProvider
	 */
	public IContentProvider getContentProvider()
	{
	    if(this.treeContentProvider == null)
	        treeContentProvider = new EObjectContainerContentProvider(getEditorForm(), getEStructuralFeature());
	    
	    return this.treeContentProvider;
	}
	
	/**
	 * Sets a label provider for the tree viewer of this section.
	 * 
	 * @param labelProvider
	 */
	public void setLabelProvider(ILabelProvider labelProvider)
	{
	    this.treeLabelProvider = labelProvider;
	}

	/**
	 * Returns this section's label provider. Default provider used is
	 * {@link org.eclipse.hyades.ui.internal.provider.WorkbenchAdapterLabelProvider}
	 * if none is set so far.
	 * 
	 * @return ILabelProvider
	 */
	public ILabelProvider getLabelProvider()
	{
	    if(this.treeLabelProvider == null)
			treeLabelProvider = new WorkbenchAdapterLabelProvider(TestWorkbenchAdapter.class);
	    
	    return treeLabelProvider;
	}

	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.base.BaseStructuredViewerSection#createActions()
	 */
	protected void createActions()
	{
		collapseAllAction = new CollapseAllAction(getTreeViewer()).useDefaultDecorators();
		
		drillDownAdapter = new DrillDownAdapter(getTreeViewer());
		getTreeViewer().addSelectionChangedListener(drillDownAdapter);
	}
	
	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.base.BaseStructuredViewerSection#buttonSelected(int)
	 */
	protected void buttonSelected(int index)
	{
		if(!isReadOnly())
		{
			if( index >=0 && index < buttonActions.length )
				buttonActions[index].run();
		}
	
		updateActionsAndButtons(getStructuredSelection());
	}
	
	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.base.BaseStructuredViewerSection#updateActionsAndButtons(org.eclipse.jface.viewers.IStructuredSelection)
	 */
	protected void updateActionsAndButtons(IStructuredSelection structuredSelection)
	{
		if(buttonActions == null || buttonActions.length < 1)
			return;
			
		if(isEditable())
		{
			for(int i = 0; i < buttonActions.length; i++)
			{
				if(buttonActions[i] != null)
				{
					if(buttonActions[i] instanceof SelectionListenerAction)
						((SelectionListenerAction)buttonActions[i]).selectionChanged(structuredSelection);
					getTreePart().setButtonEnabled(i, buttonActions[i].isEnabled());
				}
			}
		}
		else
		{
			for(int i = 0; i < buttonActions.length; i++)
			{
				getTreePart().setButtonEnabled(i, false);
			}
		}
	}
	
	public void setEditable(boolean isEditable)
	{
		this.editable = isEditable;
	}
	
	public boolean isEditable()
	{
		return this.editable;
	}
	
	protected EStructuralFeature getEStructuralFeature()
	{
		return eStructuralFeature;
	}	
}
