/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: PropertySourceExtensionManager.java,v 1.4 2005/02/16 22:22:05 qiyanli Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal;

import java.util.HashMap;
import java.util.Map;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionEvent;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.models.common.testprofile.TPFTestComponent;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.adapter.INamedElementPropertySource;
import org.eclipse.hyades.test.ui.navigator.ITypedElementProxyNode;
import org.eclipse.ui.views.properties.IPropertySource;

/** This class manages the extensibity of the property source extension point.
 *  It uses the extension point org.eclipse.hyades.test.ui.propertySourceExtension
 * @author jgout
 */
public class PropertySourceExtensionManager {
	
	private static PropertySourceExtensionManager instance;
	private static int executionEventIndex = 0;
	private static int executionResultIndex = 1;
	private static int testIndex = 2;
	private static int testComponentIndex = 3;
	private static int typedElementProxyIndex = 4;
	//- these literals should be exactly the same as declared in the extension point schema
	private static String executionEventLiteral = "executionEvent";//$NON-NLS-1$
	private static String executionResultLiteral = "executionResult";//$NON-NLS-1$
	private static String testLiteral 			= "test";//$NON-NLS-1$
	private static String testComponentLiteral = "testComponent";//$NON-NLS-1$
	private static String typedElementProxyLiteral = "typedElementProxy";//$NON-NLS-1$
	
	//- entry 0: TPFExecutionEvent
	//- entry 1: TPFExecutionResult
	//- entry 2: TPFTest
	//- entry 3: TPFTestComponent
	//- entry 4: ITypedElementProxyNode
	//- maps from element type to CMNNamedElementPropertySource
	private Map [] extensionContainer;
	
	public static PropertySourceExtensionManager getInstance() {
		if(instance == null) {
			instance  = new PropertySourceExtensionManager();
		}
		return instance;
	}
	
	private PropertySourceExtensionManager() {
		extensionContainer = new HashMap[5];
		for(int i =0; i<5; i++) {
			extensionContainer[i] = new HashMap();
		}
		registerPropertySourceExtension();
	}
	
	private void registerPropertySourceExtension() {
		IExtensionPoint extPoint = Platform.getExtensionRegistry().getExtensionPoint(TestUIPlugin.getID() + ".propertySourceExtension"); //$NON-NLS-1$
		if (extPoint != null) {
			IConfigurationElement[] extensions = extPoint.getConfigurationElements();
			for(int i = 0; i < extensions.length; i++) {
				IConfigurationElement[] supportedTypes = extensions[i].getChildren();
				for (int j = 0; j < supportedTypes.length; j++) {
					String type = supportedTypes[j].getAttribute("type"); //$NON-NLS-1$
					String kind = supportedTypes[j].getAttribute("kind"); //$NON-NLS-1$
					int index = getIndexFromLiteral(kind);
					if(index >= 0) {
						extensionContainer[index].put(type, extensions[i]);
					}
				}
			}
		}
	}

	private String getTypeFromObject(Object object) {
		if(object instanceof ITypedElementProxyNode) return ((ITypedElementProxyNode) object).getType();
		if(object instanceof TPFExecutionEvent) return ((TPFExecutionEvent)object).getEventType();
		if(object instanceof TPFExecutionResult) return ((TPFExecutionResult)object).getType();
		if(object instanceof TPFTest) return ((TPFTest)object).getType();
		if(object instanceof TPFTestComponent) return ((TPFTestComponent)object).getType();
		return null;
	}

	private int getIndexFromObject(Object object) {
		if(object instanceof ITypedElementProxyNode) return typedElementProxyIndex;
		if(object instanceof TPFExecutionEvent) return executionEventIndex;
		if(object instanceof TPFExecutionResult) return executionResultIndex;
		if(object instanceof TPFTest) return testIndex;
		if(object instanceof TPFTestComponent) return testComponentIndex;
		return -1;
	}
	
	private int getIndexFromLiteral(String kind) {
		if(typedElementProxyLiteral.equals(kind)) return typedElementProxyIndex;
		if(executionEventLiteral.equals(kind)) return executionEventIndex;
		if(executionResultLiteral.equals(kind)) return executionResultIndex;
		if(testLiteral.equals(kind)) return testIndex;
		if(testComponentLiteral.equals(kind)) return testComponentIndex;
		return -1;
	}
	
	/** checks whether the given object is one of managed by the extension
	 *  known types are:
	 *  org.eclipse.hyades.test.ui.navigator.ITypedElementProxyNode;
	 *  org.eclipse.hyades.models.common.testprofile.TPFExecutionEvent;
	 *  org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
	 *  org.eclipse.hyades.models.common.testprofile.TPFTest;
	 *  org.eclipse.hyades.models.common.testprofile.TPFTestComponent;
	 * @param object the object to test
	 * @return true is the given object is one of the known types, flase otherwise
	 */
	public boolean isExtendableObject(Object object) {
		return  object instanceof ITypedElementProxyNode ||
				object instanceof TPFExecutionEvent ||
				object instanceof TPFExecutionResult ||
				object instanceof TPFTest ||
				object instanceof TPFTestComponent;
	}
	

	/** Returns the IPropertySource implementation defined by client of extension 
	 *  point org.eclipse.hyades.test.ui.PropertySourceExtension.
	 * @param object
	 * @return the property source for the given object or <code>null</code> if not found.
	 */
	public IPropertySource getPropertySource(Object object) {
		String type = getTypeFromObject(object);
		if(type == null) return null;
		int index = getIndexFromObject(object);
		if(index >= 0) {
			Object propertySource = extensionContainer[index].get(type);
			if (propertySource instanceof IConfigurationElement) {
				//- IPropertySource implementation has not yet been loaded ... so load it right now
				IConfigurationElement configElem = (IConfigurationElement) propertySource;
				INamedElementPropertySource pse = null; 
				try {
					pse = (INamedElementPropertySource)configElem.createExecutableExtension("class");//$NON-NLS-1$
					if(pse != null) {
						extensionContainer[index].put(type, pse);
						return pse.getPropertySource(object);
					}
				} catch (CoreException e) {
					TestUIPlugin.logError(e);
					return null;
				}
			} else if(propertySource instanceof INamedElementPropertySource) {
				return ((INamedElementPropertySource)propertySource).getPropertySource(object);
			}
		}
		return null;
	}

	
}
