/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: BaseEditorPart.java,v 1.9 2005/05/05 13:39:54 jptoomey Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.editor;

import java.io.File;
import java.util.Collection;
import java.util.Iterator;

import org.eclipse.core.resources.IFile;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.hyades.models.common.facades.behavioral.impl.FacadeResourceImpl;
import org.eclipse.hyades.test.ui.internal.model.EMFUtil;
import org.eclipse.hyades.test.ui.util.RCPFileEditorInput;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.extension.IAssociationDescriptor;
import org.eclipse.hyades.ui.internal.editor.HyadesEditorPart;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.editors.text.ILocationProvider;

/**
 * @author marcelop
 * @since 0.2.0
 */
public abstract class BaseEditorPart
extends HyadesEditorPart
{
	private ResourceSet resourceSet;
	private Class editoObjectClass;
	
	/**
	 * Constructor for BaseEditorPart
	 * @param editoObjectClass
	 */
	public BaseEditorPart(Class editoObjectClass)
	{
		super();
		this.editoObjectClass = editoObjectClass;
	}
	
	/**
	 * @see org.eclipse.ui.IWorkbenchPart#dispose()
	 */
	public void dispose()
	{
		if(resourceSet != null)
		{
			EMFUtil.removeAdapters(resourceSet.getAllContents());
			resourceSet.eAdapters().clear();

			Iterator iterator = getResourceSet().getResources().iterator();
			while (iterator.hasNext()) {
				Resource resource = (Resource) iterator.next();
				if ( resource instanceof FacadeResourceImpl )
					((FacadeResourceImpl)resource).cleanup();
			}
			
			resourceSet.getResources().clear();
			resourceSet = null;
		}
		
		super.dispose();
	}

	/**
	 * Returns this editor resource set.
	 * @return ResourceSet
	 */
	protected ResourceSet getResourceSet()
	{
		return resourceSet;
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.editor.HyadesEditorPart#identifyEditorDescriptor(org.eclipse.ui.IEditorInput)
	 */
	protected IAssociationDescriptor identifyEditorDescriptor(IEditorInput input)
	throws PartInitException
	{		
		IAssociationDescriptor editorDescriptor = null;
		EObject eObject = null;
		if(input instanceof RCPFileEditorInput)
			eObject = getFileContent(((RCPFileEditorInput)input).getIOFile());
		else if(input instanceof IFileEditorInput)
			eObject = getFileContent(((IFileEditorInput)input).getFile());
		else if(input instanceof ILocationProvider)
		{
			ILocationProvider locationProvider = (ILocationProvider)input;
			eObject = getFileContent(locationProvider.getPath(locationProvider).toFile());
		}
		if(eObject == null)
			throw new PartInitException(HyadesUIPlugin.getString("_ERROR_EDT_INPUT")); //$NON-NLS-1$
		
		setEditorObject(eObject);
		editorDescriptor = identifyEditorDescriptor(eObject);

		return editorDescriptor;
	}

	/**
	 * Identifies the editor descriptor that should be used with the eObject.
	 * 
	 * <p>Implementors of this method may store the identified extension as a 
	 * preference for the next time this editor is opened.
	 * 
	 * <p><b>Important</b>: This method is supposed to return a valid 
	 * IAssociationDescriptor - ie. not null and related to an editor.
	 * 
	 * @param input
	 * @return IAssociationDescriptor
	 */
	protected abstract IAssociationDescriptor identifyEditorDescriptor(EObject eObject);

	/**
	 * Returns the "meaningful" content of a given IFile.
	 * @param file
	 * @return Object
	 */
	protected EObject getFileContent(IFile file)
	{
		resourceSet = new ResourceSetImpl();
		AdapterImpl adapterImpl = new AdapterImpl()
		{
			public void notifyChanged(Notification msg)
			{
				switch(msg.getEventType())
				{
					case Notification.ADD:
						if(msg.getNewValue() instanceof Resource)
							((Resource)msg.getNewValue()).setTrackingModification(true);
						break;
					
					case Notification.ADD_MANY:
						Collection collection = (Collection)msg.getNewValue();
						for (Iterator i = collection.iterator(); i.hasNext();)
						{
							Object element = i.next();
							if(element instanceof Resource)
								((Resource)element).setTrackingModification(true);							
						}
						break;	
				}
			}
		};
		resourceSet.eAdapters().add(adapterImpl);
		
		EObject[] eObjects = EMFUtil.load(resourceSet, file);
		if((eObjects.length == 1) && (editoObjectClass.isInstance(eObjects[0])))
			return eObjects[0];
		return null;
	}

	protected EObject getFileContent(File file)
	{
		resourceSet = new ResourceSetImpl();
		AdapterImpl adapterImpl = new AdapterImpl()
		{
			public void notifyChanged(Notification msg)
			{
				switch(msg.getEventType())
				{
					case Notification.ADD:
						if(msg.getNewValue() instanceof Resource)
							((Resource)msg.getNewValue()).setTrackingModification(true);
						break;
					
					case Notification.ADD_MANY:
						Collection collection = (Collection)msg.getNewValue();
						for (Iterator i = collection.iterator(); i.hasNext();)
						{
							Object element = i.next();
							if(element instanceof Resource)
								((Resource)element).setTrackingModification(true);							
						}
						break;	
				}
			}
		};
		resourceSet.eAdapters().add(adapterImpl);
		
		EObject[] eObjects = EMFUtil.load(resourceSet, URI.createFileURI(file.getAbsolutePath()));
		if((eObjects.length == 1) && (editoObjectClass.isInstance(eObjects[0])))
			return eObjects[0];
		return null;
	}

	
	/**
	 * @see org.eclipse.jface.viewers.ISelectionProvider#setSelection(org.eclipse.jface.viewers.ISelection)
	 */
	public void setSelection(ISelection selection)
	{
		selection = adjustSelection(selection);
		if(!selection.isEmpty())
			super.setSelection(selection);
	}
	
	public ISelection adjustSelection(ISelection selection)
	{
		if(selection instanceof IStructuredSelection)
		{
			Object[] objects = ((IStructuredSelection)selection).toArray(); 
			for (int i = 0, maxi = objects.length; i < maxi; i++)
			{
				Object adjusted = null;
				if(objects[i] instanceof EObject)
				{
					adjusted = EMFUtil.getEObject(getResourceSet(), (EObject)objects[i], false);
				}
				else if(objects[i] instanceof Resource)
				{
					adjusted = getResourceSet().getResource(((Resource)objects[i]).getURI(), false);
				}
				
				if(adjusted == null)
					return StructuredSelection.EMPTY;
					
				objects[i] = adjusted;				
			}
			selection = new StructuredSelection(objects);
		}
		return selection;
	}
}
