/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: PropertyGroupForm.java,v 1.11 2005/02/16 22:22:05 qiyanli Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.editor.form;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.hyades.models.common.configuration.CFGComparableProperty;
import org.eclipse.hyades.models.common.configuration.CFGConfigurableObject;
import org.eclipse.hyades.models.common.configuration.CFGPropertyGroup;
import org.eclipse.hyades.models.common.configuration.Common_ConfigurationPackage;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.action.model.MoveFeatureChildrenDownAction;
import org.eclipse.hyades.test.ui.action.model.MoveFeatureChildrenUpAction;
import org.eclipse.hyades.test.ui.action.model.RemoveFeatureChildrenAction;
import org.eclipse.hyades.test.ui.editor.extension.BaseEditorExtension;
import org.eclipse.hyades.test.ui.editor.form.util.EObjectTableContentProvider;
import org.eclipse.hyades.test.ui.editor.form.util.EditorForm;
import org.eclipse.hyades.test.ui.editor.form.util.EditorSection;
import org.eclipse.hyades.test.ui.editor.form.util.ExtensibleEObjectTableSection;
import org.eclipse.hyades.test.ui.editor.form.util.ExtensibleEObjectTreeSection;
import org.eclipse.hyades.test.ui.editor.form.util.WidgetFactory;
import org.eclipse.hyades.test.ui.internal.action.AddConfigPropertyAction;
import org.eclipse.hyades.test.ui.internal.dialog.ComparablePropertyInputDialog;
import org.eclipse.hyades.test.ui.internal.editor.extension.ConfigurableObjectEditorExtension;
import org.eclipse.hyades.test.ui.internal.editor.form.util.EObjectTreeSection;
import org.eclipse.hyades.test.ui.internal.model.ui.ChildrenSelectionAction;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.actions.ActionFactory;

/**
 * This is the default implementation class of extension point <i>org.eclipse.hyades.test.ui.configurationEditorForm</i>. 
 * Extensions may choose to extend this class or simply use directly.
 * 
 * @author bjiang
 * @since  1.3
 */
public class PropertyGroupForm extends EditorForm implements IPropertyGroupForm
{	
	private IConfigurationElement extensionInstance;
	private String name;
	private String description;
	private CFGPropertyGroup propertyGroup;
	private EditorSection mainSection;
	private String editorType = "";

	public static final String CONTROL_TYPE_CUSTOM = "custom"; //$NON-NLS-1$
	public static final String CONTROL_TYPE_TABLE = "table"; //$NON-NLS-1$
	public static final String CONTROL_TYPE_TREE = "tree"; //$NON-NLS-1$
	public static final String CONTROL_TYPE_FIELDS = "fields"; //$NON-NLS-1$
	private String controlType = CONTROL_TYPE_TABLE;
	
	private AddConfigPropertyAction addAction;
	private RemoveFeatureChildrenAction removeAction;	

	protected class PropertyGroupTableSection extends ExtensibleEObjectTableSection
	{		
		/**
		 * @param editorForm
		 * @param eStructuralFeature
		 * @param numOfColumns
		 * @param theButtonActions
		 */
		public PropertyGroupTableSection(EditorForm editorForm, EStructuralFeature eStructuralFeature, int numOfColumns, IAction[] theButtonActions) 
		{
			super(editorForm, eStructuralFeature, numOfColumns,	theButtonActions);
		}
		
		protected void handleDoubleClick(IStructuredSelection structuredSelection)
		{
			if(structuredSelection.getFirstElement() instanceof CFGComparableProperty)
			{
				ComparablePropertyInputDialog dialog = new ComparablePropertyInputDialog(Display.getCurrent().getActiveShell(), TestUIPlugin.getString("DLG_EDIT_PROPERTY")); //$NON-NLS-1$
				dialog.setProperty((CFGComparableProperty)structuredSelection.getFirstElement());
				dialog.open();
			}
		}
	}
		
	/**
	 * This is a {@link ITableLabelProvider} class that knows what text to display in 
	 * the table control of a {@link IPropertyGroupForm} with a given {@link CFGComparableProperty} object. 
	 */
	public class PropertyGroupTableLabelProvider extends LabelProvider implements ITableLabelProvider
	{	
		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnImage(java.lang.Object, int)
		 */
		public Image getColumnImage(Object element, int columnIndex) 
		{
			return null;
		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnText(java.lang.Object, int)
		 */
		public String getColumnText(Object element, int columnIndex) 
		{
			if(element instanceof CFGComparableProperty)
			{
				CFGComparableProperty property = (CFGComparableProperty)element;
				switch(columnIndex)
				{
					case 0:
						return property.getName();
					case 1:
					    if(property.getOperator() == null)
					        property.setOperator("=");
						return property.getOperator();
					case 2:
						return property.getValue();
				}
			}
			
			return null;
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.IPropertyGroupForm#init(org.eclipse.hyades.models.common.configuration.CFGPropertyGroup, org.eclipse.hyades.test.ui.internal.editor.extension.BaseEditorExtension, org.eclipse.hyades.test.ui.internal.editor.form.util.WidgetFactory)
	 */
	public void init(CFGPropertyGroup propertyGroup, BaseEditorExtension editorExtension, WidgetFactory widgetFactory)
	{
		super.init(editorExtension, widgetFactory);
		this.propertyGroup = propertyGroup;
		if(propertyGroup.eContainer() != null)
		    this.editorType = propertyGroup.eContainer().eClass().getInstanceClassName();
	}
	
    /* (non-Javadoc)
     * @see org.eclipse.hyades.test.ui.internal.editor.form.IPropertyGroupForm#setExtension(org.eclipse.core.runtime.IConfigurationElement)
     */
    public void setExtension(IConfigurationElement extension)
    {
        this.extensionInstance = extension;
        this.name = extension.getAttribute("name");
        this.description = extension.getAttribute("description");
        this.controlType = extension.getAttribute("controlType");
        this.editorType = extension.getAttribute("appliesTo");
        if(extension.getAttribute("addAction") != null)
	    {
	        try
	        {
	            this.addAction = (AddConfigPropertyAction)extension.createExecutableExtension("addAction");
	            addAction.setPropertyGroup(this.propertyGroup);
	        }
	        catch (CoreException e){}
	    }
    }
    
    public String getName()
    {
        if(name == null)
            name = "";
        
        return name;
    }
    
    public String getDescription()
    {
        if(description == null)
            description = "";
        
        return description;
    }
	
	/*
	 * (non-Javadoc)
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.IPropertyGroupForm#getPropertyGroup()
	 */
	public CFGPropertyGroup getPropertyGroup()
	{
		return propertyGroup;
	}
	
	/**
	 * Returns the type of the control this form contains.
	 * 
	 * @return one of "table", "tree", "fields" or "custom".
	 */
	public String getControlType()
	{
		return this.controlType;
	}

	/**
	 * Returns which {@link CFGConfigurableObject}'s property group this form is editing.
	 * 
	 * @return a {@link CFGConfigurableObject} concrete class name. 
	 */
	public String getAppliesTo()
	{
		return this.editorType;
	}
	
	/**
	 * @see org.eclipse.update.ui.forms.internal.IForm#dispose()
	 */
	public void dispose()
	{
	    if(mainSection != null)
	    {
	        mainSection.dispose();
	        mainSection = null;
	    }
	    extensionInstance = null;
		super.dispose();
	}
	
	/**
	 * Returns the data object that is manipulated by the editor that this form belongs.
	 * @return EObject 
	 */
	protected EObject getEditorObject()
	{
		return ((ConfigurableObjectEditorExtension)getBaseEditorExtension()).getEditorObject();
	}
	
	/**
	 * Creates the contents of this editor form.
	 * @param parent
	 */
	protected void createEditorFormContents(Composite parent)
	{
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 2;
		gridLayout.makeColumnsEqualWidth = true;
		gridLayout.marginWidth = 10;
		gridLayout.horizontalSpacing = 15;
		parent.setLayout(gridLayout);

		Composite leftColumn = createColumn(parent);
		Composite rightColumn = createColumn(parent);
		
		if( controlType.equals(CONTROL_TYPE_TREE) )
		{
			mainSection = createTreeSection(leftColumn);
		}
		else if( controlType.equals(CONTROL_TYPE_TABLE) )
		{
			mainSection = createTableSection(leftColumn);
		}
		else if( controlType.equals(CONTROL_TYPE_FIELDS) )
		{
		}
		else
		{
		}
		
		if(mainSection != null)
		{
			registerSection(mainSection);
			mainSection.setHeaderText(getName());
			mainSection.setCollapsable(true);
			Control control = mainSection.createControl(leftColumn, getWidgetFactory());
			control.setLayoutData(new GridData(GridData.FILL_BOTH | GridData.VERTICAL_ALIGN_BEGINNING));
		}
	}
	
	/**
	 * Creates a tree section with <i>add, remove, move up, move down</i> buttons as the main control of this form.
	 * @param parent 
	 * @return a {@link EditorSection}
	 */
	protected EditorSection createTreeSection(Composite parent)
	{
		IAction[] buttonActions = createButtonActions();
		ExtensibleEObjectTreeSection treeSection = new ExtensibleEObjectTreeSection(this, Common_ConfigurationPackage.eINSTANCE.getCFGPropertyGroup_Properties(), buttonActions);
		for(int i = 0; i < buttonActions.length; i++)
		{
		    IAction action = buttonActions[i];
		    if(action != null && action instanceof ChildrenSelectionAction)
		    {
		        ((ChildrenSelectionAction)action).setStructuredViewer(treeSection.getTreeViewer());
		    }
		}
		return treeSection;
	}
	
	/**
	 * Creates a table section with <i>add, remove, move up, move down</i> buttons as the main control of this form.
	 * @param parent 
	 * @return a {@link EditorSection}
	 */
	protected EditorSection createTableSection(Composite parent)
	{
	    IAction[] buttonActions = createButtonActions();
	    PropertyGroupTableSection tableSection = new PropertyGroupTableSection(this, Common_ConfigurationPackage.eINSTANCE.getCFGPropertyGroup_Properties(), 3, buttonActions);
		tableSection.setContentProvider(new EObjectTableContentProvider(this, Common_ConfigurationPackage.eINSTANCE.getCFGPropertyGroup_Properties()));
		tableSection.setLabelProvider(new PropertyGroupTableLabelProvider());
		for(int i = 0; i < buttonActions.length; i++)
		{
		    IAction action = buttonActions[i];
		    if(action != null && action instanceof ChildrenSelectionAction)
		    {
		        ((ChildrenSelectionAction)action).setStructuredViewer(tableSection.getTableViewer());
		    }
		}
		return tableSection;
	}
	
	/**
	 * Creates the actions of the buttons that the primary editor section will use. 
	 * @return
	 */
	protected IAction[] createButtonActions()
	{
		IAction[] buttonActions;
		if(addAction == null)
		{
		    buttonActions = new IAction[3];
		    buttonActions[0] = removeAction = new RemoveFeatureChildrenAction(getPropertyGroup(), Common_ConfigurationPackage.eINSTANCE.getCFGPropertyGroup_Properties(), false);	
			buttonActions[1] = new MoveFeatureChildrenUpAction(getPropertyGroup(), Common_ConfigurationPackage.eINSTANCE.getCFGPropertyGroup_Properties());
			buttonActions[2] = new MoveFeatureChildrenDownAction(getPropertyGroup(), Common_ConfigurationPackage.eINSTANCE.getCFGPropertyGroup_Properties());
		}
		else
		{
		    buttonActions = new IAction[4];
		    buttonActions[0] = addAction;
			buttonActions[1] = removeAction = new RemoveFeatureChildrenAction(getPropertyGroup(), Common_ConfigurationPackage.eINSTANCE.getCFGPropertyGroup_Properties(), false);	
			buttonActions[2] = new MoveFeatureChildrenUpAction(getPropertyGroup(), Common_ConfigurationPackage.eINSTANCE.getCFGPropertyGroup_Properties());
			buttonActions[3] = new MoveFeatureChildrenDownAction(getPropertyGroup(), Common_ConfigurationPackage.eINSTANCE.getCFGPropertyGroup_Properties());
		}
		
		return buttonActions;
	}

	/**
	 * Subclasses should load the persited values in the controls in this method.  
	 * The goal is to provide a common behavior for the editor forms.
	 */
	public void load()
	{		
		if( controlType.equals(CONTROL_TYPE_TREE) )
		{
			EObjectTreeSection section = (EObjectTreeSection)mainSection;
			//section.getTreeViewer().removeSelectionChangedListener(section);
			section.setInput(propertyGroup);
			//section.getTreeViewer().addSelectionChangedListener(section);
		}
		else if( controlType.equals(CONTROL_TYPE_TABLE) )
		{
			ExtensibleEObjectTableSection section = (ExtensibleEObjectTableSection)mainSection;
			section.getTableViewer().removeSelectionChangedListener(section);
			section.setInput(propertyGroup);
			section.getTableViewer().addSelectionChangedListener(section);
		}
		else if( controlType.equals(CONTROL_TYPE_FIELDS) )
		{
		}
		else
		{
		}
	}	
	
	/**
	 * Subclass should override this method to return the specific selected object in the form.
	 * This default implementation just returns the data object of the overall editor.
	 * @return IStructuredSelection
	 */
	public IStructuredSelection getSelection()
	{
		return new StructuredSelection(getEditorObject());
	}
	
	/**
	 * @see org.eclipse.ui.part.ISetSelectionTarget#selectReveal(org.eclipse.jface.viewers.ISelection)
	 */
	public void selectReveal(ISelection selection)
	{
		mainSection.selectReveal(selection);
	}
	
	/**
	 * @see org.eclipse.update.ui.forms.internal.FormSection#doGlobalAction(java.lang.String)
	 */
	public boolean doGlobalAction(String actionId)
	{
		if (actionId.equals(ActionFactory.DELETE.getId()))
		{
			if(removeAction != null)
				removeAction.run();
			return true;
		}
		return false;
	}

}
