/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: EObjectContainerContentProvider.java,v 1.3 2005/02/16 22:22:05 qiyanli Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.editor.form.util;

import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.hyades.test.ui.editor.form.util.EditorForm;
import org.eclipse.hyades.test.ui.internal.model.EMFUtil;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;

/**
 * @author marcelop, bcormier
 * @since 3.0.0
 */
public class EObjectContainerContentProvider 
extends AdapterImpl implements ITreeContentProvider
{
	private EStructuralFeature eStructuralFeature;
	private EObject container;
	private Viewer viewer;
	private EditorForm editorForm;
	private Map eStructuralFeatureByClass;
	
	/**
	 * Constructor for EObjectContainerContentProvider
	 * @param editorForm
	 * @param eStructuralFeature
	 */
	public EObjectContainerContentProvider(EditorForm editorForm, EStructuralFeature eStructuralFeature)
	{
		this.eStructuralFeature = eStructuralFeature;
		this.editorForm = editorForm;
	}
	
	/**
	 * @see org.eclipse.jface.viewers.IContentProvider#dispose()
	 */
	public void dispose()
	{
		if(container != null)
		{
			container.eAdapters().remove(this);
			Object value = container.eGet(eStructuralFeature);
			if(value instanceof Collection)
				EMFUtil.removeAdapter(((Collection)value).iterator(), this);
			else if(value != null)
				((Notifier)value).eAdapters().remove(this);

			container = null;
		}
		
		viewer = null;
		eStructuralFeature = null;
		editorForm = null;
		
		if(eStructuralFeatureByClass != null)
			eStructuralFeatureByClass.clear();
	}
	
	protected EObject getContainer()
	{
		return container;
	}
	
	public void registerParent(Class cls, EStructuralFeature esf)
	{
		if(eStructuralFeatureByClass == null)
			eStructuralFeatureByClass = new HashMap();
		eStructuralFeatureByClass.put(cls, esf);
	}	

	public void deregisterParent(Class cls)
	{
		if(eStructuralFeatureByClass != null)
			eStructuralFeatureByClass.remove(cls);
	}	
	
	protected Object getRegisteredParentChild(Object parent)
	{
		if(eStructuralFeatureByClass != null)
		{
			for (Iterator i = eStructuralFeatureByClass.entrySet().iterator(); i.hasNext();)
			{
				Map.Entry entry = (Map.Entry)i.next();
				if(((Class)entry.getKey()).isInstance(parent))
				{
					if(parent instanceof EObject)
						return ((EObject)parent).eGet((EStructuralFeature)entry.getValue());
					return null;						
				}
			}
		}
		return null;						
	} 

	/**
	 * @see org.eclipse.jface.viewers.IContentProvider#inputChanged(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
	 */
	public void inputChanged(Viewer theViewer, Object oldInput, Object newInput)
	{
		viewer = theViewer;
		
		if(container != null)
		{
			container.eAdapters().remove(this);
			Object value = container.eGet(eStructuralFeature);
			deregisterInputAdapters(container, value);
			container = null;
		}
		
		if(newInput instanceof EObject)
		{
			EObject eObject = (EObject)newInput;
			if(eStructuralFeature.eContainer() == eObject.eClass())
			{
				container = eObject;
				container.eAdapters().add(this);
				registerInputAdapters(eObject, container.eGet(eStructuralFeature));
			}
		}
	}
	
	protected Viewer getViewer()
	{
		return viewer;
	}
	
	protected void deregisterInputAdapters(EObject oldInput, Object value)
	{
		if(value instanceof Collection)
			EMFUtil.removeAdapter(((Collection)value).iterator(), this);
		else if(value != null)
			((Notifier)value).eAdapters().remove(this);			
	}
	
	protected void registerInputAdapters(EObject newInput, Object value)
	{
		if(value instanceof List)
		{	
			EMFUtil.addAdapter(((List)value).iterator(), this);
		}
		else if(value != null)
		{
			((Notifier)value).eAdapters().add(this);
		}			
	}

	/**
	 * @see org.eclipse.jface.viewers.ITreeContentProvider#getParent(java.lang.Object)
	 */
	public Object getParent(Object element)
	{
		if(container == element)
			return null;
			
		if(element instanceof EObject)
			return ((EObject)element).eContainer();
		
		return null;
	}
	
	/**
	 * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(java.lang.Object)
	 */
	public Object[] getElements(Object inputElement)
	{
		if(container == null)
			return new Object[0];

		if(eStructuralFeature.isMany())
			return ((List)container.eGet(eStructuralFeature)).toArray();

		return new Object[]{container.eGet(eStructuralFeature)}; 
	}

	/**
	 * @see org.eclipse.jface.viewers.ITreeContentProvider#hasChildren(java.lang.Object)
	 */
	public boolean hasChildren(Object element)
	{
		if(container == null)
			return false;

		if(!eStructuralFeature.isMany())
			return false;

		if(container == element)
		{
			List values = (List)container.eGet(eStructuralFeature);
			return !values.isEmpty();
		}
		
		Object child = getRegisteredParentChild(element);
		return (child != null);
	}

	/**
	 * @see org.eclipse.jface.viewers.ITreeContentProvider#getChildren(java.lang.Object)
	 */
	public Object[] getChildren(Object parentElement)
	{
		Object child = getRegisteredParentChild(parentElement);
		if(child != null)
		{
			if(child instanceof Collection)
				return ((Collection)child).toArray();
			return new Object[]{child};
		}
		
		return new Object[0];
	}

	/**
	 * @see org.eclipse.emf.common.notify.Adapter#notifyChanged(org.eclipse.emf.common.notify.Notification)
	 */
	public void notifyChanged(Notification msg)
	{
		switch(msg.getEventType())
		{
			case Notification.ADD:
			case Notification.ADD_MANY:
			case Notification.REMOVE:
			case Notification.REMOVE_MANY:
			case Notification.MOVE:
			case Notification.SET:
			case Notification.UNSET:
				break;
				
			default:
				return; 	
		}
		
		editorForm.getBaseEditorExtension().markDirty();
		adjustModel(msg);
		adjustAdapter(msg);
		updateViewer(msg);
	}
	
	protected void adjustModel(Notification msg)
	{
	}
	
	protected boolean shouldAdjustAdapter(Notification msg)
	{
		return ((msg.getNotifier() == container) && (msg.getFeature() == eStructuralFeature)); 
	}
	
	protected void adjustAdapter(Notification msg)
	{
		if(shouldAdjustAdapter(msg))
		{
			switch(msg.getEventType())
			{
				case Notification.ADD:
					((Notifier)msg.getNewValue()).eAdapters().add(this);
					break;

				case Notification.ADD_MANY:
					EMFUtil.addAdapter(((Collection)msg.getNewValue()).iterator(), this);
					break;
					
				case Notification.REMOVE:
					((Notifier)msg.getOldValue()).eAdapters().remove(this);
					break;

				case Notification.REMOVE_MANY:
					EMFUtil.removeAdapter(((Collection)msg.getOldValue()).iterator(), this);
					break;
			}
		}
		
	}

	protected boolean shouldUpdateViewer(Notification msg)
	{
		return (msg.getFeature() == eStructuralFeature);	
	}

	protected void updateViewer(Notification msg)
	{
		if(getViewer() == null)
			return;

		if(!(getViewer() instanceof StructuredViewer))
		{
			getViewer().refresh();
			return;
		}

		if(msg.getNotifier() != container)
		{
			switch(msg.getEventType())
			{
				case Notification.SET:
				case Notification.UNSET:
					attributeChanged(msg);
					return;
			}
		}

		if(!(getViewer() instanceof TreeViewer))
		{
			getViewer().refresh();
			return;
		}
			
		if(shouldUpdateViewer(msg))
		{	
			Object parent = null;
			switch(msg.getEventType())
			{
				case Notification.ADD:
					parent = getParent(msg.getNewValue());
					if(parent == null)
						parent = getViewer().getInput();
					((TreeViewer)getViewer()).add(parent, msg.getNewValue());
					getViewer().setSelection(new StructuredSelection(msg.getNewValue()), true);
					break;

				case Notification.ADD_MANY:
					Object[] addedObjects = ((Collection)msg.getNewValue()).toArray();
					parent = getParent(addedObjects[0]);
					if(parent == null)
						parent = getViewer().getInput();
						
					((TreeViewer)getViewer()).add(parent, addedObjects);
					getViewer().setSelection(new StructuredSelection(addedObjects), true);
					break;
						
				case Notification.REMOVE:
					((TreeViewer)getViewer()).remove(msg.getOldValue());
					Object value = ((EObject)msg.getNotifier()).eGet((EStructuralFeature)msg.getFeature());
					if(value instanceof List)
					{
						List values = (List)value;
						if(values.isEmpty())
						{
							getViewer().getControl().setFocus();
						}
						else
						{
							int position = msg.getPosition();
							position = (position<values.size())?position:(values.size()-1);
							getViewer().setSelection(new StructuredSelection(values.get(position)), true);
						}
					}
					else
					{
						getViewer().getControl().setFocus();
					}
					break;

				case Notification.REMOVE_MANY:
					((TreeViewer)getViewer()).remove(((Collection)msg.getOldValue()).toArray());
					getViewer().getControl().setFocus();
					break;

				case Notification.MOVE:
					getViewer().refresh();
					break;
			}
		}
	}
	
	protected void attributeChanged(Notification msg)
	{
		((StructuredViewer)getViewer()).update(msg.getNotifier(), null);
	}
}

