/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TestProvider.java,v 1.5 2005/02/16 22:22:06 qiyanli Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.launch.providers;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.models.common.testprofile.TPFTestCase;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.test.ui.IHyadesTestNavigatorConverter;
import org.eclipse.hyades.test.ui.IHyadesTestNavigatorProvider;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.internal.launch.extensions.LaunchConfigurationExtensionsManager;
import org.eclipse.hyades.test.ui.internal.model.ui.ITypeProviderFactory;
import org.eclipse.hyades.test.ui.internal.navigator.TestCaseHTNProvider;
import org.eclipse.hyades.test.ui.internal.navigator.TestSuiteHTNConverter;
import org.eclipse.hyades.test.ui.internal.navigator.TestSuiteHTNProvider;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.graphics.Image;
import org.eclipse.ui.model.IWorkbenchAdapter;

/**
 * @author jcanches
 */
public class TestProvider extends LabelProvider implements ITreeContentProvider {
	
	private IHyadesTestNavigatorConverter testSuiteConverter = new TestSuiteHTNConverter();
	private IHyadesTestNavigatorProvider testSuiteProvider = new TestSuiteHTNProvider(emptyTypeProviderFactory);
	private IHyadesTestNavigatorProvider testCaseProvider = new TestCaseHTNProvider(emptyTypeProviderFactory);
	private String mode;
	private String launchConfigurationType;
	
	/**
	 * Create a new TestProvider.
	 * @param mode The mode the test should be run under (debug, profile, run...)
	 * @param launchConfigurationType The name of the Launch Configuration type used
	 * for running the test. A null value denotes the default Hyades Launch Configuration.
	 */
	public TestProvider(String mode, String launchConfigurationType) {
		this.mode = mode;
		this.launchConfigurationType = launchConfigurationType;
	}
	
	public Object[] getChildren(Object parentElement) {
		if (parentElement instanceof IContainer) {
			try {
				IContainer container = (IContainer)parentElement;
				IResource[] members = container.members();
				List children = new ArrayList(members.length);
				for (int i = 0; i < members.length; i++) {
					if (members[i] instanceof IContainer) {
						children.add(members[i]);
					} else if (members[i] instanceof IFile) {
						Object converted = testSuiteConverter.getObjectFromResource(members[i]);
						if (converted != null && converted instanceof TPFTestSuite && isLaunchable((TPFTestSuite)converted)) {
							children.add(converted);
						}
					}
				}
				return children.toArray();
			} catch (CoreException e) {
				TestUIPlugin.logError(e);
				return new Object[0];
			}
		} else if (parentElement instanceof TPFTestSuite) {
			Object[] children = testSuiteProvider.getChildren(parentElement);
			List childlist = new ArrayList(children.length);
			for (int i = 0; i < children.length; i++) {
				if (children[i] instanceof TPFTestCase && isLaunchable((TPFTestCase)children[i])) {
					childlist.add(children[i]);
				}
			}
			return childlist.toArray();
		} else {
			return new Object[0];
		}
	}
	
	public Object getParent(Object element) {
		if (element instanceof IResource) {
			return ((IResource)element).getParent();
		} else if (element instanceof TPFTestSuite) {
			return testSuiteProvider.getParent(element);
		} else if (element instanceof TPFTestCase) {
			return testCaseProvider.getParent(element);
		}
		return null;
	}

	public boolean hasChildren(Object element) {
		return getChildren(element).length > 0;
	}
	
	public Object[] getElements(Object inputElement) {
		return getChildren(inputElement);
	}
	
	public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		// NOP
	}
	
	public Image getImage(Object element) {
		if (element instanceof IContainer) {
			ImageDescriptor descriptor;
			IWorkbenchAdapter workbenchAdapter = getWorkbenchAdapter(element);
			if (workbenchAdapter != null) {
				descriptor = workbenchAdapter.getImageDescriptor(element);
				if (descriptor != null)
					return descriptor.createImage();
			}
		} else if (element instanceof TPFTestSuite) {
			return testSuiteProvider.getImage(element);
		} else if (element instanceof TPFTestCase) {
			return testCaseProvider.getImage(element);
		}
		return null;
	}
	
	public String getText(Object element) {
		if (element instanceof IContainer) {
			IWorkbenchAdapter workbenchAdapter = getWorkbenchAdapter(element);
			if (workbenchAdapter != null) {
				return workbenchAdapter.getLabel(element);
			}
		} else if (element instanceof TPFTestSuite) {
			return testSuiteProvider.getText(element);
		} else if (element instanceof TPFTestCase) {
			return testCaseProvider.getText(element);
		}
		return null;
	}
	
	private boolean isLaunchable(TPFTest test) {
		return LaunchConfigurationExtensionsManager.getInstance().isLaunchableElement(test, mode, launchConfigurationType);
	}
	
	private IWorkbenchAdapter getWorkbenchAdapter(Object object) {
		IWorkbenchAdapter workbenchAdapter = (IWorkbenchAdapter) Platform.getAdapterManager().getAdapter(object, IWorkbenchAdapter.class);
		if (workbenchAdapter == this)
			return null;
		return workbenchAdapter;
	}
	
	private static ITypeProviderFactory emptyTypeProviderFactory = new ITypeProviderFactory() {
		public IHyadesTestNavigatorProvider getProvider(String type) {
			// Always use the default provider
			return null;
		}
	};
	
}
