/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: LaunchShortcut.java,v 1.9 2005/06/09 08:55:45 dguilbaud Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.launch.shortcuts;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.internal.runtime.Assert;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationType;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.core.ILaunchManager;
import org.eclipse.debug.internal.ui.DebugUIPlugin;
import org.eclipse.debug.ui.DebugUITools;
import org.eclipse.debug.ui.IDebugModelPresentation;
import org.eclipse.debug.ui.ILaunchShortcut;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.hyades.models.common.testprofile.TPFDeployment;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.models.common.testprofile.TPFTestCase;
import org.eclipse.hyades.models.common.testprofile.TPFTestComponent;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.internal.launch.extensions.LaunchConfigurationExtensionsManager;
import org.eclipse.hyades.test.ui.internal.model.EMFUtil;
import org.eclipse.hyades.test.ui.launch.configurations.DeploymentLaunchConfigurationFacade;
import org.eclipse.hyades.test.ui.launch.configurations.ExecutionHistoryLaunchConfigurationFacade;
import org.eclipse.hyades.test.ui.launch.configurations.TestComponentLaunchConfigurationFacade;
import org.eclipse.hyades.test.ui.launch.configurations.TestLaunchConfigurationFacade;
import org.eclipse.hyades.test.ui.launch.extensions.IDeploymentsProvider;
import org.eclipse.hyades.test.ui.launch.extensions.IExecutionHistoryDefaultsProvider;
import org.eclipse.hyades.test.ui.navigator.ITestCaseProxyNode;
import org.eclipse.hyades.test.ui.navigator.ITestSuiteProxyNode;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.dialogs.ElementListSelectionDialog;
import org.eclipse.ui.internal.Workbench;

/**
 * Created by @author pnedelec
 * @author jcanches
 *
 */
public class LaunchShortcut implements ILaunchShortcut {

	/**
	 * Consider an object (listed in a selection) and try to extract its underlying Test Suite object
	 * if it is possible.
	 * @param element Any object.
	 * @return A TPFTestSuite if the object is or is associated to a test suite, null otherwise.
	 */
	static Object getRunnableItemFromElement(Object element, String mode) {
		Object candidateElement = null;
		if (element instanceof TPFTestSuite ||
			element instanceof TPFTestCase ||
			element instanceof TPFTestComponent) {
			candidateElement = element;
		} else if (element instanceof IFile) {
			IFile file = (IFile) element;
			if (ICommonConstants.TEST_SUITE_FILE_EXTENSION.equals(file.getFileExtension())) {
				EObject[] eobjs = EMFUtil.load(null, file);
				for (int i = 0; i < eobjs.length; i++) {
					if (eobjs[i] instanceof TPFTestSuite) {
						candidateElement = eobjs[i];
						break;
					}
				}
			} else if (ICommonConstants.TEST_COMPONENT_EXTENSION.equals(file.getFileExtension())) {
				EObject[] eobjs = EMFUtil.load(null, file);
				for (int i = 0; i < eobjs.length; i++) {
					if (eobjs[i] instanceof TPFTestComponent) {
						candidateElement = eobjs[i];
						break;
					}
				}
			}
		} else if (element instanceof ITestSuiteProxyNode) {
			ITestSuiteProxyNode tsProxy = (ITestSuiteProxyNode) element;
			candidateElement = tsProxy.getTestSuite();
		} else if (element instanceof ITestCaseProxyNode) {
			ITestCaseProxyNode tcProxy = (ITestCaseProxyNode) element;
			candidateElement = tcProxy.getTestCase();
		}
		if (candidateElement != null) {
			if (LaunchConfigurationExtensionsManager.getInstance().isShortcutAvailable(candidateElement, mode)) {
				return candidateElement;
			} else {
				return null;
			}
		}
		return null;
	}

	/*
	 * @see org.eclipse.debug.ui.ILaunchShortcut#launch(org.eclipse.ui.IEditorPart, java.lang.String)
	 */
	public void launch(IEditorPart editor, String mode) {
		IEditorInput input = editor.getEditorInput();
		if (input instanceof IFileEditorInput) {
			IFileEditorInput finput = (IFileEditorInput) input;
			Object obj = getRunnableItemFromElement(finput.getFile(), mode);
			if (obj != null) {
				launchElement(obj, mode);
			}
		}
	}

	/*
	 * @see org.eclipse.debug.ui.ILaunchShortcut#launch(org.eclipse.jface.viewers.ISelection, java.lang.String)
	 */
	public void launch(ISelection selection, String mode) {
		if (selection instanceof IStructuredSelection) {
			// Use the first element from the collection
			IStructuredSelection ss = (IStructuredSelection)selection;
			Iterator it = ss.iterator();
			while (it.hasNext()) {
				Object element = it.next();
				Object launchable = getRunnableItemFromElement(element, mode);
				if (launchable != null) {
					launchElement(launchable, mode);
				}
			}
		}
	}

	/**
	 * Launch a launchable element. If a Launch Configuration already exists for the specified
	 * element, this launch configuration is used to launch the element. Otherwise,
	 * a new Launch Configuration is created. If several launch configurations match, a dialog
	 * is opened to allow the user to choose from the candidate launch configurations.
	 * @param testSuites The Test Suites to launch
	 * @param mode The launch mode (debug, run or profile).
	 */
	private void launchElement(Object launchable, String mode) {
		//  Find an existing launch configuration for the set of Test Suites
		ILaunchConfiguration lconfig = findLaunchConfiguration(launchable, mode);
		try {
			if (lconfig == null) {
				// If no match, create a new Launch Configuration
				lconfig = createLaunchConfiguration(launchable);
			}
			if (lconfig != null) {
				DebugUIPlugin.launchInBackground(lconfig, mode);
			}
		} catch (Throwable t) {
			reportFailure(t);
		}
	}
	
    /**
	 * Find a Launch Configuration that matches the launchable element.
	 * @param launchable The launchable element that the returned Launch Configuration must run.
	 * @return An existing Launch Configuration if found, null otherwise.
	 */
	private ILaunchConfiguration findLaunchConfiguration(Object launchable, String mode) {
		List candidateConfigs= Collections.EMPTY_LIST;
		ILaunchConfigurationType configsType = getConfigType(launchable);
		Assert.isNotNull(configsType);
		
		// Model Elements must belong to a new ResourceSet.
		ResourceSet shortcutResourceSet = new ResourceSetImpl();
		
		try {
			
			ILaunchConfiguration[] configs= getLaunchManager().getLaunchConfigurations(configsType);
			candidateConfigs= new ArrayList(configs.length);
			for (int i = 0; i < configs.length; i++) {
				Object configLaunchableElement = null;
				if (launchable instanceof TPFTest) {
					configLaunchableElement = TestLaunchConfigurationFacade.getTest(configs[i], shortcutResourceSet);
				} else if (launchable instanceof TPFTestComponent) {
					configLaunchableElement = TestComponentLaunchConfigurationFacade.getTestComponent(configs[i], shortcutResourceSet);
				}
				if (launchable.equals(configLaunchableElement)) {
					candidateConfigs.add(configs[i]);
				}
			}
			if (candidateConfigs.size() == 0) {
				return null;
			} else if (candidateConfigs.size() == 1) {
				return (ILaunchConfiguration)candidateConfigs.get(0);
			} else {
				return chooseFromCandidateConfigs(candidateConfigs, mode);
			}
		} catch(CoreException e) {
			TestUIPlugin.logError(e);
		}
		return null;
	}

	/**
	 * Show a selection dialog that allows the user to choose one of the specified
	 * launch configurations.  Return the chosen config, or <code>null</code> if the
	 * user cancelled the dialog.
	 */
	private ILaunchConfiguration chooseFromCandidateConfigs(List configList, String mode) {
		IDebugModelPresentation labelProvider = DebugUITools.newDebugModelPresentation();
		ElementListSelectionDialog dialog= new ElementListSelectionDialog(this.getActiveShell(), labelProvider);
		dialog.setElements(configList.toArray());
		dialog.setTitle(TestUIPlugin.getString("LaunchShortcut.ambiguousLaunchConfigurationTitle")); //$NON-NLS-1$
		if (mode.equals(ILaunchManager.DEBUG_MODE)) {
			dialog.setMessage(TestUIPlugin.getString("LaunchShortcut.selectDebugLaunchConfigPrompt")); //$NON-NLS-1$
		} else if (mode.equals(ILaunchManager.PROFILE_MODE)){
			dialog.setMessage(TestUIPlugin.getString("LaunchShortcut.selectProfileLaunchConfigPrompt")); //$NON-NLS-1$
		} else {
			dialog.setMessage(TestUIPlugin.getString("LaunchShortcut.selectRunLaunchConfigPrompt")); //$NON-NLS-1$
		}
		dialog.setMultipleSelection(false);
		int result= dialog.open();
		labelProvider.dispose();
		if (result == Window.OK) {
			return (ILaunchConfiguration)dialog.getFirstResult();
		}
		return null;		
	}
	
	/**
	 * Create a new Launch Configuration that runs the specified launchable element.
	 * @param launchable
	 * @return A new Launch Configuration.
	 */
	private ILaunchConfiguration createLaunchConfiguration(Object launchable) throws CoreException {
		ILaunchConfiguration config = null;
		ILaunchConfigurationType configType = getConfigType(launchable);
		ILaunchConfigurationWorkingCopy wc = configType.newInstance(null, getLaunchManager().generateUniqueLaunchConfigurationNameFrom(TestUIPlugin.getString("LaunchShortcut.uniqueNamePrefix"))); //$NON-NLS-1$
		if (launchable instanceof TPFTest) {
			TestLaunchConfigurationFacade.setTest(wc, (TPFTest)launchable);
		} else if (launchable instanceof TPFTestComponent) {
			TestComponentLaunchConfigurationFacade.setTestComponent(wc, (TPFTestComponent)launchable);
		}
		IDeploymentsProvider dprovider = LaunchConfigurationExtensionsManager.getInstance().getDeploymentsProvider(launchable);
		TPFDeployment depl = null;
		depl = dprovider.getDefaultDeployment(launchable);
		if (depl == null) {
			throwException(TestUIPlugin.getString("_EXC_LaunchShortcut.noDefaultDeployment"), null); //$NON-NLS-1$
		}
		DeploymentLaunchConfigurationFacade.setDeployment(wc, depl);
		String ehName = null;
		IContainer ehLocation = null;
		IExecutionHistoryDefaultsProvider eprovider = LaunchConfigurationExtensionsManager.getInstance().getExecutionHistoryDefaultsProvider(launchable);
		ehName = eprovider.getDefaultName(launchable);
		if (ehName == null) {
			ehName = LaunchConfigurationExtensionsManager.getInstance().getDefaultExecutionHistoryDefaultsProvider().getDefaultName(launchable);
		}
		ehLocation = eprovider.getDefaultLocation(launchable);
		if (ehLocation == null) {
			ehLocation = LaunchConfigurationExtensionsManager.getInstance().getDefaultExecutionHistoryDefaultsProvider().getDefaultLocation(launchable);
		}
		if (ehName == null || ehLocation == null) {
			throwException(TestUIPlugin.getString("_EXC_LaunchShortcut.noDefaultExecHistoryInfo"), null); //$NON-NLS-1$
		}
		ExecutionHistoryLaunchConfigurationFacade.setExecutionHistoryName(wc, ehName);
		ExecutionHistoryLaunchConfigurationFacade.setExecutionHistoryLocation(wc, ehLocation);
		config = wc.doSave();
		return config;
	}
	
    private void reportFailure(Throwable exception) {
		TestUIPlugin.logError(exception);
		MessageBox msgBox = new MessageBox(
				Workbench.getInstance().getActiveWorkbenchWindow().getShell(),
				SWT.ICON_ERROR | SWT.OK);
		msgBox.setText(TestUIPlugin.getString("LaunchShortcut.errorDialogTitle"));
		String message = null;
		if (exception instanceof CoreException) {
			CoreException ce = (CoreException)exception;
			if (ce.getStatus() != null && ce.getStatus().getMessage() != null) {
				message = ce.getStatus().getMessage();
			}
		}
		if (message == null) {
			message = "An exception occured. See error log for more details";
		}
		msgBox.setMessage(message);
		msgBox.open();
    }
    
    private void throwException(String localizedMessage, Throwable exception) throws CoreException {
    	IStatus status = new Status(IStatus.ERROR, TestUIPlugin.getID(), 0, localizedMessage, exception);
    	throw new CoreException(status);
    }
    
	/**
	 * Return the appropriate Launch Configuration Type for launching the supplied
	 * launchable element.
	 * @param launchable
	 * @return
	 */
	private ILaunchConfigurationType getConfigType(Object launchable) {
		ILaunchConfigurationType configType = null;
		if (launchable instanceof TPFTest) {
			configType = TestLaunchConfigurationFacade.getLaunchConfigurationType();
		} else if (launchable instanceof TPFTestComponent) {
			configType = TestComponentLaunchConfigurationFacade.getLaunchConfigurationType();
		}
		return configType;
	}

	/**
	 * Return the shell of the active window. Conveniency method.
	 * @return
	 */
	private Shell getActiveShell() {
		IWorkbenchWindow window = TestUIPlugin.getInstance().getWorkbench().getActiveWorkbenchWindow();
		if (window == null) {
			return null;
		}
		return window.getShell();
	}
	
	/**
	 * Return the Debug Plug-in's Launch Manager. Conveniency method.
	 * @return
	 */
	private ILaunchManager getLaunchManager() {
		return DebugPlugin.getDefault().getLaunchManager();
	}

}
