/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ExecutionHistoryTab.java,v 1.9 2005/02/16 22:22:06 qiyanli Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.launch.tabs;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.ui.AbstractLaunchConfigurationTab;
import org.eclipse.hyades.test.ui.TestUIImages;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.internal.launch.extensions.LaunchConfigurationExtensionsManager;
import org.eclipse.hyades.test.ui.internal.launch.providers.ExecutionHistoryLocationProvider;
import org.eclipse.hyades.test.ui.launch.configurations.ExecutionHistoryLaunchConfigurationFacade;
import org.eclipse.hyades.test.ui.launch.extensions.IExecutionResultsLocationFilter;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

/**
 * Provides the Test Execution History Tab. This tab allows the user to choose the name
 * and the location of the Test Execution History file.
 * @author jcanches
 */
public class ExecutionHistoryTab extends AbstractLaunchConfigurationTab {
	
	private TreeViewer tree;
	private Text nameText;
	private Button override;
	protected boolean userHasChosenName;
	protected boolean userHasChosenOverrideState;
	protected boolean userHasChosenLocation;
	protected boolean disableNotification = false;
	private String launchConfigurationType;
	
	public ExecutionHistoryTab(String launchConfigurationType) {
		this.launchConfigurationType = launchConfigurationType;
	}
	
	public void createControl(Composite parent) {
        Composite composite = new Composite(parent, SWT.NULL);
        GridLayout gl = new GridLayout();        
        composite.setLayout(gl);
        composite.setLayoutData(new GridData(GridData.FILL_BOTH));
        composite.setFont(parent.getFont());

        Label lblName = new Label(composite, SWT.NONE);
        GridData gdl = new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING);
        gdl.horizontalSpan = 2;
        lblName.setLayoutData(gdl);
        lblName.setText(TestUIPlugin.getString("ExecutionHistoryTab.name")); //$NON-NLS-1$

        nameText = new Text(composite, SWT.BORDER);
        GridData gdt = new GridData(GridData.FILL_HORIZONTAL);
        gdt.horizontalSpan = 2;
        nameText.setLayoutData(gdt);
        nameText.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				if (!disableNotification) {
					userHasChosenName = true;
					updateLaunchConfigurationDialog();
				}
			}
		});
        
		override = new Button(composite, SWT.CHECK);
		override.setText(TestUIPlugin.getString("ExecutionHistoryTab.override")); //$NON-NLS-1$
		override.setToolTipText(TestUIPlugin.getString("ExecutionHistoryTab.overrideTooltip")); //$NON-NLS-1$
		override.setSelection(false); // By default this check is left unselected
		override.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				if (!disableNotification) {
					userHasChosenOverrideState = true;
					updateLaunchConfigurationDialog();
				}
			}
		});
        GridData gdl3 = new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING);
        gdl3.horizontalSpan = 2;
        override.setLayoutData(gdl3);
        
        Label lblLocation = new Label(composite, SWT.NONE);
        GridData gdl2 = new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING);
        gdl2.horizontalSpan = 2;
        lblLocation.setLayoutData(gdl2);
        lblLocation.setText(TestUIPlugin.getString("ExecutionHistoryTab.location")); //$NON-NLS-1$

        createLocationSelector(composite);

        setControl(composite);
	}
	
    private void createLocationSelector(Composite composite) {
    	GridData grid = new GridData(GridData.FILL_BOTH);
		grid.widthHint	= composite.getClientArea().width/2 ;
		grid.heightHint	= composite.getClientArea().height/2;

        tree = new TreeViewer(composite);
        tree.getControl().setLayoutData(grid);
        IExecutionResultsLocationFilter filter = LaunchConfigurationExtensionsManager.getInstance().getExecutionResultsLocationFilter(this.launchConfigurationType);
        ExecutionHistoryLocationProvider provider = new ExecutionHistoryLocationProvider(filter);
        tree.setContentProvider(provider);
        tree.setLabelProvider(provider);
        tree.setInput(ResourcesPlugin.getWorkspace().getRoot());
        
        tree.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				if (!disableNotification) {
					userHasChosenLocation = true;
					updateLaunchConfigurationDialog();
				}
			}
		});
    }
	
	public void setDefaults(ILaunchConfigurationWorkingCopy configuration) {
		ExecutionHistoryLaunchConfigurationFacade.setUsingDefaultName(configuration, true);
		ExecutionHistoryLaunchConfigurationFacade.setUsingDefaultOverrideState(configuration, true);
        ExecutionHistoryLaunchConfigurationFacade.setUsingDefaultLocation(configuration, true);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#initializeFrom(org.eclipse.debug.core.ILaunchConfiguration)
	 */
	public void initializeFrom(ILaunchConfiguration configuration) {
        try {
    		userHasChosenName = !ExecutionHistoryLaunchConfigurationFacade.isUsingDefaultName(configuration);
    		userHasChosenOverrideState = !ExecutionHistoryLaunchConfigurationFacade.isUsingDefaultOverrideState(configuration);
    		userHasChosenLocation = !ExecutionHistoryLaunchConfigurationFacade.isUsingDefaultLocation(configuration);

    		this.disableNotification = true;
    		try {
	    		String name = ExecutionHistoryLaunchConfigurationFacade.getExecutionHistoryName(configuration);
	    		if (name == null) {
	    			nameText.setText("");
	    		} else {
	    			nameText.setText(name);
	    		}
	    		
	    		boolean overrideState = ExecutionHistoryLaunchConfigurationFacade.isExecutionHistoryOverriden(configuration);
	    		override.setSelection(overrideState);
	    		
	    		IContainer location = ExecutionHistoryLaunchConfigurationFacade.getExecutionHistoryLocation(configuration);
	    		if (location == null) {
	    			tree.setSelection(new StructuredSelection());
	    		} else {
	    			StructuredSelection sel = new StructuredSelection(location);
	    			tree.setSelection(sel, /*reveal*/true);
	    		}
    		} finally {
    			this.disableNotification = false;
    		}
    		
        } catch (CoreException e) {
            TestUIPlugin.logError(e);
        }
	}
	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#performApply(org.eclipse.debug.core.ILaunchConfigurationWorkingCopy)
	 */
	public void performApply(ILaunchConfigurationWorkingCopy configuration) {
		if (userHasChosenName) {
	        String name = getExecutionHistoryName();
	        ExecutionHistoryLaunchConfigurationFacade.setExecutionHistoryName(configuration, name);
		} else {
			ExecutionHistoryLaunchConfigurationFacade.setUsingDefaultName(configuration, true);
		}
        
		if (userHasChosenOverrideState) {
	        boolean overrideState = getOverrideState();
	        ExecutionHistoryLaunchConfigurationFacade.setExecutionHistoryOverriden(configuration, overrideState);
		} else {
			ExecutionHistoryLaunchConfigurationFacade.setUsingDefaultOverrideState(configuration, true);
		}

		if (userHasChosenLocation) {
	        IContainer location = getSelectedLocation();
	        ExecutionHistoryLaunchConfigurationFacade.setExecutionHistoryLocation(configuration, location);
		} else {
			ExecutionHistoryLaunchConfigurationFacade.setUsingDefaultLocation(configuration, true);
		}
	}

	public String getName() {
		return TestUIPlugin.getString("ExecutionHistoryTab.tabName"); //$NON-NLS-1$
	}
	
	public Image getImage() {
		return TestUIImages.INSTANCE.getImage(TestUIImages.IMG_LCONFIG_EXECRESULTS_TAB);
	}

	private String getExecutionHistoryName() {
		return nameText.getText();
	}
	
	public boolean getOverrideState() {
		return override.getSelection();
	}
	
	private IContainer getSelectedLocation() {
		ISelection selection = tree.getSelection();
		if (!selection.isEmpty() && selection instanceof IStructuredSelection) {
			Object obj = ((IStructuredSelection)selection).getFirstElement();
			if (obj instanceof IContainer) {
				return (IContainer)obj;
			}
		}
		return null;
	}
	
    /*
     * @see org.eclipse.debug.ui.ILaunchConfigurationTab#isValid(org.eclipse.debug.core.ILaunchConfiguration)
     */
    public boolean isValid(ILaunchConfiguration configuration) {
        //- Sets the state of the Run button
        setErrorMessage(null);
		
		String name;
		try {
			name = ExecutionHistoryLaunchConfigurationFacade.getExecutionHistoryName(configuration);
		} catch (CoreException e) {
			TestUIPlugin.logError(e);
			name = null;
		}
		if (name == null || name.equals("")) { //$NON-NLS-1$
        	setErrorMessage(TestUIPlugin.getString("_ERROR_ExecutionHistoryTab.invalidName")); //$NON-NLS-1$
        	return false;
        }
		
		IContainer location;
		try {
			location = ExecutionHistoryLaunchConfigurationFacade.getExecutionHistoryLocation(configuration);
		} catch (CoreException e) {
			TestUIPlugin.logError(e);
			location = null;
		}
		if (location == null) {
        	setErrorMessage(TestUIPlugin.getString("_ERROR_ExecutionHistoryTab.invalidLocation")); //$NON-NLS-1$
        	return false;
        }

        return true;      		
    }

}
