/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TestAndDeploymentTab.java,v 1.12 2005/02/16 22:22:06 qiyanli Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.launch.tabs;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.ui.AbstractLaunchConfigurationTab;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.common.testprofile.TPFDeployment;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.models.common.testprofile.TPFTestCase;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.test.ui.TestUIImages;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.internal.launch.extensions.LaunchConfigurationExtensionsManager;
import org.eclipse.hyades.test.ui.internal.launch.providers.DeploymentProvider;
import org.eclipse.hyades.test.ui.internal.launch.providers.TestProvider;
import org.eclipse.hyades.test.ui.internal.model.EMFUtil;
import org.eclipse.hyades.test.ui.launch.configurations.DeploymentLaunchConfigurationFacade;
import org.eclipse.hyades.test.ui.launch.configurations.TestLaunchConfigurationFacade;
import org.eclipse.hyades.test.ui.launch.extensions.IDeploymentsProvider;
import org.eclipse.hyades.test.ui.launch.extensions.ITestLaunchConfigurationValidator;
import org.eclipse.jface.viewers.IElementComparer;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ListViewer;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.List;
import org.eclipse.ui.IWorkbenchPage;
	
/**
 * Provides the Test & Deployment selection Tab used by the basic Hyades Launch Configuration
 * @author jcanches
 */
public class TestAndDeploymentTab extends AbstractLaunchConfigurationTab {
	
	protected TreeViewer testTreeViewer;
	protected ListViewer deploymentListViewer;
	private String mode;
	private String launchConfigurationType;
	
	public TestAndDeploymentTab(String mode, String launchConfigurationType) {
		this.mode = mode;
		this.launchConfigurationType = launchConfigurationType;
	}

	public void createControl(Composite parent) {
        Composite composite = new Composite(parent, SWT.NULL);
        GridLayout gl = new GridLayout();
        gl.numColumns = 2;
        composite.setLayout(gl);
        composite.setLayoutData(new GridData(GridData.FILL_BOTH));
        composite.setFont(parent.getFont());
        
        // First column
        Composite col1 = new Composite(composite, SWT.NULL);
        GridLayout col1l = new GridLayout();
        col1l.numColumns = 1;
        col1.setLayoutData(new GridData(GridData.FILL_BOTH));
        col1.setLayout(col1l);
        
        Label lblTest = new Label(col1, SWT.NONE);
        lblTest.setText(TestUIPlugin.getString("TestAndDeploymentTab.selectTestPrompt")); //$NON-NLS-1$
        
        Control testSelectControl = createTestSelector(col1);

        // Second column
        Composite col2 = new Composite(composite, SWT.NULL);
        GridLayout col2l = new GridLayout();
        col2l.numColumns = 1;
        col2.setLayoutData(new GridData(GridData.FILL_BOTH));
        col2.setLayout(col2l);
        
        Label lblDeployment = new Label(col2, SWT.NONE);
        lblDeployment.setText(TestUIPlugin.getString("TestAndDeploymentTab.selectDeploymentPrompt")); //$NON-NLS-1$

        Control deploymentSelectControl = createDeploymentSelector(col2);
        
        setControl(composite);
	}
	
    private Control createTestSelector(Composite composite) {
    	GridData grid = new GridData(GridData.FILL_BOTH);
		grid.widthHint	= 250 ;
		grid.heightHint	= 150;		

    	testTreeViewer = new TreeViewer(composite);
    	testTreeViewer.getControl().setLayoutData(grid);
    	
    	TestProvider provider = new TestProvider(mode, launchConfigurationType);
        testTreeViewer.setContentProvider(provider);
        testTreeViewer.setLabelProvider(provider);
    	testTreeViewer.setInput(ResourcesPlugin.getWorkspace().getRoot());
    	
    	testTreeViewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				TPFTestSuite deploymentInput = fillDeploymentsList(getSelectedTest());
				TPFDeployment defaultDeployment;
				if (deploymentInput == null) {
					defaultDeployment = null;
				} else {
					IDeploymentsProvider dprovider = LaunchConfigurationExtensionsManager.getInstance().getDeploymentsProvider(deploymentInput);
					defaultDeployment = dprovider.getDefaultDeployment(deploymentInput);
				}
				ISelection selection;
				if (defaultDeployment != null) {
					selection = new StructuredSelection(defaultDeployment);
				} else {
					selection  = new StructuredSelection();
				}
				deploymentListViewer.setSelection(selection);
				updateLaunchConfigurationDialog();
			}
		});
    	
    	return testTreeViewer.getControl();
    }
    
    private Control createDeploymentSelector(Composite composite){
    	GridData grid = new GridData(GridData.FILL_BOTH);
		grid.widthHint	= 250 ;
		grid.heightHint	= 150;
		
    	deploymentListViewer = new ListViewer(composite, SWT.SINGLE | SWT.H_SCROLL | SWT.V_SCROLL | SWT.BORDER);
    	deploymentListViewer.getControl().setLayoutData(grid);
    	
    	DeploymentProvider provider = new DeploymentProvider();
    	deploymentListViewer.setContentProvider(provider);
    	deploymentListViewer.setLabelProvider(provider);
    	deploymentListViewer.setComparer(provider);
    	deploymentListViewer.setInput(null);
    	
    	deploymentListViewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				updateLaunchConfigurationDialog();
			}
		});
    	
    	return deploymentListViewer.getControl();
    }
    
    /**
     * @param test
     * @return The TestSuite that is the input of the deployment list
     */
    protected TPFTestSuite fillDeploymentsList(TPFTest test) {
    	TPFTestSuite deploymentInput = null;
		if (test != null) {
			if (test instanceof TPFTestSuite) {
				deploymentInput = (TPFTestSuite)test;
			} else if (test instanceof TPFTestCase) {
				deploymentInput = ((TPFTestCase)test).getTestSuite();
			}
		}
		TestAndDeploymentTab.this.deploymentListViewer.setInput(deploymentInput);
		return deploymentInput;
    }
	
	public void setDefaults(ILaunchConfigurationWorkingCopy configuration) {
        TestLaunchConfigurationFacade.setTest(configuration, getContext());
	}
	
	protected TPFTest getSelectedTest() {
		ISelection selection = testTreeViewer.getSelection();
		if (!selection.isEmpty() && selection instanceof IStructuredSelection) {
			Object obj = ((IStructuredSelection)selection).getFirstElement();
			if (obj instanceof TPFTest) {
				return (TPFTest)obj;
			}
		}
		return null;
	}
	
	protected TPFDeployment getSelectedDeployment() {
		ISelection selection = deploymentListViewer.getSelection();
		if (!selection.isEmpty() && selection instanceof IStructuredSelection) {
			Object obj = ((IStructuredSelection)selection).getFirstElement();
			if (obj instanceof TPFDeployment) {
				return (TPFDeployment)obj;
			}
		}
		return null;
	}
	
	private void setDeploymentSelection(TPFDeployment deployment) {
		// Because the list viewer's setSelection does not use the list viewer's comparer
		// for finding the element to select, we have to do this ourselves.
		List list = deploymentListViewer.getList();
		list.deselectAll();
		if (deployment == null) return;
		int n = list.getItemCount();
		IElementComparer comparer = deploymentListViewer.getComparer();
		for (int i = 0; i < n; i++) {
			if (comparer.equals(deployment, deploymentListViewer.getElementAt(i))) {
				list.select(i);
				return;
			}
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#initializeFrom(org.eclipse.debug.core.ILaunchConfiguration)
	 */
	public void initializeFrom(ILaunchConfiguration configuration) {
        try {
        	TPFTest test = TestLaunchConfigurationFacade.getTest(configuration, EMFUtil.getResourceSet());
        	if (test != null) {
        		testTreeViewer.setSelection(new StructuredSelection(test), /*reveal*/true);
        		fillDeploymentsList(test);
        		TPFDeployment depl = DeploymentLaunchConfigurationFacade.getDeployment(configuration, EMFUtil.getResourceSet());
        		setDeploymentSelection(depl);
        	} else {
        		testTreeViewer.setSelection(new StructuredSelection());
        		deploymentListViewer.setSelection(new StructuredSelection());
        	}
        } catch (CoreException e) {
            TestUIPlugin.logError(e);
        }
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#performApply(org.eclipse.debug.core.ILaunchConfigurationWorkingCopy)
	 */
	public void performApply(ILaunchConfigurationWorkingCopy configuration) {
        TPFTest test = this.getSelectedTest();
        TestLaunchConfigurationFacade.setTest(configuration, test);
        
        TPFDeployment depl = this.getSelectedDeployment();
        DeploymentLaunchConfigurationFacade.setDeployment(configuration, depl);
	}

	public String getName() {
		return TestUIPlugin.getString("TestAndDeploymentTab.testTabName"); //$NON-NLS-1$
	}
	
	public Image getImage() {
		return TestUIImages.INSTANCE.getImage(TestUIImages.IMG_LCONFIG_TEST_TAB);
	}

    /*
     * @see org.eclipse.debug.ui.ILaunchConfigurationTab#isValid(org.eclipse.debug.core.ILaunchConfiguration)
     */
    public boolean isValid(ILaunchConfiguration launchConfig) {
        //- Sets the state of the Run button
        setErrorMessage(null);
        TPFTest test;
        try {
			test = TestLaunchConfigurationFacade.getTest(launchConfig, EMFUtil.getResourceSet());
		} catch (CoreException e) {
			test = null;
		}
        if (test == null) {
        	setErrorMessage(TestUIPlugin.getString("_ERROR_TestAndDeploymentTab.noTestError")); //$NON-NLS-1$
        	return false;
        }
        if (deploymentListViewer.getList().getItemCount() == 0) {
        	setErrorMessage(TestUIPlugin.getString("_ERROR_TestAndDeploymentTab.noDeploymentAvailable")); //$NON-NLS-1$
        	return false;
        }
        TPFDeployment depl;
        try {
			depl = DeploymentLaunchConfigurationFacade.getDeployment(launchConfig, EMFUtil.getResourceSet());
		} catch (CoreException e1) {
			depl = null;
		}
        if (depl == null) {
        	setErrorMessage(TestUIPlugin.getString("_ERROR_TestAndDeploymentTab.noDeploymentError")); //$NON-NLS-1$
        	return false;
        }
        // Invoke an extension point's validator
        ITestLaunchConfigurationValidator validator = LaunchConfigurationExtensionsManager.getInstance().getLaunchConfigurationValidator(test);
        if (validator != null) {
        	ITestLaunchConfigurationValidator.Diagnostic diag = validator.validate(launchConfig, EMFUtil.getResourceSet());
        	if (diag != null) {
        		setErrorMessage(diag.getMessage());
        		return diag.getSeverity() != ITestLaunchConfigurationValidator.Diagnostic.ERROR;
        	}
        }
        return true;
    }

    /**
     * @return the first test element from the workbench selection
     */
    public static TPFTest getContext() {
        try {
            IWorkbenchPage page = TestUIPlugin.getInstance().getWorkbench().getActiveWorkbenchWindow().getActivePage();
            if (page != null) {
                ISelection selection = page.getSelection();
                if (selection instanceof IStructuredSelection) {
                    IStructuredSelection ss = (IStructuredSelection) selection;
                    Object obj = ss.getFirstElement();
                    if (obj != null) {
                        if (obj instanceof TPFTest) {
                            return (TPFTest)obj;
                        }
                        if (obj instanceof IFile) {
                            IFile file = (IFile) obj;
                            if (ICommonConstants.TEST_SUITE_FILE_EXTENSION.equals(file.getFileExtension())) {
                            	return loadTestSuite(file);
                            }
                        }
                    }
                }
            }
        } catch (Throwable t) {
            TestUIPlugin.logError(t);
        }
        return null;
    }

	private static TPFTestSuite loadTestSuite(IFile resource) {
		EObject[] eobjs = EMFUtil.load(null, resource);
		for (int i = 0; i < eobjs.length; i++) {
			if (eobjs[i] instanceof TPFTestSuite) {
				return (TPFTestSuite)eobjs[i];
			}
		}
		return null;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#activated(org.eclipse.debug.core.ILaunchConfigurationWorkingCopy)
	 */
	public void activated(ILaunchConfigurationWorkingCopy workingCopy) {
		// This tab's content is not dependent on other tabs' content. Nothing
		// to do
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#deactivated(org.eclipse.debug.core.ILaunchConfigurationWorkingCopy)
	 */
	public void deactivated(ILaunchConfigurationWorkingCopy workingCopy) {
		// Everything is applied as soon as selected. Nothing to do
	}
	
}
