/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: EMFUtil.java,v 1.14 2005/03/02 23:41:34 bjiang Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.model;

import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.ecore.xmi.XMLResource;
import org.eclipse.hyades.test.ui.TestUIPlugin;

/**
 * Contains generic utility methods to work with EMF objects.
 * 
 * @author marcelop
 * @since 0.2.0
 */
// TODO to be merged with hyades.models.hierarchy.util.EMFUtil by Marius Slavescu.
public class EMFUtil {
	public static final Map RESOURCE_OPTIONS = new HashMap();
	private static ResourceSet resourceSet = new ResourceSetImpl();
	private static Resource resourceBeingSaved;
	{
		RESOURCE_OPTIONS.put(XMLResource.OPTION_DECLARE_XML, Boolean.TRUE);
		RESOURCE_OPTIONS.put(XMLResource.OPTION_SKIP_ESCAPE, Boolean.FALSE);
	}

	/**
	 * Saves a resource.
	 * @param resource
	 * @param progressMonitor
	 * @throws IllegalArgumentException if the resource is null
	 * @throws Exception if an exception happens during the saving attempt.
	 */
	public static void save(Resource resource) throws Exception {
		if (resource == null)
			throw new IllegalArgumentException(TestUIPlugin.getString("_ERROR_ARG_NULL", "resource")); //$NON-NLS-1$  //$NON-NLS-2$

		boolean isModified = resource.isModified();
		try {
			resourceBeingSaved = resource;
			resource.save(RESOURCE_OPTIONS);
		} catch (Exception e) {
			resource.setModified(isModified);
			throw e;
		} finally {
			resourceBeingSaved = null;
		}
	}

	/** Unload a resource using its URI from the test navigator resource set if this resource is not one we are currently saving
	 *  This method is used in the updateContent of the resource change updater of the test navigator
	 * @param uri the URI of the resource to unload
	 */
	public static void unloadIfNeeded(URI uri) {
		Resource resource = getResourceSet().getResource(uri, false);
		//- we unload the resource from the resource set if this is not a resource we are currently saving
		if(resource != null && resourceBeingSaved == null) {
			resource.unload();
		}
	}

	/**
	 * Returns an EObject from the resource set that has the same URI fragment or
	 * id of the specifie eObject.  If the eObject is not in the resource set or can't
	 * be loaded, this method returns <code>null</code>. If the resource set is <code>null</code>
	 * the default (internal) one is used.
	 * 
	 * @param resourceSet
	 * @param eObject
	 * @param load
	 * @return EObject
	 */
	public static EObject getEObject(ResourceSet rs, EObject eObject, boolean load) {
		ResourceSet localRS = rs;
		if (localRS == null)
			localRS = resourceSet;

		if (eObject == null)
			return null;

		Resource resource = eObject.eResource();
		if (resource == null)
			return null;

		if (localRS.equals(resource.getResourceSet()))
			return eObject;

		String uriFragment = resource.getURIFragment(eObject);
		resource = localRS.getResource(resource.getURI(), load);
		if (resource == null)
			return null;

		eObject = resource.getEObject(uriFragment);
		if (eObject == null)
			return null;

		return eObject;
	}

	/**
	 * Returns the EObjects in the specified file from the Resource Set without 
	 * loading it.
	 * 
	 * <p>If the file is <code>null</code> or if the file is not loaded
	 * into it, this method returns an empty array.
	 * 
	 * @param resourceSet 
	 * @param file
	 * @return EObject[]
	 */
	public static EObject[] getEObjects(ResourceSet rs, IFile file) {
		if (file == null)
			return new EObject[0];

		return getEObjects(rs, file.getFullPath().toString());
	}

	/**
	 * Returns the EObjects in the specified file paht from the Resource Set 
	 * without loading it.
	 * 
	 * <p>If the resource set is <code>null</code> or if the file is not loaded
	 * into it, this method returns an empty array.
	 * 
	 * @param resourceSet 
	 * @param file
	 * @return EObject[]
	 */
	public static EObject[] getEObjects(ResourceSet rs, String fullPath) {
		if (fullPath == null)
			return new EObject[0];

		ResourceSet localRS = rs;
		if (localRS == null)
			localRS = resourceSet;

		Resource resource = localRS.getResource(URI.createPlatformResourceURI(fullPath), false);
		if (resource == null)
			return new EObject[0];

		return (EObject[]) resource.getContents().toArray(new EObject[resource.getContents().size()]);
	}
	
	/**
	 * Returns the EObjects in the resource specified by the <i>fileUIR</i>.
	 * 
	 * @param fileURI
	 * @param load
	 * @return
	 * @author bjiang
	 */
	public static EObject[] getEObjects(URI fileURI, boolean load)
	{
		ResourceSet rs = new ResourceSetImpl();
		Resource res = rs.getResource(fileURI, load);
		EList list = res.getContents();
		EObject[] eObjects = new EObject[list.size()];
		for(int i = 0; i < list.size(); i++)
		{
			eObjects[i] = (EObject)list.get(i);
		}
		return eObjects;
	}	

	/**
	 * Loads the specified file returning all the root EObjects.  If the
	 * resource set argument is <code>null</code> the default one is used.
	 * 
	 * <p>If the load attempt fails this method adds an entry to the
	 * log and returns an empty array.
	 * 
	 * @param resourceSet 
	 * @param file
	 * @return EObject[]
	 */
	public static EObject[] load(ResourceSet rs, IFile file) {
		if (file == null)
			return new EObject[0];

		return load(rs, file.getFullPath().toString());
	}

	/**
	 * Loads the specified file returning all the root EObjects.  If the
	 * resource set argument is <code>null</code> the default one is used.
	 * 
	 * <p>If the load attempt fails this method adds an entry to the
	 * log and returns an empty array.
	 * 
	 * @param resourceSet 
	 * @param fullPath
	 * @return EObject[]
	 */
	public static EObject[] load(ResourceSet rs, String fullPath) {
		if (fullPath == null)
			return new EObject[0];

		URI uri = URI.createPlatformResourceURI(fullPath);
		return load(rs, uri);
	}

	/**
	 * Loads the specified file returning all the root EObjects.  If the
	 * resource set argument is <code>null</code> the default one is used.
	 * 
	 * <p>If the load attempt fails this method adds an entry to the
	 * log and returns an empty array.
	 * 
	 * @param resourceSet 
	 * @param fullPath
	 * @return EObject[]
	 */
	public static EObject[] load(ResourceSet rs, URI uri) {
		if (uri == null)
			return new EObject[0];

		Resource resource = null;
		try {
			if (rs != null) {
				resource = rs.getResource(uri, true);
			} else {
				resource = resourceSet.getResource(uri, true);
				//				Resource.Factory factory = getResourceFactory(FileUtil.getFileExtension(fullPath));
				//				if(factory != null)
				//				{
				//					resource = factory.createResource(uri);
				//					resource.load(RESOURCE_OPTIONS);
				//				}
			}
		} catch (Exception e) {
			TestUIPlugin.logError(e);
		}

		if (resource == null)
			return new EObject[0];

		return (EObject[]) resource.getContents().toArray(new EObject[resource.getContents().size()]);
	}

	/**
	 * Returns the resource factory associated with a given file extension.
	 * If the file extension is <code>null</code> or if there isn't a specifc
	 * factory for the file extension, the this method returns the generic
	 * factory.
	 * @param fileExtension
	 * @return the Resource.Factory or <code>null</code> if none. 
	 */
	public static Resource.Factory getResourceFactory(String fileExtension) {
		if (fileExtension != null) {
			Resource.Factory factory = (Resource.Factory) Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().get(fileExtension);
			if (factory != null)
				return factory;
		}

		return (Resource.Factory) Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().get(Resource.Factory.Registry.DEFAULT_EXTENSION);
	}

	/**
	 * Returns the workspace file that is storing a given eObject or
	 * <code>null</code> if the object is not persisted. 
	 * @param eObject
	 * @return IFile
	 */
	public static IFile getWorkspaceFile(EObject eObject) {
		if (eObject == null)
			return null;
		return getWorkspaceFile(eObject.eResource());
	}

	public static IFile getWorkspaceFile(URI uri) {
		String filePath = getWorkspaceFilePath(uri.trimFragment());
		IWorkspaceRoot workspaceRoot= ResourcesPlugin.getWorkspace().getRoot();
		if (filePath == null)
		{
			return null;
		}
		else
		{
			IResource workspaceResource = workspaceRoot.findMember(filePath);
			if ((workspaceResource == null) || (workspaceResource.getType() != IResource.FILE)) 
			{
				return null;
			}
			return (IFile) workspaceResource;
		}
	}
	
	/**
	 * Returns the workspace file that corresponds to a given resource or
	 * <code>null</code> if the resource is not available in the 
	 * workspace.
	 * @param resource
	 * @return IFile
	 */
	public static IFile getWorkspaceFile(Resource resource) {
		if (resource == null) return null;
		return getWorkspaceFile(resource.getURI());
	}
	
	public static IContainer getParent(Resource resource)
	{
		IFile file = EMFUtil.getWorkspaceFile(resource);
		if(file != null)
		{
			return file.getParent();
		}
		return null;
	}
	/**
	 * Returns file path for resource or empty string if not found
	 * @param eObject
	 * @return String file path
	 */
	public static String getFilePath(Resource resource)
	{
		if (resource == null)
		{
			return "";
		}
		
		IFile file = getWorkspaceFile(resource);
		String filePath = null;
		if (file == null)
		{
			filePath = resource.getURI().toFileString();
		}
		else
		{
			filePath = file.getFullPath().toString();
		}
		if (filePath == null)
		{
			return "";
		}
		return filePath;
	}

	/**
	 * Returns file path for EObject or empty string if not found
	 * @param eObject
	 * @return String file path
	 */
	public static String getFilePath(EObject eObject)
	{
		return getFilePath(eObject.eResource());
	}
	
	
	public static String getWorkspaceFilePath(Resource resource) {
		if (resource == null)
			return null;

		return getWorkspaceFilePath(resource.getURI());
	}

	/**
	 * @param resourceURI
	 * @return
	 */
	private static String getWorkspaceFilePath(URI uri) {
		String resourceURI = uri.toString();
		if (!resourceURI.startsWith("platform:/resource")) //$NON-NLS-1$
			return null;
		return resourceURI.substring("platform:/resource".length()); //$NON-NLS-1$
	}

	/**
	 * Returns a resource without loading it.  If the resource is <code>null</code>,
	 * a new resource is created if a resource factory is available for the file
	 * extension.
	 * @param resourceSet
	 * @param file
	 * @return Resource
	 */
	public static Resource getResource(ResourceSet rs, IFile file) {
		URI uri = URI.createPlatformResourceURI(file.getFullPath().toString());
		return getResource(rs, uri, false);

		//		Resource.Factory factory = getResourceFactory(file.getFileExtension());
		//		if(factory != null)
		//			return factory.createResource(uri);
		//			
		//		return null;
	}

	public static Resource getResource(ResourceSet rs, URI uri, boolean load) {
		ResourceSet localRS = rs;

		if (localRS == null)
			localRS = resourceSet;
		return localRS.getResource(uri, load);
	}

	/**
	 * Reloads a given eObject returning a <b>new</b> instance from the
	 * same resource.
	 * 
	 * <p>This method unloads and reloads the entire eObject's resource
	 * and tries to find the object in the new resource content.
	 * 
	 * <p>Important: after reloading the resource all the objects
	 * in that resource are reinstantiated.  This has a big impact if an object
	 * is being cached somewhere.
	 * 
	 * @param eObject
	 * @return the reloaded eObject
	 */
	public static EObject reload(EObject eObject) {
		if (eObject == null)
			return null;

		Resource resource = eObject.eResource();
		if (resource == null)
			return eObject;

		String uriFragment = resource.getURIFragment(eObject);
		if (resource.isLoaded())
			resource.unload();

		try {
			resource.load(RESOURCE_OPTIONS);
		} catch (IOException e) {
			TestUIPlugin.logError(e);
			return null;
		}

		return resource.getEObject(uriFragment);
	}

	/**
	 * Removes the specified eObject from its container and breaks
	 * all the references to this object within the context of its resource
	 * set. 
	 * @param eObject
	 * @return the changed resources including the eObject's resource.
	 */
	public static Resource[] remove(EObject eObject) {
		if (eObject == null)
			return new Resource[0];

		if (eObject.eResource() == null) {
			EcoreUtil.remove(eObject);
			return new Resource[0];
		}

		Resource resource = eObject.eResource();
		Set changedResources = org.eclipse.hyades.models.hierarchy.util.EMFUtil.delete(eObject, resource.getResourceSet());
		changedResources.add(resource);

		if (eObject.eContainer() != null)
			EcoreUtil.remove(eObject);

		return (Resource[]) changedResources.toArray(new Resource[changedResources.size()]);
	}

	/**
	 * Removes the specified resource from its resource set and breaks
	 * all the references to objects inside this resource within the context 
	 * of the resource set. 
	 * @param resource
	 * @return the changed resources excluding the specified resource. 
	 */
	public static Resource[] remove(Resource resource) {
		if (resource == null)
			return new Resource[0];

		Set changedResources = org.eclipse.hyades.models.hierarchy.util.EMFUtil.delete(resource);
		return (Resource[]) changedResources.toArray(new Resource[changedResources.size()]);
	}

	/**
	 * Adds the specified adapter to the notifiers in the given iterator.
	 * @param objectIterator
	 * @param adapter
	 */
	public static void addAdapter(Iterator objectIterator, Adapter adapter) {
		while (objectIterator.hasNext()) {
			Object object = objectIterator.next();
			if (object instanceof Notifier)
				 ((Notifier) object).eAdapters().add(adapter);
		}
	}

	/**
	 * Removes the specified adapter from the notifiers in the given 
	 * iterator.
	 * @param objectIterator
	 * @param adapter
	 */
	public static void removeAdapter(Iterator objectIterator, Adapter adapter) {
		while (objectIterator.hasNext()) {
			Object object = objectIterator.next();
			if (object instanceof Notifier)
				 ((Notifier) object).eAdapters().remove(adapter);
		}
	}

	/**
	 * Removes all adapters from the notifiers in the given 
	 * iterator.
	 * @param objectIterator
	 */
	public static void removeAdapters(Iterator objectIterator) {
		while (objectIterator.hasNext()) {
			Object object = objectIterator.next();
			if (object instanceof Notifier)
				 ((Notifier) object).eAdapters().clear();
		}
	}
	/**
	 * @return the resource set cached in this class
	 */
	public static ResourceSet getResourceSet() {
		return resourceSet;
	}

	/** Returns the resource which contains (at the first level) the given <code>object</code>.
	 * @param rs resource set used to search in or <code>null</code> if cached one has to be used.
	 * @param object the EMF element we are loking for its resource
	 * @return the resource which contains the looked for object
	 */
	public static Resource getResource(ResourceSet rs, EObject object) {
		ResourceSet localRS = rs;

		if (localRS == null) localRS = resourceSet;
		EList resources = localRS.getResources();
		for(Iterator it1 = resources.iterator(); it1.hasNext(); ) {
			Resource res = (Resource)it1.next();
			EList resContent = res.getContents();
			for(Iterator it2 = resContent.iterator(); it2.hasNext(); ) {
				if(object.equals(it2.next())) return res;
			}
		}
		return null;
	}
	
    public static EObject getEObjectInResourceSet(EObject object, ResourceSet rs) {
    	URI uri = object.eResource().getURI();
    	String uriFragment = object.eResource().getURIFragment(object);
    	Resource res = rs.getResource(uri, /*loadOnDemand*/true);
    	EObject obj = res.getEObject(uriFragment);
    	return obj;
    }

    /** Returns the object resolved by the URI.
	 * @param rs resource set used to search in or <code>null</code> if cached one has to be used.
     * @param uri the URI to resolve.
     * @param loadOnDemand whether to create and load the resource, if it doesn't already exists.
     * @return the object resolved by the URI.
     * @see ResourceSet#getEObject(org.eclipse.emf.common.util.URI, boolean)
     * @author jgout
     */
    public static EObject getEObject(ResourceSet rs, URI uri, boolean loadOnDemand) {
		ResourceSet localRS = rs;

		if (localRS == null) localRS = resourceSet;
		return localRS.getEObject(uri, loadOnDemand);
    }
    
	/**
	 * Returns the complete EMF URI which includes the resource information.
	 * @param object
	 * @return the uri of the given object
	 */
	public static URI getObjectURI(Object object) {
		if (object != null) {
			if (object instanceof EObject) {
				EObject eObject = (EObject) object;
				if (eObject.eResource() != null)
					return eObject.eResource().getURI().appendFragment(eObject.eResource().getURIFragment(eObject));
			}
		}
		return null;
	}

}
