/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: PasteExtensionManager.java,v 1.3 2005/02/16 22:22:06 qiyanli Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.action;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.navigator.actions.IPaster;
import org.eclipse.swt.dnd.Clipboard;

/**
 * @author jgout
 * @since 3.3
 */
public class PasteExtensionManager {
	
	private static PasteExtensionManager instance;
	private List strictPasters;
	private List simplePasters;
	
	static class PasterExtensionInfo {
		private boolean strict;
		private Class target;
		private IPaster paster;
		private IConfigurationElement element;
		
		public PasterExtensionInfo (String target, boolean strict, IConfigurationElement element) {
			this.strict = strict;
			try {
				this.target = Platform.getBundle(element.getDeclaringExtension().getNamespace()).loadClass(target);
			} catch (ClassNotFoundException e) {
				TestUIPlugin.logError(e);
			}
			this.element = element;
			paster = null;
		}
		
		public IPaster getPaster() {
			if(paster == null) {
				try {
					paster = (IPaster)element.createExecutableExtension("class"); //$NON-NLS-1$
				} catch (CoreException e) {
					TestUIPlugin.logError(e);
				}
			}
			return paster;
		}

		public boolean isStrict() {
			return strict;		
		}

		public Class getTarget() {
			return target;
		}
	}
	
	
	public static PasteExtensionManager getInstance() {
		if(instance == null) {
			instance = new PasteExtensionManager();
		}
		return instance;
	}
	
	private PasteExtensionManager() {
		strictPasters = new LinkedList();
		simplePasters = new LinkedList();
		IExtensionPoint extPoint = Platform.getExtensionRegistry().getExtensionPoint(TestUIPlugin.getID() + ".testNavigatorPasteExtension"); //$NON-NLS-1$
		if (extPoint != null) {
			IConfigurationElement[] members = extPoint.getConfigurationElements();
			for (int i = 0; i < members.length; i++) {
				IConfigurationElement element = members[i];
				if ("paster".equals(element.getName())) { //$NON-NLS-1$
					String target = element.getAttribute("target"); //$NON-NLS-1$
					boolean strict = Boolean.getBoolean(element.getAttribute("strict")); //$NON-NLS-1$
					addPasteExtension(target, element, strict);
				}
			}
		}
	}
		
	/**
	 * @param target
	 * @param element
	 * @param strict
	 */
	private void addPasteExtension(String target, IConfigurationElement element, boolean strict) {
		PasterExtensionInfo info = new PasterExtensionInfo(target, strict, element);
		if(strict) {
			strictPasters.add(info);
		} else {
			simplePasters.add(info);
		}
	}
	
	/** Returns whether the given object has been registered to be a valid paste target
	 * @param clipboard the clipboard wherein copied objects have been potentially stored.
	 * @param selection target selection where the paste will occure
	 * @return <code>true</code> if the paste action can be performed in the context of selection and clipboard content and <code>false</code> otherwise.
	 */
	public boolean isPasteAllowedFor(Clipboard clipboard, Object selection) {
		IPaster [] pasters = getPasters(selection);
		for (int i = 0; i < pasters.length; i++) {
			IPaster paster = pasters[i];
			//- delegates to client, stop as soon as one client is ok to deal (azccording to the context)
			if(paster != null && paster.isPasteAllowedFor(clipboard, selection)) {
				return true;
			}			
		}
		return false;
	}
	
	/** Retruns all registered paster that can deal with the given target object.
	 * 
	 * @param selection the current selection (location where the user wants to paste)
	 * @return IPaster [] the array of pasters registered for the given selection.
	 */
	public IPaster [] getPasters(Object selection) {
		if(selection != null) {
			List pasters = new LinkedList();
			Class target = selection.getClass();
			//- priority is given to strict extensions
			for (Iterator it = strictPasters.iterator(); it.hasNext();) {
				PasterExtensionInfo info = (PasterExtensionInfo) it.next();
				if(target == info.getTarget()) {
					pasters.add(info.getPaster());
				}
			}
			for (Iterator it = simplePasters.iterator(); it.hasNext();) {
				PasterExtensionInfo info = (PasterExtensionInfo) it.next();
				if(info.getTarget().isAssignableFrom(target)) {
						pasters.add(info.getPaster());
				}
			}
			return (IPaster[]) pasters.toArray(new IPaster[pasters.size()]);
		} else {
			return new IPaster[0];
		}
	}
}
