/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: FileProxyManager.java,v 1.8 2005/05/11 08:56:35 dguilbaud Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.proxy;

import java.io.StringReader;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Set;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.test.ui.TestUIConstants;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.internal.model.EMFUtil;
import org.eclipse.hyades.test.ui.navigator.IFileProxyFactory;
import org.eclipse.hyades.test.ui.navigator.IPersistableFileProxyFactory;
import org.eclipse.hyades.test.ui.navigator.IPersistableProxyNode;
import org.eclipse.hyades.test.ui.navigator.IPersistableTypedElementProxyFactory;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.hyades.test.ui.navigator.ITypedElementProxyFactory;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.XMLMemento;

/**
 * @author jgout
 * @since 3.2
 */
public class FileProxyManager {
	
	private static final String TAG_PROXY_STATE = "proxyState"; //$NON-NLS-1$
	private static final String TAG_LAST_SAVE_STAMP = "lastSaveStamp"; //$NON-NLS-1$
	private static final String MARKER_PROXYSTATE = "org.eclipse.hyades.test.ui.proxyStateMarker"; //$NON-NLS-1$
	private static FileProxyManager instance;
	private HashMap proxies;
	
	public static FileProxyManager getInstance() {
		if(instance == null) {
			instance = new FileProxyManager();
		}
		return instance;
	}
	
	private FileProxyManager() {
		proxies = new HashMap();
	}
		
	/** Returns the proxy node corresponding to the given file.
	 *  The proxy is stored locally to avoid multiple creation.
	 * @param file the file to convert
	 * @return the IProxyNode matching with the given file or null if no factory can handle this file
	 */
	public IProxyNode getProxy(IFile file) {
		if(isaKnownFile(file)) {
			//- this file has already been converted, thus return the proxy is stored in the map
			return (IProxyNode)proxies.get(file);
		} else {
			//- try to load the proxy from the file
			IProxyNode proxy = loadProxyState(file);
			if(proxy == null) {
				//- load fails, so build it
				proxy = updateProxy(file);
			}
			return proxy;
		}
	}

	private IProxyNode loadProxyState(IFile file) {
		try {
			//- check first that the file still exists
			if(!file.exists()) {
				return null;
			}
			IMarker[] markers = file.findMarkers(MARKER_PROXYSTATE, false, IResource.DEPTH_ZERO);//$NON-NLS-1$
			if(markers.length > 0) {
				//- check if this file has not changed since the last save
				long currentModStamp = file.getModificationStamp();
				String stamp = markers[0].getAttribute(TAG_LAST_SAVE_STAMP, null);
				long lastSaveStamp = stamp != null ? Long.parseLong(stamp) : 0;
				//- if this file has changed since last save do not read the saved state
				if(currentModStamp != lastSaveStamp) return null;
				//- marker exists so retrieve factory id and proxy state
				String factoryID = markers[0].getAttribute(TestUIConstants.TAG_FACTORY_ID, null);
				String proxyState = markers[0].getAttribute(TAG_PROXY_STATE, null);//$NON-NLS-1$
				//- get modification stamp from marker
				//- compare this stamp with file.modicaficationStamp()
				StringReader reader = new StringReader(proxyState);
				IMemento memento = XMLMemento.createReadRoot(reader);
				//- build the proxy using the factory known by its ID
				IProxyNode proxy = buildProxy(file, factoryID, memento);
				//- register this file and its proxy in the map
				if(proxy != null) {
					proxies.put(file, proxy);
				}
				return proxy;
			} else {
				return null;
			}
		} catch (CoreException e) {
			TestUIPlugin.logError(e);
			//- create the proxy as if it was never been seen before
			return null;
		}	
	}
	
	private IProxyNode buildProxy(IFile file, String factoryID, IMemento memento) {
		//- try first to get a File Factory form this id
		IFileProxyFactory factory = FileFactoryManager.getInstance().getFactory(factoryID);
		if(factory != null) {
			if(factory instanceof IPersistableFileProxyFactory) {
				try {
					return ((IPersistableFileProxyFactory)factory).recreate(file, memento);
				} catch (IllegalArgumentException e) {
					//- problem encountered during load
					return null;
				}
			} else {
				//- saved marker contains id of a factory which can not recreate elements
				TestUIPlugin.logError("The factory id: "+factoryID+" should be instanceof IPersistableFileProxyFactory");  //$NON-NLS-1$//$NON-NLS-2$
				return null;
			}
		} else {
			//- this is not a IPersistableFileProxyFactory, try now IPersistableTypedElementProxyFactory
			ITypedElementProxyFactory fact = TypedElementFactoryManager.getInstance().getFactoryFromID(factoryID);
			if(fact != null) {
				if(fact instanceof IPersistableTypedElementProxyFactory) {
					try {
						return ((IPersistableTypedElementProxyFactory)fact).recreate(memento, file.getParent());
					} catch (IllegalArgumentException e) {
						//- problem encountered during load
						return null;
					}
				} else {
					//- saved marker contains id of a factory which can not recreate elements
					TestUIPlugin.logError("The factory id: "+factoryID+" should be instanceof IPersistableTypedElementProxyFactory");  //$NON-NLS-1$//$NON-NLS-2$
					return null;							
				}
			} else {
				//- saved marker contains an unknown id of a factory
				TestUIPlugin.logError("The persisted proxy for file: "+file.getName()+" contains an unknown factory ID");  //$NON-NLS-1$//$NON-NLS-2$
				return null;
			}
		}
	}

	/**
	 * @param file
	 * @param proxy
	 */
	private void saveProxyState(IFile file, IPersistableProxyNode proxy) {
		StringWriter writer = new StringWriter();
		IMemento memento = XMLMemento.createWriteRoot("proxyStateMemento");  //$NON-NLS-1$
		proxy.saveState(memento);
		try {
			((XMLMemento)memento).save(writer);
		} catch (Exception e) {
			TestUIPlugin.logError(e);
		}
			
		class UpdateMarkerJob extends Job {
			private IFile _file;
			private IPersistableProxyNode _proxy;
			private StringWriter _writer;
			
			public UpdateMarkerJob(IFile _file, IPersistableProxyNode _proxy, StringWriter _writer) {
				super("update marker for file proxy"); //$NON-NLS-1$
				this._file = _file;
				this._proxy = _proxy;
				this._writer = _writer;
			}
			
			protected IStatus run(IProgressMonitor monitor) {
				//- before creating the marker remove old one if exists
				try {
					_file.deleteMarkers(MARKER_PROXYSTATE, false, IResource.DEPTH_ZERO);
					//- create a file marker to store the proxy state
					IMarker marker = _file.createMarker(MARKER_PROXYSTATE);
					marker.setAttribute(TAG_LAST_SAVE_STAMP, Long.toString(_file.getModificationStamp()));
					marker.setAttribute(TestUIConstants.TAG_FACTORY_ID, _proxy.getFactoryID());
					marker.setAttribute(TAG_PROXY_STATE, _writer.toString());
					return new Status(IStatus.OK, TestUIPlugin.getID(), 0,  "ok", null); //$NON-NLS-1$
				} catch (CoreException e) {
					TestUIPlugin.logError(e);
					return new Status(IStatus.ERROR, TestUIPlugin.getID(), 0,"unable to update marker for file: "+_file.getName() , e); //$NON-NLS-1$
				}
			}
		}
		Job job = new UpdateMarkerJob(file, proxy, writer);
		job.setRule(file);
		job.schedule();
	}

	/** Returns whether the given file is already in the local proxy data base. 
	 * @param file
	 * @return true if the given file exists as proxy and false otherwise
	 */
	public boolean isaKnownFile(IFile file) {
		return proxies.containsKey(file);
	}

	/** Computes a new proxy using registered file proxy factories.
	 *  This method updates the cached proxy for the given file 
	 * @param file the file from which the proxy is derived.
	 * @return the proxy associated to the given file.
	 */
	public IProxyNode updateProxy(IFile file) {
		IProxyNode proxy = null;
		//- get all factories (FileProxyFactory) for this file element
		ArrayList factories = FileFactoryManager.getInstance().getFactories(file.getFileExtension());
		//- iterate until one can find a correct factory
		for (Iterator it = factories.iterator(); it.hasNext();) {
			IFileProxyFactory factory = (IFileProxyFactory) it.next();
			proxy = factory.create(file);
			if (proxy != null) {
				break;
			}
		}
		//- register this file and its proxy in the map
		if(proxy != null) {
			proxies.put(file, proxy);
		}
		//- save in the file marker the newly created proxy state if needed
		if (proxy instanceof IPersistableProxyNode) {
			saveProxyState(file, (IPersistableProxyNode)proxy);
		}
		return proxy;
	}
	
	/** Returns the proxy which has the given uid. This search starts from the given proxy node and cross its complete sub tree of proxies.
	 * 
	 * @param proxy root of the proxy tree where the search is done. 
	 * @param uid the uid of the searched proxy.
	 * @return a proxy node which has the given uid or <code>null</code> if there is no such proxy found in the complete given tree of proxy.
	 */
	public IProxyNode findProxyByID(IProxyNode proxy, String uid) {
		if(proxy == null) return proxy;
		//- only for root node that has no fragment part
		if(uid.length() > 0) {
			//- proxy found ?
			if(proxy.getIdentifier().equals(uid)) return proxy;
			else {
				//- search in its children array
				IProxyNode[] children = proxy.getChildren();
				for (int i = 0; i < children.length; i++) {
					proxy = findProxyByID(children[i], uid);
					if(proxy != null) {
						//- found !!
						return proxy;
					}
				}
				//- not found in this sub tree
				return null;
			}
		} else return proxy;
	}

	/** Removes the proxy entry associated to the given file.
	 * @param file
	 */
	public IProxyNode removeProxy(IFile file) {
		return (IProxyNode)proxies.remove(file);
	}
		
	/** Retrieves the node associated to the given EMF object. 
	 * This parameter is supposed been translated in a proxy.
	 * 
	 * @param object a EMF instance.
	 * @return the proxy associated to the given EMF instance or <code>null</code> if such proxy node does not exist.
	 */
	public IProxyNode getCorrespondingProxy(EObject object) {
		IFile file = EMFUtil.getWorkspaceFile(object);
		if (file == null) return null;
		IProxyNode fileProxy = FileProxyManager.getInstance().getProxy(file);
		return FileProxyManager.getInstance().findProxyByID(fileProxy, EMFUtil.getObjectURI(object).fragment());
	}
	

	/** Retrieves the node using its underlying resource and its identifier. 
	 * 
	 * @param fileName name of the underlying resource of the searched proxy
	 * @param identifier identifier of the searched proxy
	 * @return a proxy node or <code>null</code> if not found.
	 */
	public IProxyNode getCorrespondingProxy(String fileName, String identifier) {
		Set keys = proxies.keySet();
		IProxyNode root = null;
		for (Iterator it = keys.iterator(); it.hasNext();) {
			IFile file = (IFile) it.next();
			if(fileName.equals(file.getLocation().toOSString())) {
				root = (IProxyNode)proxies.get(file);
				break;
			}
		}
		if(root != null) {
			return FileProxyManager.getInstance().findProxyByID(root, identifier);
		}
		return null;
	}
}
