/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DatapoolNewWizard.java,v 1.15 2005/04/04 22:22:03 bjiang Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.wizard;

import java.io.IOException;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.hyades.edit.datapool.IDatapool;
import org.eclipse.hyades.edit.datapool.IDatapoolEquivalenceClass;
import org.eclipse.hyades.models.common.datapool.DPLDatapool;
import org.eclipse.hyades.models.common.util.DatapoolUtil;
import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.test.ui.TestUI;
import org.eclipse.hyades.test.ui.TestUIExtension;
import org.eclipse.hyades.test.ui.TestUIImages;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.datapool.internal.util.CSVImportExportUtil;
import org.eclipse.hyades.test.ui.datapool.internal.util.CorruptCSVFileException;
import org.eclipse.hyades.test.ui.internal.model.EMFUtil;
import org.eclipse.hyades.ui.internal.util.UIUtil;
import org.eclipse.hyades.ui.internal.wizard.HyadesNewWizard;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.ui.PartInitException;

import sun.io.MalformedInputException;

/**
 * Extends the HyadesNewWizard to create the Hyades Generic DataPool.
 * 
 * @author psun
 * @since 0.0.1
 */
public class DatapoolNewWizard extends HyadesNewWizard {

	DatapoolCSVImportPage csvImportPage = null;
	
	/**
	 * Constructor for DataPoolNewWizard
	 */
	public DatapoolNewWizard()
	{
		super();
		setWindowTitle(TestUIPlugin.getString("WIZ_DATAPOOL_TTL")); //$NON-NLS-1$
		setDefaultPageImageDescriptor(TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_WIZBAN_NEWDATAPOOL));

		IDialogSettings dialogSettings = TestUIPlugin.getInstance().getDialogSettings();
		IDialogSettings settings = dialogSettings.getSection("DataPoolNewWizard"); //$NON-NLS-1$
		if(settings == null)
			settings = dialogSettings.addNewSection("DataPoolNewWizard"); //$NON-NLS-1$
		setDialogSettings(settings);
	}

	/**
	 * Returns the type for the datapool created by this wizard.
	 * @return String
	 */
	protected String getType()
	{
		return TestUIExtension.GENERIC_DATAPOOL_TYPE;
	}
	
	/**
	 * @see org.eclipse.hyades.ui.internal.wizard.HyadesNewWizard#getFileExtension()
	 */
	protected String getFileExtension()
	{
		return ICommonConstants.DATAPOOL_FILE_EXTENSION;
	}

	/**
	 * @see org.eclipse.jface.wizard.IWizard#addPages()
	 */
	public void addPages()
	{
		getLocationPage().setTitle(TestUIPlugin.getString("WIZ_DATAPOOL_GEN_PG_TTL")); //$NON-NLS-1$
		getLocationPage().setDescription(TestUIPlugin.getString("WIZ_DATAPOOL_GEN_PG_LOC_DSC")); //$NON-NLS-1$
		addPage(getLocationPage());
		
		getAttributeWizardPage().setTitle(TestUIPlugin.getString("WIZ_DATAPOOL_GEN_PG_TTL")); //$NON-NLS-1$
		getAttributeWizardPage().setDescription(TestUIPlugin.getString("WIZ_DATAPOOL_GEN_PG_ATT_DSC")); //$NON-NLS-1$
		addPage(getAttributeWizardPage());
		
		csvImportPage = new DatapoolCSVImportPage("CSVImport");
		csvImportPage.setTitle(TestUIPlugin.getString("WIZ_DATAPOOL_CSV_PG_TTL"));
		csvImportPage.setDescription(TestUIPlugin.getString("WIZ_DATAPOOL_CSV_PG_DSC"));
		addPage(csvImportPage);
	}
	
	/**
	 * @see org.eclipse.hyades.ui.test.internal.wizard.HyadesNewWizard#createObject(org.eclipse.core.resources.IFile)
	 */
	protected boolean createObject(IFile file)
	throws Exception
	{
		Resource resource = createResource(file);
		EObject eObject = createEObject();
		if(eObject == null)
			return false;
		resource.getContents().add(eObject);
		EMFUtil.save(resource);
		return true;
	}
	
	/**
	 * Creates the EObject setting the attributes based on the information
	 * the user has defined in this wizard's pages.
	 * @return a not null EObject
	 */
	protected EObject createEObject()
	{
		DPLDatapool datapool = (DPLDatapool)DatapoolUtil.createNewDatapool(getLocationPage().getItemName(), getAttributeWizardPage().getItemDescription());
		
		String csvFileName = csvImportPage.getCSVFileName();
		boolean isRowVariableNameType = csvImportPage.firstRowContainsVariableNameType();
		boolean isColEqClsName = csvImportPage.firstColumnContainsEquivalenceClassName();
		String importEncoding = csvImportPage.getImportEncoding();
		if(csvFileName.length() != 0)
		{
			try
			{
				CSVImportExportUtil.getInstance().importCSV((IDatapool)datapool, csvFileName, isRowVariableNameType, isColEqClsName, importEncoding);
			}
			catch(IOException e)
			{
				Status status = null;
				if(e instanceof MalformedInputException)
					status = new Status(IStatus.ERROR,
										ResourcesPlugin.PI_RESOURCES,
										IStatus.OK,
										TestUIPlugin.getString("_ERROR_WIZ_DATAPOOL_CSV_ERRDLG_CORRUPT"),
										null);
				else
					status = new Status(IStatus.ERROR,
										ResourcesPlugin.PI_RESOURCES,
										IStatus.OK,
										e.getMessage(),
										null);
				new ErrorDialog(getShell(), 
								TestUIPlugin.getString("WIZ_DATAPOOL_CSV_ERRDLG_TITLE"),
								TestUIPlugin.getString("_ERROR_WIZ_DATAPOOL_CSV_ERRDLG_IMPFILE"),
								status,
								IStatus.ERROR).open();
				datapool = null;
			}
			catch(CorruptCSVFileException e)
			{
				Status status = new Status(IStatus.ERROR,
											ResourcesPlugin.PI_RESOURCES,
											IStatus.OK,
											e.getMessage(),
											null);
				new ErrorDialog(getShell(), 
								TestUIPlugin.getString("WIZ_DATAPOOL_CSV_ERRDLG_TITLE"),
								TestUIPlugin.getString("_ERROR_WIZ_DATAPOOL_CSV_ERRDLG_CORRUPT"),
								status,
								IStatus.ERROR).open();
				datapool = null;
			}
		}
		else
			//spinner is available in SWT 3.1, initial dimension option implemented in TPTP 4.0.
			createDatapoolInitialDimension(datapool, 1, 1);
		
		return datapool;
	}
	
	protected void createDatapoolInitialDimension(IDatapool datapool, int numOfVariables, int numOfRecords)
	{
	    for(int i = 0; i < numOfVariables; i++)
	        datapool.appendVariable(datapool.constructVariable());
		
	    IDatapoolEquivalenceClass eqClass;
		if(datapool.getEquivalenceClassCount() > 0)
		{
			int index = datapool.getDefaultEquivalenceClassIndex();
		    if(index < 0 || index >= datapool.getEquivalenceClassCount())
		    {
		    	index = 0;
			    datapool.setDefaultEquivalenceClassIndex(index);
		    }
		    eqClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(index);
		}
		else
		{
		    eqClass = datapool.constructEquivalenceClass();
		    datapool.appendEquivalenceClass(eqClass);
		    datapool.setDefaultEquivalenceClassIndex(0);
		}
		
		for(int i = 0; i < numOfRecords; i++)
		    eqClass.appendRecord(eqClass.constructRecord());
	}
	
	/**
	 * Creates the EMF resource in which the object is added to.
	 * @return a not null Resource
	 */
	protected Resource createResource(IFile file)
	{
		URI uri = URI.createPlatformResourceURI(file.getFullPath().toString());
		Resource.Factory factory = EMFUtil.getResourceFactory(ICommonConstants.DATAPOOL_FILE_EXTENSION);
		return factory.createResource(uri);
	}

		
	/**
	 * @see org.eclipse.hyades.ui.internal.wizard.HyadesNewWizard#openEditor(org.eclipse.core.resources.IFile)
	 */
	protected void openEditor(IFile file) 
	throws PartInitException
	{
		if(TestUIPlugin.getInstance().getPreferenceStore().getBoolean(TestUI.OPEN_EDITOR))
			UIUtil.openEditor(file, TestUIExtension.DATAPOOL_EDITOR_PART_ID, false);
	}
}
