/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TestCaseDefinitionPage.java,v 1.5 2005/02/16 22:22:12 qiyanli Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.wizard;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.hyades.models.common.facades.behavioral.ITestSuite;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.test.ui.TestUIExtension;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.internal.navigator.TestSuiteHTNConverter;
import org.eclipse.hyades.test.ui.wizard.ITestCaseWizard;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.extension.IAssociationDescriptor;
import org.eclipse.hyades.ui.extension.IAssociationMapping;
import org.eclipse.hyades.ui.extension.ITypeValidator;
import org.eclipse.hyades.ui.internal.wizard.selection.IWizardElement;
import org.eclipse.hyades.ui.internal.wizard.selection.WizardElement;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.ui.JavaElementLabelProvider;
import org.eclipse.jdt.ui.StandardJavaElementContentProvider;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.jface.window.Window;
import org.eclipse.jface.wizard.IWizardPage;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.dialogs.ElementTreeSelectionDialog;
import org.eclipse.ui.dialogs.ISelectionStatusValidator;
import org.eclipse.ui.views.navigator.ResourceSorter;

/**
 * A page to define common attributes for the Test Case generation.
 * 
 * @author pnedelec
 * @since 3.0.0
 */
public class TestCaseDefinitionPage extends WizardPage {
    private Text TSNameText;
    private ITestSuite TSElem;

    /**
     * Constructor
     */
    public TestCaseDefinitionPage(String pageName, IWorkbench wkbench, IStructuredSelection selection) {
        super(pageName);
        this.setTitle(TestUIPlugin.getString("WIZ_TST_CASE_DESC_PG_TTL")); //$NON-NLS-1$
        this.setDescription(TestUIPlugin.getString("WIZ_TST_CASE_DESC_PG_DSC")); //$NON-NLS-1$

        init(selection);
    }

    private void init(IStructuredSelection selection) {
        Object obj = selection.getFirstElement();
        if (obj instanceof ITestSuite) {
            setTestSuite((ITestSuite) obj);
        } else if (obj instanceof IResource) {
            Object ts = new TestSuiteHTNConverter().getObjectFromResource((IResource) obj);
            if (ts instanceof ITestSuite) {
                setTestSuite((ITestSuite) ts);
            }
        }
    }

    /**
     * @see org.eclipse.jface.dialogs.IDialogPage#createControl(org.eclipse.swt.widgets.Composite)
     */
    public void createControl(Composite parent) {
        Composite composite = new Composite(parent, SWT.NONE);
        GridLayout layout = new GridLayout();
        layout.numColumns = 3;
        composite.setLayout(layout);

        createTestSuiteControl(composite);

        setPageComplete(validatePage());
        setControl(composite);
    }

    private void createTestSuiteControl(Composite composite) {
        Label label = new Label(composite, SWT.LEFT | SWT.WRAP);
        label.setFont(composite.getFont());
        label.setText(TestUIPlugin.getString("WIZ_SEL_TST_SUITE")); //$NON-NLS-1$
        GridData gd = new GridData();
        label.setLayoutData(gd);

        TSNameText = new Text(composite, SWT.SINGLE | SWT.BORDER);
        TSNameText.setText(getTestSuite() == null ? "" : getTestSuite().getName()); //$NON-NLS-1$
        TSNameText.setEnabled(false);
        TSNameText.setFont(composite.getFont());
        gd = new GridData();
        gd.horizontalAlignment = GridData.FILL;
        gd.grabExcessHorizontalSpace = true;
        TSNameText.setLayoutData(gd);
        TSNameText.addModifyListener(new ModifyListener() {
            public void modifyText(ModifyEvent e) {
                setPageComplete(validatePage());
            }
        });

        Button browse = new Button(composite, SWT.PUSH);
        browse.setText(TestUIPlugin.getString("BROWSE")); //$NON-NLS-1$
        browse.addSelectionListener(new SelectionListener() {
            public void widgetSelected(SelectionEvent e) {
                browseTS();
            }

            public void widgetDefaultSelected(SelectionEvent e) {
            }
        });
    }

    private class TestSuiteValidator implements ISelectionStatusValidator {
        private final IStatus errorStatus = new Status(IStatus.ERROR, TestUIPlugin.getID(), IStatus.ERROR, TestUIPlugin.getString("_ERROR_WIZ_SEL_INVALID"), null); //$NON-NLS-1$
        private final IStatus okStatus = new Status(IStatus.OK, TestUIPlugin.getID(), IStatus.OK, "", null); //$NON-NLS-1$
        private TestSuiteHTNConverter converter;

        public TestSuiteValidator() {
            this.converter = new TestSuiteHTNConverter();
        }

        public IStatus validate(Object[] selection) {
            int selectionLength = selection.length;
            if (selectionLength > 1) { return errorStatus; }
            for (int i = 0; i < selectionLength; i++) {
                if (selection[i] instanceof IResource) {
                    Object obj = converter.getObjectFromResource((IResource) selection[i]);
                    if (obj != null && obj instanceof TPFTestSuite) { return okStatus; }
                }
            }
            return errorStatus;
        }
    }

    private class TestSuiteFilter extends ViewerFilter {
        public boolean select(Viewer viewer, Object parentElement, Object element) {
            if (element instanceof IPackageFragmentRoot) {
                return !((IPackageFragmentRoot) element).isArchive();
            } else if (element instanceof IFile) {
                Object obj = new TestSuiteHTNConverter().getObjectFromResource((IFile) element);
                return (obj != null && obj instanceof TPFTestSuite);
            }
            return true;
        }
    }

    /**
     * Behaviors after clicking the Browse button
     */
    protected void browseTS() {
        ILabelProvider lp = new JavaElementLabelProvider(JavaElementLabelProvider.SHOW_DEFAULT);
        ITreeContentProvider cp = new StandardJavaElementContentProvider();
        String title = TestUIPlugin.getString("WIZ_SEL_TST_SUITE_DLG_TTL"); //$NON-NLS-1$
        String message = TestUIPlugin.getString("WIZ_SEL_TST_SUITE_DLG_MSG"); //$NON-NLS-1$

        ElementTreeSelectionDialog dialog = new ElementTreeSelectionDialog(getShell(), lp, cp);
        dialog.setValidator(new TestSuiteValidator());
        dialog.addFilter(new TestSuiteFilter());
        dialog.setTitle(title);
        dialog.setMessage(message);
        dialog.setInput(JavaCore.create(ResourcesPlugin.getWorkspace().getRoot()));
        dialog.setSorter(new ResourceSorter(ResourceSorter.NAME));

        ITestSuite ts = null;
        if (dialog.open() == Window.OK) {
            Object[] objects = dialog.getResult();
            if (objects[0] instanceof IResource) {
                TestSuiteHTNConverter converter = new TestSuiteHTNConverter();
                Object obj = converter.getObjectFromResource((IResource) objects[0]);
                if (obj != null && obj instanceof ITestSuite) {
                    ts = (ITestSuite) obj;
                    setTestSuite(ts);
                }
            }
        }
        if (ts != null) {
            setText(ts.getName());
        }
    }

    private void setText(String name) {
        if (TSNameText != null) TSNameText.setText(name);
    }

    /**
     * @return true if the page is valid
     */
    protected boolean validatePage() {
        return (getTestSuite() != null);
    }

    /**
     * @see org.eclipse.jface.wizard.IWizardPage#canFlipToNextPage()
     */
    public boolean canFlipToNextPage() {
        //when the super is called, the getNextPage() is called...
        return isPageComplete();
    }

    /**
     * @see org.eclipse.jface.wizard.IWizardPage#getNextPage()
     */
    public IWizardPage getNextPage() {
        IWizardPage page = super.getNextPage();
        if (page instanceof TestCaseSelectionPage) {
            final TestCaseSelectionPage selPage = (TestCaseSelectionPage) page;
            if (isCurrentPage()) {
                HashMap testCaseTypesByTestSuiteType = new HashMap();
                ITestSuite testSuite = getTestSuite();
                if (testSuite == null) return page;

                String testSuiteType = testSuite.getType();
                if (testSuiteType == null) return page;

                Set testCaseTypes = new HashSet();
                testCaseTypesByTestSuiteType.put(testSuiteType, testCaseTypes);

                IAssociationMapping testSuiteTypeValidatorMapping = TestUIExtension.getTestSuiteMappingRegistry().getAssociationMapping(HyadesUIPlugin.EP_TYPE_VALIDATORS);
                IAssociationDescriptor[] typeValidatorDescriptors = testSuiteTypeValidatorMapping.getAssociationDescriptors(testSuiteType);
                if (typeValidatorDescriptors.length == 0) super.getNextPage();

                IAssociationMapping testCaseWizardMapping = TestUIExtension.getTestCaseMappingRegistry().getAssociationMapping(TestUIPlugin.EP_TEST_CASE_WIZARD);
                String[] mappingTypes = testCaseWizardMapping.getTypes();

                for (int i = 0, maxi = mappingTypes.length; i < maxi; i++) {
                    for (int j = 0, maxj = typeValidatorDescriptors.length; j < maxj; j++) {
                        ITypeValidator typeValidator = (ITypeValidator) typeValidatorDescriptors[j].createImplementationClassInstance();
                        if (typeValidator.isValidType(mappingTypes[i])) {
                            testCaseTypes.add(mappingTypes[i]);
                            break;
                        }
                    }

                }
                List wizardElements = new ArrayList(testCaseTypes.size());
                for (Iterator i = testCaseTypes.iterator(); i.hasNext();) {
                    String testCaseType = (String) i.next();
                    IAssociationDescriptor[] wizardDescriptors = testCaseWizardMapping.getAssociationDescriptors(testCaseType);
                    for (int j = 0, maxj = wizardDescriptors.length; j < maxj; j++) {
                        Object wizard = wizardDescriptors[j].createImplementationClassInstance();
                        if (wizard instanceof ITestCaseWizard) {
                            WizardElement wizardElement = new WizardElement((ITestCaseWizard) wizard);
                            wizardElement.setId(wizardDescriptors[j].getId());
                            wizardElement.setLabel(wizardDescriptors[j].getName());
                            wizardElement.setDescription(wizardDescriptors[j].getDescription());
                            wizardElement.setImageDescriptor(wizardDescriptors[j].getImageDescriptor());

                            wizardElements.add(wizardElement);
                        }
                    }
                }
                selPage.setTestSuite(getTestSuite());
                selPage.setWizardElements((IWizardElement[]) wizardElements.toArray(new IWizardElement[wizardElements.size()]));
            }
        }
        return page;
    }

    /**
     * @return Returns the tSElem.
     */
    public ITestSuite getTestSuite() {
        return this.TSElem;
    }

    /**
     * @param elem
     *            The tSElem to set.
     */
    public void setTestSuite(ITestSuite elem) {
        this.TSElem = elem;
    }
}
