/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: EObjectProxyNode.java,v 1.5 2005/02/16 22:22:14 qiyanli Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.navigator;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.Platform;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.hyades.test.ui.TestUIConstants;
import org.eclipse.hyades.test.ui.internal.model.EMFUtil;
import org.eclipse.ui.IMemento;

/** EMF object proxy element for the test navigator.
 * Subclasses should provide IProxyNode methods implementation.
 * EMF objects wrapped by subclasses will be stored in the test navigator resource set
 * @author jgout
 */
public abstract class EObjectProxyNode implements IProxyNode {
	private URI originatorURI;
	private Object parent;
	
	public EObjectProxyNode(IMemento memento, Object parent) {
		String uri = memento.getString(TestUIConstants.TAG_URI);
		if(uri == null) throw new IllegalArgumentException("Malformed saved proxy state: unable to retrieve uri field"); //$NON-NLS-1$
		else {
			this.originatorURI = URI.createURI(uri);
		}
		this.parent = parent;
	}
	
	/**
	 *  Creates a proxy node from a emf object.
	 *  CAUTION: this object should be stored in a EMF resource in order to get its URI
	 * @param eObject the originator object of the proxy
	 * @param parent the parent of this node
	 */
	public EObjectProxyNode (EObject eObject, Object parent) {
		originatorURI = EcoreUtil.getURI(eObject);
		this.parent = parent;
	}
	
	/**
	 * Returns the EMF object that is the originator of this proxy.
	 * The returned EMF object is stored in the resource which is contained in the test navigaor resource set.
	 * @return originator EMF object
	 */
	public EObject getEObject() {
		return EMFUtil.getEObject(null, originatorURI, true);
	}
	
	public Object getParent() {
		return parent;
	}
	
	public URI getOriginatorURI() {
		return originatorURI;
	}
	
	/** Identifier for an EMF object is its URI fragment part. If there is no fragment identifier is empty, this means that the object is a root object.  
	 * 
	 */
	public String getIdentifier() {
		String fragment = originatorURI.fragment();
		if(fragment != null) {
			return fragment;
		} else {
			return ""; //$NON-NLS-1$
		}
	}

	/**
	 * Underlying resource of an EMF object is the file containing this object.
	 */
	public IFile getUnderlyingResource() {
		return EMFUtil.getWorkspaceFile(originatorURI);
	}

	/** 
 	 * For performance reason, EObjectProxyNode instances are not, by default, adaptable in EObject.
	 */
	public Object getAdapter(Class adapter) {
		return Platform.getAdapterManager().getAdapter(this, adapter);
	}

}
