/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: FileProxyNode.java,v 1.4 2005/03/21 17:47:35 dguilbaud Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.navigator;

import java.lang.reflect.InvocationTargetException;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.hyades.test.ui.internal.navigator.action.RenamerUIStatus;
import org.eclipse.hyades.test.ui.navigator.actions.IProxyNodeRenamer;
import org.eclipse.hyades.test.ui.navigator.actions.RenamerUIInlineEditor;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.eclipse.ui.internal.SharedImages;
import org.eclipse.ui.internal.ide.IDEWorkbenchPlugin;
import org.eclipse.ui.internal.progress.ProgressMonitorJobsDialog;

/** Proxy node for file.
 *  Default behavior display the file using its name, the file icon and no children.
 * 
 * @author jgout
 * @since 3.2
 */
public abstract class FileProxyNode implements IProxyNode, IProxyNodeRenamer {
	
	protected IFile file;

	public FileProxyNode(IFile file) {
		//- file without children default file node
		this.file = file;
	}
		
	public String getText() {
		return file.getName();
	}

	public Image getImage() {
		SharedImages images = new SharedImages();
		return images.getImage(ISharedImages.IMG_OBJ_FILE);
	}

	public Object getParent() {
		return file.getParent();
	}
	
	public IFile getFile() {
		return file;
	}
	
	public IProxyNode [] getChildren() {
		//- by default file proxy does not have children
		return new IProxyNode[0];
	}
	
	/** A file proxy node is by definition a root level node, its identifier is empty.
	 * 
	 */
	public String getIdentifier() {
		return ""; //$NON-NLS-1$
	}
	
	/** The underlying resource of a file proxy node is the file itself.
	 * 
	 */
	public IFile getUnderlyingResource() {
		return file;
	}
	
	public Object getAdapter(Class adapter) {
		//- this proxy is adaptable in IResource if it has been placed instead of a file
		if(adapter.equals(IResource.class) && getParent() instanceof IContainer) {
			return file;
		} else if (adapter == IProxyNodeRenamer.class) {
			return this;
		} else {
			return Platform.getAdapterManager().getAdapter(this, adapter);
		}
	}
	
	public boolean isApplicableFor() {
		return true;
	}

	public boolean performRename(String newName) {
		//- do nothing if still the same name
		if (!newName.equals(file.getName())) {
			//- need to check if the new name is a valid one
			IWorkspace workspace = IDEWorkbenchPlugin.getPluginWorkspace();
			IStatus status = workspace.validateName(newName, file.getType());
			if (!status.isOK()) {
				MessageDialog.openError(Display.getCurrent().getActiveShell(), HyadesUIPlugin.getString("W_ERROR"), status.getMessage()); //$NON-NLS-1$
			}
			else {
				final IPath newPath = file.getFullPath().removeLastSegments(1).append(newName);
				WorkspaceModifyOperation op = new WorkspaceModifyOperation() {
					public void execute(IProgressMonitor monitor) {
						try {
							file.move(newPath, IResource.KEEP_HISTORY | IResource.SHALLOW, new SubProgressMonitor(monitor, 50));
						} catch (CoreException e) {
							e.printStackTrace();
						}
					}
				};
				try {
					new ProgressMonitorJobsDialog(Display.getCurrent().getActiveShell()).run(true, false, op);
				} catch (InvocationTargetException e) {
					e.printStackTrace();
				} catch (InterruptedException e) {
					//- can't be canceled
				}
			}
		}
		return false;
	}
	
	public RenamerUIStatus performUserInteraction(String oldName) {
		return new RenamerUIInlineEditor();
	}

}
