/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: RangeFilterHelper.java,v 1.2 2005/05/05 16:40:47 dnsmith Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.logging.adapter.internal.filters;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;

import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.impl.CommonBaseEventImpl;

/**
 * @author smith
 * @since Mar 24, 2005
 * 
 * This class is a helper class to implement range filtering
 * 
 */
public class RangeFilterHelper {

	public static final String RANGEFILTER	= "RangeFilter";
	public static final String EVENTS 		= "events";
	public static final String SECONDS	 	= "seconds";
	
	/* Counter for the number of events processed by the filter component */
	private int eventCounter = 0;
	
	/* Flag to indicate if CommonBaseEvents are to be cached */
	private boolean cacheCBE = false;

	/* CommonBaseEvent cache */
	private CommonBaseEvent [] cbeCacheArray = null;
	
	/* CommonBaseEvent Cache array list */
	private ArrayList cbeCacheArrayList = null;
	
	/* The limit associated with this range filter */
	private int rangeLimit = 0;
	
	/* Flag to indicate if the range filter is time based */
	private boolean time = false;
	
	/* time used for doing the filter range calculations */
	private long baseTime = 0;
	
	/* Comparator for comparing Common Base Events based on creationTime value */
	private CBEComparator cbeComparator = null;

	/*
	 * 
	 * @author smith
	 * @since Mar 28, 2005
	 * 
	 * This class represents a cache element.
	 *
	 */
	protected class CacheElement {
		private CommonBaseEvent cbe;
		private boolean filterTrue;
		private long creationTime;
		
		public CacheElement(CommonBaseEvent evt, boolean filterResult) {
			this.cbe = evt;
			this.filterTrue = filterResult;
			this.creationTime = evt.getCreationTimeAsLong();
		}
		
		/**
		 * @return Returns the cbe.
		 */
		public CommonBaseEvent getCbe() {
			return cbe;
		}
		/**
		 * @return Returns the creationTime.
		 */
		public long getCreationTime() {
			return creationTime;
		}
		/**
		 * @return Returns the filterTrue.
		 */
		public boolean isFilterTrue() {
			return filterTrue;
		}
	}
	
	/*
	 * 
	 * @author smith
	 * @since Mar 28, 2005
	 * 
	 * This class represents a Comparator for comparing Common Base Events based on creationTime value.
	 *
	 */
	protected class CBEComparator implements Comparator {
		public int compare(Object e1, Object e2) {
					
			long etime1 = ((CacheElement)e1).getCreationTime();
			long etime2 = ((CacheElement)e2).getCreationTime();
			
			if (etime1 < etime2) {
				return -1;
			}
			else if (etime1 > etime2) {
				return 1;
			}
			else {
				/* If the timestamps are the same then check the 
				 * sequence numbers to determine order.
				 */
				CommonBaseEvent eCBE1 = ((CacheElement)e1).getCbe();
				CommonBaseEvent eCBE2 = ((CacheElement)e2).getCbe();
				
				if (eCBE1.getSequenceNumber() < eCBE2.getSequenceNumber()) {
					return -1;
				}
				else if (eCBE1.getSequenceNumber() > eCBE2.getSequenceNumber()) {
					return 1;
				}
			}
			return 0;
		}
	}
	
	/**
	 * Constructor
	 */
	public RangeFilterHelper() {
		super();
	}

	/**
	 * Gets the value for the attribute specifed in the path parameter based on the cbe parameter
	 * @param cbe Common Base Event
	 * @param path path list that represents an attribute name
	 * @return value for the specified attribute
	 * @throws InvalidAttributeNameException
	 */
	public String getValueFromPath(CommonBaseEvent cbe, List path) throws InvalidAttributeNameException {
		if (cbe == null | path == null) {
			return null;
		}
		
		Iterator i = path.iterator();
		
		/* Ensure this is a range filter attribute name */
		if (path.isEmpty() || !i.next().equals(RANGEFILTER)) {
			throw new InvalidAttributeNameException();
		}
		
		if (i.hasNext()) {
			String attributeName = (String)i.next();
			// If events is specified Then return the event counter
			if (attributeName.equals(EVENTS)) {
				eventCounter++;
				return Integer.toString(eventCounter);
			}
			// Else if seconds is specified Then return the delta with the base time
			else if (attributeName.equals(SECONDS)) {
				time = true;
				long createTime = cbe.getCreationTimeAsLong();
				
				if (baseTime == 0) {
					baseTime = createTime;
					return "0";
				}
				
				if (baseTime > createTime) {
					return Long.toString((baseTime - createTime)/1000);
				}
				else {
					return Long.toString((createTime - baseTime)/1000);
				}
			}
		}
		
		// Cannot find a known attribute name in the 
		// attribute name path so throw an exception.
		throw new InvalidAttributeNameException();
	}
	
	/**
	 * @return Returns the cacheCBE flag.
	 */
	public boolean isCacheCBE() {
		return cacheCBE;
	}
	
	/**
	 * Cache a Common Base Event and its filter result
	 * @param cbe Common Base Event
	 * @param filterResult boolean filter result
	 */
	public void cache(CommonBaseEvent cbe, boolean filterResult) {
		/* bugzilla 93464
		 * Clone the CommonBaseEvent object because the one passed in might be reused.
		 * It cannot be assumed that new objects are passed to the Filter component
		 * each time.
		 */
		CommonBaseEvent newCBE;
		try {
			newCBE = (CommonBaseEvent)(((CommonBaseEventImpl)cbe).clone());
		}
		catch (CloneNotSupportedException e) {
			newCBE = cbe;
		}
		
		CacheElement element = new CacheElement(newCBE, filterResult);
		
		if (time) {
			// Create the cache if it does not exist
			if (cbeCacheArrayList == null) {
				cbeCacheArrayList = new ArrayList();
				cbeCacheArrayList.add(element);
			}
			else {
				cbeCacheArrayList.add(element);
				Collections.sort(cbeCacheArrayList, getCbeComparator());
				
				// Check that all events in the cache match the range filter
				long maxEvtTime = ((CacheElement)(cbeCacheArrayList.get(cbeCacheArrayList.size()-1))).getCreationTime();
				
				// Determine how many events need to be removed from the cache
				int removeCount;				
				for(removeCount=0; removeCount < cbeCacheArrayList.size(); removeCount++) {
					long ctime = ((CacheElement)cbeCacheArrayList.get(removeCount)).getCreationTime();
					
					if ((maxEvtTime-ctime)/1000 <= rangeLimit) {
						break;
					}
				}
				
				// Remove events that don't match the range filter
				for(int j=0; j < removeCount; j++ ) {
					// The list is in sorted order so alway remove the first element.
					cbeCacheArrayList.remove(0);
				}
			}				
		}
		else {
			// Create the cache if it does not exist
			if (cbeCacheArrayList == null) {
				// cbeCacheArray = new CommonBaseEvent[rangeLimit];
				cbeCacheArrayList = new ArrayList(rangeLimit);
			}

        	// If the cache is full then remove the oldest element add the new CommonBaseEvent to the end of it
			if (cbeCacheArrayList.size() == rangeLimit) {
				// Remove the oldest element in the cache and append the new one to the end of the cache
				cbeCacheArrayList.remove(0);
			}
			cbeCacheArrayList.add(element);
		}
		
		
	}
	/**
	 * Sets the cacheCBE flag
	 * @param cacheCBE The cacheCBE to set.
	 */
	public void setCacheCBE(boolean cacheCBE) {
		this.cacheCBE = cacheCBE;
	}
	/**
	 * @return Returns the cbeCacheArray.
	 */
	public CommonBaseEvent[] getCbeCacheArray() {
		// If the cache array has not been set but there is a cache array list 
		// Then copy the filtered CommonBaseEvents from the array list to the
		// array.
		if (cbeCacheArray == null && cbeCacheArrayList != null) {
			if (cbeCacheArrayList.size() > 0) {
				cbeCacheArray = new CommonBaseEvent[cbeCacheArrayList.size()];
				Iterator i = cbeCacheArrayList.iterator();
				int j = 0;
				while (i.hasNext()) {
					CacheElement e = (CacheElement)i.next();
					// Only copy filtered events to the array
					if (e.isFilterTrue()) {
						cbeCacheArray[j++] = e.getCbe();
					}
				}
			}
		}
		return cbeCacheArray;
	}
	/**
	 * @return Returns the rangeLimit.
	 */
	public int getRangeLimit() {
		return rangeLimit;
	}
	/**
	 * @param rangeLimit The rangeLimit to set.
	 */
	public void setRangeLimit(int rangeLimit) {
		this.rangeLimit = rangeLimit;
	}
	/**
	 * @return Returns the cbeComparator.
	 */
	protected CBEComparator getCbeComparator() {
		if (cbeComparator == null) {
			cbeComparator = new CBEComparator();
		}
			
		return cbeComparator;
	}
}
