/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse
 * Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: AdapterConfigValidator.java,v 1.8 2005/02/16 22:20:29 qiyanli Exp $
 * 
 * Contributors: IBM - Initial API and implementation
 ******************************************************************************/
package org.eclipse.hyades.logging.adapter.internal.util;

import java.lang.reflect.Constructor;
import java.net.URL;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.xml.sax.ErrorHandler;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

import org.eclipse.hyades.logging.adapter.AdapterException;
import org.eclipse.hyades.logging.adapter.IProcessUnit;
import org.eclipse.hyades.logging.adapter.util.AdapterUtilities;
import org.eclipse.hyades.logging.adapter.util.Messages;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.EventFactory;
import org.eclipse.hyades.logging.events.cbe.EventFactoryHome;
import org.eclipse.hyades.logging.events.cbe.impl.EventFactoryContext;

/*
 * 
 * @author smith
 *
 * This class is used to validate the adapter file against the set of Generic Log Adapter 
 * schema files and log any validation errors.
 * 
 */
 public class AdapterConfigValidator {

	private String currentPlugin;
	private boolean isWorkbench = false;
	
    // Logger to log validation errors to
	private IProcessUnit logger;

	private String schemaLocation = Messages.getString("HyadesGAValidationSchemaLocation");
	private EventFactory eventFactory = null;

	public AdapterConfigValidator(IProcessUnit inLogger) {

		isWorkbench = AdapterUtilities.isWorkbench();
		this.logger = inLogger;
		logger.setLogger(inLogger);

		if (isWorkbench) {
			currentPlugin = Messages.getString("HyadesGAValidationCurrentPlugin");
		}
	}

	/*
	 * Get the array of schema files
	 */
	private String getExternalSchemaLocationValues(boolean isWorkbench)
			throws AdapterException {

		String retValue = "";
		String fileSeparator = System.getProperty("file.separator");
		String namespacePrefix = Messages.getString("HyadesGAValidationNamespacePrefix");

		//Get the Schema URLs list from the <plugin>/schema directory
		if (isWorkbench) {
			try {
				Class pathClass = Class.forName("org.eclipse.core.runtime.Path");
				Constructor pathConstructor = pathClass.getConstructor(new Class[] { schemaLocation.getClass() });
				Object path = pathConstructor.newInstance(new Object[] { schemaLocation });

				Class platformClass = Class.forName("org.eclipse.core.runtime.Platform");
				Class pluginClass = Class.forName("org.eclipse.core.runtime.Plugin");

				Object plugin = platformClass.getMethod("getPlugin", new Class[] { currentPlugin.getClass() }).invoke(null,	new Object[] { currentPlugin });

				URL osgiUrl = (URL) (pluginClass.getMethod("find",	new Class[] { Class.forName("org.eclipse.core.runtime.IPath") }).invoke(plugin, new Object[] { path }));

				File f = new File(((URL) (platformClass.getMethod("asLocalURL",	new Class[] { osgiUrl.getClass() }).invoke(null, new Object[] { osgiUrl }))).getFile());

				String absoluteUrl = f.getAbsolutePath();

				if (f.isDirectory()) {
//					String[] xsdFiles = f.list();
					File[] files = f.listFiles();
					if (files.length > 0) {
						for (int i = 0; i < files.length; i++) {
							if (!files[i].isDirectory() && (files[i].getName().endsWith(".xsd") || files[i].getName().endsWith(".XSD"))) {
								String namespace = namespacePrefix + files[i].getName();
//								String url = absoluteUrl + fileSeparator + files[i].getName();
								retValue = retValue + " " + namespace + " " + files[i].toURI();
							}
						}
					}
				}			
			} catch(Exception e){
				throw new AdapterException(Messages.getString("HyadesGAValidation_Getting_Schemas_ERROR_", e.getMessage()));
			}
		}
		//Get the Schema URIs from the system GLA_HOME/schema directory
		else {
			if (System.getProperty("GLA_HOME") != null) {

				String glaHome = System.getProperty("GLA_HOME");
				
				// Strip any double quotes from GLA_HOME value
				if (glaHome.startsWith("\"") && glaHome.endsWith("\"")) {
					glaHome = glaHome.substring(1,glaHome.length()-1);
				}
				
				String schemasURLHome = glaHome + fileSeparator + schemaLocation;
			
				File schemasDir = new File(schemasURLHome);

				if (schemasDir.isDirectory()) {
//					String[] xsdFiles = schemasDir.list();
					File[] files = schemasDir.listFiles();
					if (files.length > 0) {							
						for (int i = 0; i < files.length; i++) {	
							if (!files[i].isDirectory() && (files[i].getName().endsWith(".xsd") || files[i].getName().endsWith(".XSD"))) {
								String namespace = namespacePrefix + files[i].getName();							
								retValue = retValue + " " + namespace + " " + files[i].toURI();
							}
						}
					}
				}
			    else {
			    	logMessage(Messages.getString("HyadesGAValidation_GLAHome_Invalid_ERROR_", glaHome));
			    }
			} else {
				logMessage(Messages.getString("HyadesGAValidation_No_GLAHome_ERROR_"));
			}	
		}
		return retValue.trim();
	}
	
	/*
	 * Get the document builder
	 */
	private DocumentBuilder getValidateDocumentBuilder()
			throws AdapterException, IOException {
		DocumentBuilder builder = null;

		try {
		
			DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		
			factory.setNamespaceAware(true);

            //NOTE: The Sun JDK v.1.4.x that includes the Crimson XML parser implementation
            //does not support schema validation.
            
            //Attempt to configure the XML parser to validate the XML file based on the associated schema(s):
            try {
                    
                    factory.setAttribute(Messages.getString("HyadesGAValidationJAXP_SCHEMA_LANGUAGE"), Messages.getString("HyadesGAValidationW3C_XML_SCHEMA"));
                    factory.setAttribute(Messages.getString("HyadesGAValidationEXTERNAL_SCHEMA_LOCATION"), getExternalSchemaLocationValues(isWorkbench));

                    factory.setValidating(true);

            } catch (Exception e) {
                    
                //Create a non-validating XML parser and log the occurrence:        		
                CommonBaseEvent commonBaseEvent = getEventFactory().createCommonBaseEvent();
                /* bugzilla 84698 - changed the severity from WARNING to HARMLESS so it won't be 
                 * displayed to the user during import.
                 */
        		commonBaseEvent.setSeverity(CommonBaseEvent.SEVERITY_HARMLESS);
        		commonBaseEvent.setMsg(Messages.getString("HyadesGAValidation_Validation_Not_Supported_WARN_"));
        		
        		logger.log(commonBaseEvent);
            }
    		
			builder = factory.newDocumentBuilder();			
			builder.setErrorHandler(new ErrorHandler() {
				public void warning(SAXParseException e) throws SAXException {
					logMessage(e, "Warning");
				}
				public void error(SAXParseException e) throws SAXException {
					logMessage(e, "Error");
				}
				public void fatalError(SAXParseException e) throws SAXException {
					logMessage(e, "fatalError");
				}
			});

		} catch (ParserConfigurationException pce) {
			throw new AdapterException(pce.getMessage());
		}
		return builder;
	}

	/*
	 * Validate the input stream against a set of schema files
	 */
	public void validate(InputStream inAppStream) throws AdapterException {

		try {
			DocumentBuilder builder = getValidateDocumentBuilder();			
			builder.parse(inAppStream);

		} catch (SAXException se) {
			throw new AdapterException(se.getMessage());
		} catch (Exception e) {
			throw new AdapterException(e.getMessage());
		}
	}

	/* 
	 * Get a CBE Event factory
	 */
	public EventFactory getEventFactory() {
		if (eventFactory == null) {

			EventFactoryHome eventFactoryHome = EventFactoryContext.getInstance().getEventFactoryHome("org.eclipse.hyades.logging.adapter.internal.util.RuntimeEventXMLFileEventFactoryHomeImpl", this.getClass().getClassLoader());

			if (eventFactoryHome != null) {
				eventFactory = eventFactoryHome.getEventFactory("org.eclipse.hyades.logging.adapter.impl.Component");
				eventFactory.setCompleteEvent(true);
			} else {
				eventFactory = EventFactoryContext.getInstance().getSimpleEventFactoryHome().getAnonymousEventFactory();
			}
		}

		return eventFactory;
	}

	/*
	 * Log a SAX parse exception
	 */
	public void logMessage(SAXParseException e, String type) {

		String msg = null;
		short severity = 0;

		CommonBaseEvent event = getEventFactory().createCommonBaseEvent();
		if (type.equals("fatalError")) {
			msg = Messages.getString("HyadesGAValidation_Fatal_ERROR_", e.getMessage());
			severity = CommonBaseEvent.SEVERITY_FATAL;
		} else if (type.equals("Error")) {
			msg = Messages.getString("HyadesGAValidation_ERROR_", e.getMessage());
			severity = CommonBaseEvent.SEVERITY_CRITICAL;
		} else if (type.equals("Warning")) {
			msg = Messages.getString("HyadesGAValidation_WARN_", e.getMessage());
			severity = CommonBaseEvent.SEVERITY_WARNING;
		}

		event.setSeverity(severity);
		event.setMsg(msg);
		logger.log(event);
	}
	
	/*
	 * Log a message
	 */
	public void logMessage(String msg) {

		CommonBaseEvent event = getEventFactory().createCommonBaseEvent();

		event.setSeverity(CommonBaseEvent.SEVERITY_CRITICAL);
		event.setMsg(msg);
		logger.log(event);
	}	
	
}