package org.eclipse.hyades.logging.adapter.outputters;

import java.io.FileWriter;
import java.io.IOException;
import java.util.Hashtable;

import org.eclipse.hyades.logging.adapter.AdapterException;
import org.eclipse.hyades.logging.adapter.util.Messages;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;

/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: CBEConvergentFileOutputter.java,v 1.6 2005/03/14 23:25:20 dnsmith Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 * 
 * Change History:
 * Bugzilla  Description
 * 86677     Added ConvergentWriter internal class to maintain use count
 * 
 **********************************************************************/

/**
 * A specialization of the <code>CBEFileOutputter</code> for atomically
 * writing Common Base Events from multiple contexts to a single output file.
 * <p>
 * 
 * @author Richard K. Duggan
 * @author Paul E. Slauenwhite
 * @version March 14, 2005
 * @since April 23, 2004
 * @see org.eclipse.hyades.logging.adapter.outputters.CBEFileOutputter
 */
public class CBEConvergentFileOutputter extends CBEFileOutputter {
    
    /**
     * Static cache (e.g. <code>java.util.Hashtable</code>) of file 
     * writers for atomically writing to output files.
     * <p>
     * <code>FileWriter</code> values are keyed by the absolute path 
     * of their physical output file.
     */
    private static Hashtable fileWriters = new Hashtable();
    
    /**
     * Static platform-dependent line separator string.
     */
    private static final String LINE_SEPARATOR = System.getProperty("line.separator");

    private boolean stopped = false;
    
    /*
     * bugzilla 86677
     * 
     * @author Dave N. Smith
     *
     * This class extends FileWriter by maintaining a use count 
     */
    private class ConvergentWriter extends FileWriter {
    	// private FileWriter fileWriter;
    	private int userCount;
    	private Object lock;
    	public ConvergentWriter(String fileName) throws IOException {
    		super(fileName);
    		// Initialize userCount and lock
    		userCount = 0;
    		lock = new Object();
    	}

		/**
		 * @return Returns the userCount.
		 */
		public int getUserCount() {
			synchronized(lock) {
				return userCount;
			}
		}
		
		/**
		 * Increments the userCount.
		 */
		public void incrementUserCount() {
			synchronized(lock) {
				userCount++;
			}
		}
		
		/**
		 * Decrements the userCount.
		 */
		public void decrementUserCount() {
			synchronized(lock) {
				userCount--;
			}
		}
    }
    
	/**
	 * Set the FileWriter for this File Outputter
	 */
	protected void setFileWriter() throws AdapterException 
	{
		synchronized(fileWriters) {
			
            //If a file writer is cached for this output file, set
            //the current <code>FileWriter</code> handle to the 
            //cached instance to ensure atomic file writing:
		    if(fileWriters.containsKey(rawLogFileName)) {
		        fw = ((ConvergentWriter)(fileWriters.get(rawLogFileName)));
		        // Increment user count
		        ((ConvergentWriter)fw).incrementUserCount();
		    }		    
		    //Otherwise, cache the newly created <code>FileWriter</code> 
		    //handle for future use:
		    else{
				try {
					// Create the FileWriter
					fw = new ConvergentWriter(rawLogFileName);
				}
				catch (IOException ioe) {
				    
				    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();
				    
				    event.getMsgDataElement().setMsgCatalogId("HyadesGA_CBE_File_Outputter_Create_ERROR_");
				    event.getMsgDataElement().setMsgCatalogTokensAsStrings(new String[]{getUniqueID(),rawLogFileName, ioe.getMessage()});							    

				    event.setSeverity(CommonBaseEvent.SEVERITY_FATAL);

				    log(event);

					/* RKD: At this point there is no point if allowing this context to continue as we cannot access this outputter file.
					 */
				    throw new AdapterException(Messages.getString("HyadesGA_CBE_File_Outputter_Create_ERROR_", getUniqueID(),rawLogFileName, ioe.getMessage()));
				}

				// Increment the user count
		    	((ConvergentWriter)fw).incrementUserCount();
		    	// Cache the FileWriter
		        fileWriters.put(rawLogFileName,fw);
		    }
		}
	}	
	
    /**
     * @see org.eclipse.hyades.logging.adapter.outputters.CBEFileOutputter#writeToFile(java.lang.String)
     */
    protected void writeToFile(String event) throws IOException {
		
        //Ensure writing to the single output file is atomic:
	    synchronized(fw) {
		    		   
		    fw.write(event);
			fw.write(LINE_SEPARATOR);
			fw.flush();
		}		
	}
    
	/**
	 * Closes the file we are writing to.
	 * @see org.eclipse.hyades.logging.adapter.IComponent#stop()
	 */
	public void stop()
	{
		if (!stopped) {
			stopped = true;
			if (fw != null) {
				// Ensure closing the output file is atomic
				synchronized(fw) {
					ConvergentWriter cw = (ConvergentWriter)fw;
					// Check the user count before closing the file.  If there are others using it then 
					// Only decrement the user count
					if ( cw.getUserCount() > 1) {
						cw.decrementUserCount();
					}
					// If we are the only one writing to the file then close it
					else if ( cw.getUserCount() == 1) {
						// First decrement the user count
						cw.decrementUserCount();
						try {
							cw.close();
						}
						catch(IOException e) {
							/* We cannot close the file.  The file must already be closed.  Perhaps
							 * we should log an event in this scenario.
							 */
						}
						
					}
				}
			}
		}
	}
}
