package org.eclipse.hyades.logging.adapter.parsers;

/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: AttributeStatement.java,v 1.15 2005/04/28 20:22:27 dnsmith Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 * 
 * Change History:
 * Bugzilla  Description
 * 91218     Performance fix - use a single ArrayList for the result 
 *           instead of instantiating a new one every time.
 * 
 **********************************************************************/

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.ListIterator;

import org.eclipse.hyades.logging.adapter.AdapterException;
import org.eclipse.hyades.logging.adapter.IParser;
import org.eclipse.hyades.logging.adapter.util.Messages;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.ExtendedDataElement;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
 * @author rduggan
 *
 * This class represents a particular attribute to be
 * parsed from the record.  It can have multiple substitution
 * rules that are used to determine the value for the
 * attribute based on the data in the record.
 *  
 */
public class AttributeStatement implements IStatement {

	/* The parser that this AttributeStatement is a child of */
	private IParser parser;
	
	List substitutions;

	/* The name of this attribute */
	private String name;
	
	/* The default value for this attribute */
	private String defaultValue;
	
	/* List to store the default value for this attribute. (bugzilla 91218)
	 * There will only be one default value for the attribute. */
	private List defaultValueList = new ArrayList(1);
	
	/* Whether ths Attribute keeps state between rule runs */
	private boolean usePreviousMatchSubstitutionAsDefault=false;
	
	/* Whether this attribute is required for the element to be valid */
	private boolean isRequiredByParent=false;
	
	/* List to store the result for the attribute. (bugzilla 91218)
	 * There will only be one result for the attribute. */
	private List resultList=new ArrayList(1);
	
	
	/**
	 * 
	 */
	public AttributeStatement() {
	}
	
	/**
	 * 
	 */
	public AttributeStatement(IParser parser) {
		this.parser=parser;
	}
	


	/**
	 * @see org.eclipse.hyades.logging.adapter.IStatement#prepare(org.w3c.dom.Element)
	 */
	public void prepare(Element node, String path) throws PreparationException {
	
		if(path==null) {
			this.name=node.getAttribute(Messages.getString("HyadesGANameAttributeName"));
		}
		else {
			this.name=path+"."+node.getAttribute(Messages.getString("HyadesGANameAttributeName"));
		}
		
		/* Retrieve the index if there is one specified and embed this into the name */
		String index=node.getAttribute(Messages.getString("HyadesGAIndexAttributeName"));
		if(index!=null && !index.equals("")) {
			this.name=this.name+"["+index+"]";
		}
		
		/* Retrieve the default value for this attribute */
		defaultValue = node.getAttribute(Messages.getString("HyadesGAdefaultValueAttributeName"));
		if(defaultValue == null || defaultValue.equals("")) {
			defaultValue = null;
		}
		else {
			/* If there is a default value then create the directed graph to store it 
			 * and add it to the list. */
			defaultValueList.add(new DirectedGraphImpl(this.name, defaultValue));
		}
		
		/* Retrieve whether we are keeping previous matches around */
		String previous=node.getAttribute(Messages.getString("HyadesGAUsePreviousMatchSubstitutionAsDefault"));
		if(previous==null || previous.equals("") || previous.equals("false")) {
			usePreviousMatchSubstitutionAsDefault=false;
		}
		else {
			usePreviousMatchSubstitutionAsDefault=true;
		}
		
		/* Is this attribute required by my parent element to be valid? */
		String isRequired=node.getAttribute(Messages.getString("HyadesGAisRequiredByParentAttributeName"));
		if(isRequired!=null && isRequired.equals("true")) {
			isRequiredByParent=true;
		}
		else {
			isRequiredByParent=false;
		}
		
		/* Create a SubstituteStatement per Substitution rule */
		NodeList children=node.getChildNodes();
		int count=children.getLength();
		if(count>0) {
			substitutions=new ArrayList();
			for(int i=0; i<count; i++) {
				Element child=null;
				try {
					child=(Element)children.item(i);
				}
				catch(ClassCastException e) {
					/* We can ignore this child as it is not a Element */	
					continue; 	
				}
				if(child.getNodeName().endsWith(Messages.getString("HyadesGASubstitutionRuleTagName"))) {

					IStatement statement=new SubstituteStatement(parser);
					
					try {
						statement.prepare(child, name);
						substitutions.add(statement);
					}
					catch(PreparationException e) {
						/* SubstitutionStatement rule has a problem.  We will log the fact this is a problem and ignore this rule during the run */
						if(parser!=null && parser instanceof Parser) {
							CommonBaseEvent event = ((Parser)parser).getEventFactory().createCommonBaseEvent();
							if(e.getDetails()!=null) {
								event.setMsg(e.getMessage() + "  " + e.getDetails() );
							}
							else {
								event.setMsg(e.getMessage());
							}
						
						    event.setSeverity(CommonBaseEvent.SEVERITY_CRITICAL);
							
							ExtendedDataElement ede=((Parser)parser).getEventFactory().createExtendedDataElement();
							ede.setName("EXCEPTION DETAILS");
							ede.setType(ExtendedDataElement.TYPE_STRING);
							if(e.getDetails()!=null) {
								ede.setValues(new String[] {e.getDetails()});
							}
							else {
								ede.setValues(new String[] {e.getLocalizedMessage()});
							}
							event.addExtendedDataElement(ede);
							
							ede=((Parser)parser).getEventFactory().createExtendedDataElement();
							ede.setName("PATH");
							ede.setType(ExtendedDataElement.TYPE_STRING);
							ede.setValues(new String[] {e.getDigraph()});
							event.addExtendedDataElement(ede);
							
						    parser.log(event);
							
						}
						else {
							throw e;
						}
					}
				}
			}
			if (substitutions.isEmpty())
				substitutions = null;	
		}
	}

	/**
	 * @see org.eclipse.hyades.logging.adapter.IStatement#run(java.lang.String)
	 */
	public List run(String line, HashMap inputs) throws AdapterException, StatementException {		
		/*
		 * Iterate through each of the child rules looking for a match.  If there
		 * is a match then stop the iteration and return.
		 */
		if (substitutions!=null) {
			List result;
			ListIterator iter=substitutions.listIterator();
			SubstituteStatement statement = null;
			while(iter.hasNext()) {
				statement=(SubstituteStatement)iter.next();
				
				try {
					result = statement.run(line, inputs);
					/* Clear the result list before adding the new result to it */ 
					resultList.clear();
					resultList.addAll(result);
					
					return resultList;
				}
				catch(StatementException e) {
					/* there was no match */
				}
			}
			/* If there is no match and previous is enabled then return the previous 
			 * result list.
			 */
			if(usePreviousMatchSubstitutionAsDefault && !resultList.isEmpty()) {
				return resultList;
			}
			/* Else if there is a default value then return it */
			else if (!defaultValueList.isEmpty()) {
				return defaultValueList;
			}
			
		}
		/* If there are no substitution rules, check whether there is a default value and if so, return it.*/
		else if (!defaultValueList.isEmpty()) {
			return defaultValueList;
		}
		
		throw StatementException.instance();
	}

	/**
	 * @see org.eclipse.hyades.logging.adapter.parsers.IStatement#isRequiredByParent()
	 */
	public boolean isRequiredByParent() {
		return isRequiredByParent;
	}

	/**
	 * @see org.eclipse.hyades.logging.adapter.parsers.IStatement#isChildChoice()
	 */
	public boolean isChildChoice() {
		return false;
	}
}
