/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: Parser.java,v 1.22 2005/05/17 17:17:25 dnsmith Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.logging.adapter.parsers;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;


import org.eclipse.hyades.logging.adapter.AdapterException;
import org.eclipse.hyades.logging.adapter.AdapterInvalidConfig;
import org.eclipse.hyades.logging.adapter.IParser;
import org.eclipse.hyades.logging.adapter.MessageString;
import org.eclipse.hyades.logging.adapter.impl.ProcessUnit;
import org.eclipse.hyades.logging.adapter.util.Messages;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
 * @author rduggan
 *
 * This is an implementation of a parser that uses the
 * JDK 1.4 java.util.regex library to process regular expressions.
 *
 * @version 0.1
 * @since 0.1
 */
public class Parser extends ProcessUnit implements IParser {
	
	
	/* The sets of rules this parser is configured to work with.
	 */
	protected List ruleSets;
	
	 /*
	  * The separator for fields within a message
	  */
	 protected String separatorToken;
	 protected Pattern separatorPattern;
	 
	 /*
	  * The designation token between the name and the value of a field
	  */
	 protected String designationToken;
	 protected Pattern designationPattern;
	
	
	/**
	 * Overrides the update implementation provided by the ProcessUnit
	 * superclass.  Updates the configuration.
	 *
	 * THIS IMPLEMENTATION IS TIGHTLY COUPLED WITH THE SCHEMA.  If the schema
	 * changes for the parser this will need to be changed accordingly.
	 *
	 * @see org.eclipse.hyades.logging.adapter.IComponent#update()
	 */
	public void update() throws AdapterInvalidConfig{
		
		/* Let our superclass update itself first */
		super.update();
		
		/* Get the configuration that the adapter assigned us.  We
		 * are actually handed the cc:Parser node.  We are interested
		 * in the Parser:Rules child andwe will only consider the first
		 * one.  There should be only one!
		 */
		Element element = getConfiguration();
		
		if(!element.getNodeName().equals(Messages.getString("HyadesGAParserTagName"))) {
			throw new AdapterInvalidConfig(Messages.getString("HyadesGA_CBE_Parser_Preparation_No_Parser_Element_ERROR_",Messages.getString("HyadesGAParserTagName")));
		}
		
		
		
		/* Extract our configuration information from the config.
		 * RKD:  The original code from research created a RulesNode
		 * and told it to prepare itself using the element tht is the
		 * root of the configuration.  It then proceeded to delegate
		 * all of the work to the RulesNode.  I have changed that so
		 * that all of the sets of rules are held directly by this
		 * parser and each ruleset contains a set of rules.
		 */
		
		NodeList rules=element.getChildNodes();
		
		
		separatorToken=element.getAttribute(Messages.getString("HyadesGASeparatorTokenAttributeName"));
		designationToken=element.getAttribute(Messages.getString("HyadesGADesignationTokenName"));
		
		if(separatorToken!=null && !separatorToken.equals("")) {
			compileSeparatorToken(separatorToken);
		}
		
		if(designationToken!=null && !designationToken.equals("")) {
			compileDesignationToken(designationToken);
		}

		/* Prepare the rules */ 
		
		prepareRules(rules);

	}

	/*
	 * Compile the separator token and set the separator pattern
	 */
	protected void compileSeparatorToken(String token) throws AdapterInvalidConfig {
		try {
			separatorPattern=Pattern.compile(token, Pattern.MULTILINE);
		}
		catch(PatternSyntaxException e) {
			throw new AdapterInvalidConfig(Messages.getString("HyadesGA_CBE_Parser_Preparation_Separator_Token_Pattern_Invalid_ERROR_",e.getMessage()));
		}
	}
	
	/*
	 * Compile the designation token and set the designation pattern
	 */	
	protected void compileDesignationToken(String token) throws AdapterInvalidConfig {
		try {
			designationPattern=Pattern.compile(token, Pattern.MULTILINE);
		}
		catch(PatternSyntaxException e) {
			throw new AdapterInvalidConfig(Messages.getString("HyadesGA_CBE_Parser_Preparation_Designation_Token_Pattern_Invalid_ERROR_",e.getMessage()));
		}
	}

	/*
	 * Prepare the rules based on the configuration elements
	 */
	protected void prepareRules(NodeList rules) throws AdapterInvalidConfig {
		int ruleCount = rules.getLength();
		if (ruleCount >0) {
			ruleSets=new ArrayList();
			for (int i = 0; i < ruleCount; i++)	{
				Element rule=null;
				try {
					rule=(Element)rules.item(i);
				}
				catch(ClassCastException e) {
					/* We can ignore this child as it is not a Element */
					continue; 	 	
				}
				if(rule!=null) {
					IStatement statement=null;
					if(rule.getNodeName().endsWith(Messages.getString("HyadesGARuleElementTagName"))) {
						statement=new GroupStatement(this);
					}
					else if(rule.getNodeName().endsWith(Messages.getString("HyadesRuleAttributeTagName"))) {
						statement=new AttributeStatement(this);
					}
					else {
						throw new AdapterInvalidConfig();
					}
					
					/* Try and prepare the rules for processing */
					try {
						statement.prepare(rule, null);	
						ruleSets.add(statement);
					}
					catch(PreparationException e) {
						/* If the rule is bad we will log the fact it is bad and that we are ignoring this rule */
						if (e.getDetails() == null){
						    
						    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();
						    event.setMsg(e.getLocalizedMessage());
						    event.setSeverity(CommonBaseEvent.SEVERITY_CRITICAL);
							
						    log(event);
						}
						else{

						    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();
						    event.setMsg(e.getLocalizedMessage()+" "+e.getDetails()+". ");
						    event.setSeverity(CommonBaseEvent.SEVERITY_CRITICAL);

						    log(event);
						}

					    CommonBaseEvent secondEvent = getEventFactory().createCommonBaseEvent();
					    
					    secondEvent.getMsgDataElement().setMsgCatalogId("HyadesGASubstitutionRuleIgnored_WARN_");
					    secondEvent.getMsgDataElement().setMsgCatalogTokensAsStrings(new String[]{e.getDigraph()});							    

					    secondEvent.setSeverity(CommonBaseEvent.SEVERITY_CRITICAL);

					    log(secondEvent);
					}
				}	
			}
		}
	}
	
	/**
	 * This implementation processes MessageString[] instances where each MessageString
	 * contains a single message.
	 * @see org.eclipse.hyades.logging.adapter.IProcessUntit#processEventItems(java.lang.Object msgs)
	 */
	public Object[] processEventItems(Object[] msgs) {
		/* If we have no rules to run just return */
		if (ruleSets!=null && msgs != null)	{
			
			ArrayList collection=new ArrayList();
			
			for(int i=0; i<msgs.length; i++) {
				
				/* Don't try and process null messages */
				if(msgs[i]==null) {
					continue;
				}
				/* Prepare the line by first breaking it up into tokens.  We do this once
				 * then run the resulting list against each of the rules
				 */
				HashMap map=prepareEnvData(((MessageString)msgs[i]).getValue());
				
				/* Here we iterate through each of the rules..wouldn't it be better
				 * if we just ran the rules with each of the tokens until we get a match..could there be multiple
				 * match statements..this if Big-O n^2, we should be able to make this
				 * linear
				 */
				Iterator iter=ruleSets.listIterator();
				
				List result=null;
				
				while(iter.hasNext()) {
					IStatement statement=(IStatement)iter.next();
					try {
						result=statement.run(((MessageString)msgs[i]).getValue(), map);
						/* Save the results */
						collection.add(result);
					}
					catch (AdapterException e) {

					    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();
					    event.setMsg(e.getLocalizedMessage());
					    event.setSeverity(CommonBaseEvent.SEVERITY_CRITICAL);
						
					    log(event);
					}
					catch(StatementException e) {
						/* No results from the current statement */
					}
				}
			}
			
			incrementItemsProcessedCount(collection.size());
			trace("HyadesGATrace_Component_Processing_End_INFO_", getName(), Integer.toString(collection.size()));

			// handle empty collection case
			if (collection.size() == 0) {
				return null;
			}
			return collection.toArray();
		}
		trace("HyadesGATrace_Component_Processing_End_INFO_", getName(), Integer.toString(0));
		return null;
	}
	
	/**
	 *
	 * @see org.eclipse.hyades.logging.adapter.IProcessUntit#processEventItems(java.lang.Object msgs)
	 */
	public Object[] testProcessEventItems(Object[] msgs) throws AdapterInvalidConfig {
		if(!(msgs instanceof MessageString[])) {
			throw new AdapterInvalidConfig("This parser will only accept arrays of MessageString");
		}
		return msgs;		
	}
	

	
	/**
	 * Separate the message into individual properties based upon the separator.
	 * This method will split the names and vales apart if there is a designation
	 * token in the name.
	 *
	 * RKD:  This method does not take into account quoted strings.  If a separator
	 * token or designation token appears within a quoted portion of a record should
	 * it be honoured?
    *
	 * @param line
	 * @param env
	 */
	protected HashMap prepareEnvData(String line) {
		
		HashMap result=new HashMap();
		
		String [] list;
		
		/* If we have no separatorToken we just return */
		if(separatorToken==null || separatorToken.equals("")) {
			return result;
		}
		
		/* We use the split method to split up the
		 * line into tokens.
		 */
		list = separatorPattern.split(line);
		
		/* Iterate over each of the tokens. */
		
		for(int i=0; i<list.length; i++) {

			/* The key and the value to place in the hash */
			StringBuffer hashVal = new StringBuffer();
			String hashKey = "";	
		
			String val="";
			
			String current=list[i];
			
			/* zero length strings will be ignored */
			if (current.length() > 0) {
			
				/* If we have a designation token then we need to search for it and split the pair up.  Otherwise
				 * we just add this to the list
				 * bugzilla 87932 - all values will be trimmed before being added to the hash map
				 * and only non-empty strings will be added to the hash map.
				 */
				if(designationToken==null || designationToken.equals("")) {
					/* bugzilla 87932 - trim position values before adding them to the hash map 
					 * to be consistent with hash values. */
					val = current.trim();
					if (val.length() > 0) {
						result.put(new Long(i), val);
					}
				}
				else {
					Matcher matcher = designationPattern.matcher(current);

					if(matcher.find()) {
				
						/* We have found the designation token so get the key and value.*/
						
						/* Make sure there is a key ie something before the designation token */
						if(matcher.start()>0) {
							hashKey=(current.substring(0, matcher.start())).trim();
							/* Only get the value if the match wasn't at the end of the string */
							if (matcher.end() < current.length()) {
								val=(current.substring(matcher.end())).trim();
								/* add the key/value to the hash map if they are both non-empty */
								if (val.length() > 0 && hashKey.length() > 0) {
									result.put(hashKey,val);
								}
							}
							
						}
					
					}
					else {
						/* We have a designation token but there is none in this separated element,
						 * we will add this into the position list based upon index
						 */
						/* bugzilla 87932 - trim position values before adding them to the hash map 
						 * to be consistent with hash values. */
						val = current.trim();
						if (val.length() > 0) {
							result.put(new Long(i), val);
						}
					}
				}
			}
		}
		return result;
	}
}
