/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: PositionParser.java,v 1.5 2005/04/28 20:22:27 dnsmith Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.logging.adapter.parsers;

import java.util.ArrayList;
import java.util.List;
import java.util.ListIterator;
import java.util.StringTokenizer;

/**
 * This class provides the ability to process 'Position' attribute
 * values.
 * 
 * @author rduggan
 * @version 0.1
 * @since 0.1
 */
public class PositionParser {
	
	/* The separator for position statements */
	static final String POSITION_SEPARATOR = "@@";
	
	/* The hash start for position statements */
	static final String HASH_START = "$h(";
	
	/* The hash end for position statements */
	static final String HASH_END = ")";


	/**
	 * Parse a position attribute value and return the tokens that are embedded within
	 * the string.
	 * 
	 * @param strPositions - the position string to parse
	 * @param bStripHash - whether to strip the hash information from the attributes 
	 * @return - the tokenized position information
	 * @since 0.1
	 */
	public static List getPositionedString(String strPositions, boolean bStripHash) {
		if(strPositions==null) {
			return null; 
		} 
		
		List list = new ArrayList();
		List newList=new ArrayList();
		
		/* Tokenize the string based upon the separator */
		StringTokenizer s = new StringTokenizer(strPositions, POSITION_SEPARATOR);
	    while (s.hasMoreTokens()) {
				list.add(s.nextToken());
		}
		
		/* Iterate over the tokenized string and process each token. */
		ListIterator iterator=list.listIterator();
		while(iterator.hasNext()) {
			String current=(String)iterator.next();
			if(current!=null) {
				current.trim();
				if(!current.equals("")) {
					Object result=null;
					
					/* If this is a hash position Then get the hash key */
					if(isHashPosition(current)) {
						/* If we have to strip the hash information do it now */ 
						if(bStripHash) {
							result = current.substring(4,current.length()-2).trim();
						}
						else {
							result = current;
						}
					}
					else {
						/* This may be a index and not a key. */
						try { 
							result=Long.decode(current);
						}
						catch(Exception e) {
							/* This is not a number, ignore it */
						}
					}
					/* If there is a position key then add it to the list */
					if(result!=null) {
						newList.add(result);
					}
				}	
			}
		}
		return newList;
	  }

	/**
	 * Determines if this token is a hashed position token.
	 * 
	 * @param text
	 * @return
	 * @since 0.1
	 */
	private static boolean isHashPosition(String text) {
	  if (text !=null) {
		 return text.startsWith(HASH_START) && text.endsWith(HASH_END);
	  }
	  return false;		
	}

}
