/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: BufferedPeriodicReader.java,v 1.13 2005/03/23 08:09:06 dnsmith Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.logging.adapter.util;

import java.io.BufferedReader;
import java.io.EOFException;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.RandomAccessFile;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.StringTokenizer;


/**
 * BufferedPeriodicReader reads a file line by line and populates the returned
 * string array. It holds a pointer offset to the last read byte in order to resume.
 */
public class BufferedPeriodicReader
{
	private String filename = null;
	private String converter = null;
	private String [] converterArray = null;
	
	/* The shell to run the convertor in. */
	private String shell = null;
	
	/* The reader for the current file. */
	private RandomAccessFile currentFile = null;
	
	/* The stat info for the file on disk */
	private File currentFileStat=null;
	
	/* The last pointer into the file where we finished reading at. */
	private long lastBookmark = 0;
	
	/* The last time this file was changed */
	private long lastModified=0;

	/* The last size of this file when it was opened (bugzilla 78169) */
	private long lastSize=0;

	/* The size of the confidence buffer for determining if an append has occured within the file */
	private int confidenceBufferSize=1024;
	
	/* The size of the fotter within the file which contains variable content */
	private int fileFooterSize=90;
	
	/* Current thread lock for spawning child threads. */
	private Object currentThreadLock = new Object();

	/* Counter for tracking completed child threads. */
	private int childThreadsDone = 0;

	/* line separator characters */
	private static String EOL_CHARS = System.getProperty("line.separator");
	
	/* Last line separator character
	 * bugzilla 70772 - on z/OS the line separator character is x'15' but the System.getProperty returns x'0A'
	 * so we explicitly set the last line separator character to x'15' or decimal 21 on z/OS.
	 */
	private static char EOL_LAST_CHAR= (System.getProperty("os.name", "Windows").equals("z/OS") || System.getProperty("os.name", "Windows").equals("OS/390")) ? 21 : EOL_CHARS.charAt(EOL_CHARS.length()-1);
	 
	/** 
	 * Initial size of buffer to read data from log file into. 
	 * This value may need to be adjusted to improve performance.
	 */ 
	private int initialBufferSize = 2048;
	
	/** character set of the file we are dealing with */
	private String charset = "UTF-8";

	private boolean multiFile = false; // specify whether multi files are being monitored
	
	/* A buffer of historic info that we store whenever we hit the end of a file
	 * This historic info is used to locate the previous end of file.  Note:  We
	 * store the information in this buffer in reverse order.  This is because we
	 * traverse the new file in reverse order as well.
	 */	
	private byte[] historicInfo=null;
	
	/**
	 *
	 * @param newFilename
	 */
	public BufferedPeriodicReader(String newFilename) {
		this(newFilename,null,null);
	}
	
	/**
	 *
	 * @param newFilename
	 * @param converter
	 */
	public BufferedPeriodicReader(String newFilename, String converter)	{
		this(newFilename,converter,null);
	}

	/**
	 *
	 * @param newFilename
	 * @param converter
	 * @param shell
	 */
	public BufferedPeriodicReader(String newFilename, String converter, String shell) {
		this.filename = newFilename;
		this.converter = converter;
		this.converterArray = null;
		this.shell = shell;
	}	

	/**
	 *
	 * @param newFilename
	 * @param convArray
	 */
	public BufferedPeriodicReader(String newFilename, String [] convArray) {
		this.filename = newFilename;
		this.converter = null;
		this.shell = null;
		this.converterArray = convArray;
	}
	
	public void setCharset(String cs) {
		charset = cs;
	}
	
	/**
	 * prepare file for the reading
	 */
	public void prepare() throws Exception {		
		/* If the path to the converter is set, convert the file first. */
		if ((converter != null && converter.length() > 0) || 
			 converterArray != null && converterArray.length > 0) {
			String platform = System.getProperty("os.name", "");
			
			/*  
			 * bugzilla 72067 - if the converter command contains
			 * double quotes then convert it to a String array before
			 * running it.  This enables support for spaces in file and directory names. 
			 */	
			// Check if the converter command contains quotes
			if ((converterArray == null || converterArray.length == 0) && -1 != converter.indexOf("\"")) {
				// Change the converter command string to an array of strings
				int quoteIndex = converter.indexOf("\"");
				int matchingQuoteIndex = 0;
				String currentString = converter;
				String beforeQuote;
				ArrayList converterList = new ArrayList();
				
				// If a shell was specified then make it the first element in the 
				// command array.  For AIX, always run the command in shell.
				if (shell != null || (shell == null && !platform.startsWith("Windows") && !platform.startsWith("Linux"))) {
					// If no shell is specified, use sh
					if (shell == null) {
						shell = "sh";
					}
					converterList.add(shell);
				}
				
				// Process the quoted blocks in the command
				while (quoteIndex != -1) {
					if (quoteIndex > 0) {
						beforeQuote = currentString.substring(0,quoteIndex);
						StringTokenizer tokenizer = new StringTokenizer(beforeQuote);
						while (tokenizer.hasMoreTokens()) {
							converterList.add(tokenizer.nextToken());
						}
						currentString = currentString.substring(quoteIndex);
						quoteIndex = 0;
					}
					
					matchingQuoteIndex = currentString.substring(quoteIndex+1).indexOf("\"");
					if (matchingQuoteIndex == -1) {
						// there is no matching quote so quit
						break;
					}
					matchingQuoteIndex++;
					converterList.add(currentString.substring(quoteIndex+1,matchingQuoteIndex));
					if (currentString.length() > matchingQuoteIndex+1) {
						currentString = currentString.substring(matchingQuoteIndex+1).trim();
						quoteIndex = currentString.indexOf("\"");
					}
					else {
						quoteIndex = -1;
						currentString = null;
					}
				}
				
				// If we had matching quotes then we should have a string array
				if (matchingQuoteIndex != -1) {
					// Check for remaining tokens after the last quote
					if (currentString != null && currentString.length() > 0) {
						StringTokenizer tokenizer = new StringTokenizer(currentString);
						while (tokenizer.hasMoreTokens()) {
							converterList.add(tokenizer.nextToken());
						}
					}
					// Get the string array
					converterArray = new String[converterList.size()];
					Iterator it = converterList.iterator();
					for(int i=0; i< converterList.size(); i++) {
						converterArray[i] = (String)it.next();
					}
					// converterArray = (String[])converterList.toArray();
					converter = null;
				}
			}
			/* End of fix for bugzilla 72067 */
			
			try {
				//Execute the converter command as a runtime process and monitor any stdout and stderr output:
				Process converterProcess;
				
				//PROBLEM: Because some native platforms only provide limited buffer size for standard input streams, failure to read the input stream of the process may cause the process to block, and even deadlock.
				//SOLUTION: Process input streams must be read on seperate threads.
				synchronized (currentThreadLock) {
					//Reset the counter for tracking completed child threads:
					childThreadsDone = 0;
					
					// Execute the converter command:
					
					if (converterArray != null) {
						converterProcess = Runtime.getRuntime().exec(converterArray);
					}
					// Use the converter command alone on Windows
					else if (platform.startsWith("Windows")) {
						converterProcess = Runtime.getRuntime().exec(converter);
					}
					// Run the converter in a shell on Linux only if one was specified
					else if (platform.startsWith("Linux")) {						
						String converterCmd;
						// If no shell is specified we won't prepend one
						if (shell == null) {
							converterCmd = converter;
						}
						else {
							// Add the shell to the converter command
							converterCmd = shell + " " + converter;
						}

						//Execute the converter command string
						converterProcess = Runtime.getRuntime().exec(converterCmd);
					}
					// Run the converter in a shell on AIX
					else if (platform.startsWith("AIX")) {
						// For AIX we need to specify the command as an array of strings
						
						// If no shell is specified, use sh
						if (shell == null) {
							shell = "sh";
						}
						
						String [] cmdarr = {shell, converter};
						//Execute the converter command string array
						converterProcess = Runtime.getRuntime().exec(cmdarr);
					}
					// Run the converter in a shell on all UNIX platforms
					else {
						// If no shell is specified, use sh
						if (shell == null) {
							shell = "sh";
						}
						// Add the shell to the converter command
						String converterCmd = shell + " " + converter;
						
						//Execute the converter command string
						converterProcess = Runtime.getRuntime().exec(converterCmd);
					}
					
					//Monitor any stdout and stderr output from the execution of the converter command:
					new ConverterOutputReader(converterProcess.getErrorStream()).start();
					new ConverterOutputReader(converterProcess.getInputStream()).start();

					//ASSUMPTION: The process is only terminated after all its streams (i.e. stdout and stderr) have terminated.
					//Wait for all of the stdout and stderr output to be read before continuing:
					try {
						/* Wait no more than 5 minutes (300000 ms):
						 * RKD:  We need to log a message if this time expires.
						 */
						currentThreadLock.wait(300000);
					}
					catch (InterruptedException e) {
						/* Do not care if the wait is interrupted */
					}
				}

				/* Capture the exit value (by convention, the value 0 indicates normal termination) from the terminated converter process: */
				int exitValue = -1;
				
				try {
					exitValue = converterProcess.waitFor();
				}
				catch (InterruptedException i) {
					/* Destroy the converter process if it has not yet exited: */
					converterProcess.destroy();
				}

				/* Throw an exception if the process has not exited or exits abnormally (by convention, the value 0 indicates normal termination):
				 * RKD:  I think we should just log the exception occured.  Conventions are fine as long as everyone conforms to them.  Experience
				 * has proven this is not always the case.
				 */
				
				if (exitValue != 0) {
					throw new Exception(Messages.getString("HyadesGABufferedPeriodicReader_Converter_Process_Exit_Value_ERROR_",Integer.toString(exitValue)));
				}
			}
			catch (Exception e) {
				/*
   				 * Can't log an event here because this class is also
                 * called by the logging.parsers plugin
                 * So an exception will be thrown.
                 */			    
				throw new Exception(Messages.getString("HyadesGABufferedPeriodicReader_Converter_Command_Failed_ERROR_",e.toString()));
			}
		}
		
		/* RKD:  There are six conditions we need to be aware of and handle at this point.
		 *       1.  File has had content appended to the end.
		 *       2.  File has been replaced but otherwise content is appended to the end
		 *       3.  File has been replaced with data shifted up/left (circular log)
		 *       4.  File has been replaced with all new data.
		 *       5.  New file has been created on size or time boundaries
		 *       6.  Rotating logs with all of the 5 above conditions.
		 *
		 *  The solution below addresses the top four scenarios.  The algorithm is as follows:
		 *     1.  Always copy the last n bytes of the file from the last time we were at the end of
		 *         the file (n is configurable)
		 *     2.  Always save the filepointer from the last time we read the file.
		 *     3.  When the file is reopened, and it has been changed since the last time we read it
		 *         and has grown in size, the content of the file before
		 *         the current filepointer compared to determine if the filepointer is still
		 *         valid.  If it is, then compare the last n bytes of the previous read (where n
		 *         is a configurable value) if these n bytes are identical then conditions 1 and
		 *         2 are satisfied.   Set the new filepointer and remove our historic info saved in 1.
		 *         Else goto 4.
		 *     3.  Start searching from the end of the file until we find a match for
		 *         the historic buffer.  When found compare n bytes to determine this
		 *         is the old end of file. If this occurs condition 3 is satisfied.  Set the new
		 *         filepointer and close the old file.  Else condition 4 is satisfied, close the old
		 *         file and start from the beginning of the new file.
		 */
		
		try	{
			currentFileStat=new File(filename);
			/* If the file has been modified since we opened it OR
			 *    (bugzilla 78169) its size has changed since we opened it
			 * Then open the file again and save the modified time and size
			 */
			if(currentFileStat.lastModified()>lastModified || currentFileStat.length() != lastSize) {
				currentFile = new RandomAccessFile(filename, "r");
				/* RKD:  We likely need to lock the file read only and then set the lastmodified date before reading
				 * the file.  That way there it does not get changed underneigth us.  I have left this out for now.
				 */
				lastModified=currentFileStat.lastModified();
				lastSize = currentFile.length();
			}
		}
		catch (Exception e) {
			// The file cannot be opened for some reason
			// Throw an exception so this condition can be logged.
			currentFile = null;
			throw new Exception(Messages.getString("HyadesGABufferedPeriodicReader_Log_File_Open_Failed_ERROR_", filename,e.toString()));
		}
		
		
		
		/* Is there an old file.  if so we need to satisfy the conditions above.  Otherwise just open the new file. */
		if(lastBookmark!=0 && currentFile!=null) {
			try {
				/* determine how many characters we need to read to validate the file offsets */
				long previousOffset=lastBookmark;
				long currentOffset=currentFile.length();		
				long charactersToCompare=Math.min(Math.min(confidenceBufferSize, previousOffset), currentOffset);
				
				/* Walk the end of the file and try and determine if this is an append */
				if(currentOffset>=previousOffset) {	
					boolean append=true; 				
					for(int i=fileFooterSize; i<charactersToCompare; i++) {
						currentFile.seek(previousOffset-1-i);
						char value=(char)currentFile.read();
						if(historicInfo[i]!=(byte)value) {
							append=false;
							break;
						}
					}	
					/* If this is an append then the footer needs to be accouted for */
					if(append) {
						currentFile.seek(previousOffset-fileFooterSize);
						for(int i=(int)fileFooterSize-1; i>=0; i--) {	
							int value=currentFile.read();
							if(historicInfo[i]!=(byte)value) {
								break;
							}
						}
						historicInfo=null;
						lastBookmark=currentFile.getFilePointer();
						return;
					}
				}
				
				/* We are testing condition 3 here */
				long matchedChars=0;
				int historicInfoOffset=fileFooterSize;
				currentOffset-=fileFooterSize;
				long currentCurrentFileOffset=currentOffset;
				do {
					currentFile.seek(--currentCurrentFileOffset);
					if(historicInfo[historicInfoOffset++]==currentFile.read()) {
						matchedChars++;
					}
					else {
						currentOffset--;
						currentCurrentFileOffset=currentOffset;
						historicInfoOffset=fileFooterSize;
						matchedChars=0;
					}
				}while(matchedChars<charactersToCompare-fileFooterSize && currentOffset>0);
				
				/* if we have satified our file offset comparison we have condition 3, otherwise condition 4. */
				if(matchedChars==charactersToCompare-fileFooterSize) {
					/* The footer information is gone so we need to determine where to start from.  This
					 * can be done by traversing the historic info until we get a difference.
					 */
					currentFile.seek(currentOffset);
					for(int i=fileFooterSize-1; i>=0; i--) {	
						int value=currentFile.read();
						if(historicInfo[i]!=(byte)value) {
							historicInfo=null;
							lastBookmark=currentFile.getFilePointer();
							return;
						}
					}
					historicInfo=null;
					lastBookmark=currentOffset+fileFooterSize;
					return;
				}
				else {
					historicInfo=null;
					lastBookmark=0;
					return;
				}
			}
			catch (IOException e) {

				/*
   				 * Can't log an event here because this class is also
                 * called by the logging.parsers plugin
                 * So an exception will be thrown.
                 */			    
				throw new Exception(Messages.getString("HyadesGABufferedPeriodicReader_Determine_File_Modification_ERROR_", filename,e.getMessage()));
			}
			catch (Throwable e) {

				/*
   				 * Can't log an event here because this class is also
                 * called by the logging.parsers plugin
                 * So an exception will be thrown.
                 */			    
				throw new Exception(Messages.getString("HyadesGABufferedPeriodicReader_Determine_File_Modification_ERROR_", filename,e.toString()));
			}
		}
	}
	
	/**
	 * close file
	 */
	public void close()	{
		try {
			if (currentFile != null) {
				currentFile.close();
			}
		}
		catch (Exception e) {
		}
		currentFile = null;
	}
	/**
	 *
	 * @return
	 * @throws IOException
	 */
	public String readLine() throws IOException
	{
		boolean endOfFile=false;
		
		if (currentFile != null) {
			String last = null;
			currentFile.seek(lastBookmark);
			
			/* read a line */
			if (currentFile != null) {
				try {
					/* RKD:  START of workaround for RandomAccessFile.readLine()
					 * There are issues with using the RandomAccessFile.readLine() method
					 * as it takes each byte in a line and maps that to a character.  When dealing
					 * with DBCS character sets it fails.  Additionally, the readLine() method
					 * does not recognize end of line charaters on EBCDIC machines.
					 */
					/* bugzilla 77982 - use the same code here as in 
					 *	org.eclipse.hyades.logging.parsers.Parser.readLine()
	                 */
					
		            //Create a byte buffer with a default size of 2048 bytes (e.g. upper-bound for expected line length) which may need to be resized:
		            byte[] buffer = new byte[initialBufferSize];
		            
		            //The current size or number of bytes in the byte buffer:
		            int bufferSize = 0;

		            //A single byte from the file:
		            int singleByte = 0;
		            
		            //Iterate the remaining bytes in the file until the EOF (e.g. -1) or EOL character(s) (e.g. '\n', '\r' or '\r\n'):
		            while((singleByte = currentFile.read()) != -1) {

						// Check to see if we have hit the end of line character
						if (singleByte == EOL_LAST_CHAR) {
							if(EOL_CHARS.length()>1 && bufferSize > 0) {
								/* Handle case of line separator greater then 1 character in length */
								for(int k=EOL_CHARS.length()-2; k>=0;k--) {
									if((byte)EOL_CHARS.charAt(k) == buffer[bufferSize-1]) {
										/* Remove the line separator characters from the saved line */
										buffer[--bufferSize] = 0;
									}
									else {
										/* Stop procesing end of line characters if there are no more */
										break;
									}
								}
							}
							/* bugzilla 70772 - sometimes on UNIX machines lines end with \r\n
							 * so both characters need to be stripped
							 */
							else if (bufferSize > 0 && EOL_LAST_CHAR == '\n' && buffer[bufferSize-1] == '\r'){
								buffer[--bufferSize] = 0;
							}
							break;
						}

		                //If the byte buffer has overflowed, resize by another initial buffer size:
		                if(bufferSize == buffer.length){
		                    
		                    byte[] tempBuffer = new byte[buffer.length + initialBufferSize];
		                    
		                    System.arraycopy(buffer,0,tempBuffer,0,bufferSize);
		                    
		                    buffer = tempBuffer;                                       
		                }
		                
		                //Valid downcast (int --> byte) since the RandomAccessFile.read() API returns an integer in the range 0 to 255 (0x00 - 0x0ff). 
		                buffer[bufferSize++] = ((byte)(singleByte));
		            }
		            
		            if (singleByte == -1) {
		            	endOfFile = true;
		            }
		            
		            // If we've reached the end of file then return null
		            if (bufferSize == 0 && endOfFile) {
		            	last = null;
		            }
		            else {
		            	// Else If a line of data was read or a blank line was read Then return the line
		                // Convert the byte buffer to a string based on the charset specified in the configuation:
		                last = new String(buffer, 0, bufferSize, charset);
		            }					
					/* RKD:  END of workaround for RandomAccessFile.readLine() */
				}
				catch (EOFException e) {
					last=null;
				}
				catch (NullPointerException e) {
					last = null;
				}
				
				if(endOfFile) {
					/* Load the end of the file into a buffer so we can compare later.  We actually load the
					 * data in the buffer in reverse order
					 */
					int bytesToSave=(int)Math.min(currentFile.length(), confidenceBufferSize);
					currentFile.seek(currentFile.length()-bytesToSave);
					historicInfo=new byte[bytesToSave];
					for(int i=bytesToSave-1; i>=0; i--) {
						historicInfo[i]=(byte)currentFile.read();
					}
					
					/* We are done with the file.  Close it if we are not reading multiple files. */
					if(!multiFile) {
						close();
					}
					else {
						/* bugzilla 78405 - set the lastBookmark to the end of the file so we don't read any more data 
						 * unless it is appended to the end of the file 
						 */
						lastBookmark = currentFile.getFilePointer();
					}
				}
				else {
					lastBookmark = currentFile.getFilePointer();
				}
				
				return last;
			}
		}
		return null;
	}

	//PROBLEM: Because some native platforms only provide limited buffer size for standard input streams, failure to read the input stream of the process may cause the process to block, and even deadlock.
	//SOLUTION: Process input streams must be read on seperate threads.
	//Class used to read input streams from the converter process on seperate threads.
	private class ConverterOutputReader extends Thread
	{
		private BufferedReader reader;
		
		public ConverterOutputReader(InputStream inputStream)
		{
			reader = new BufferedReader(new InputStreamReader(inputStream));
		}
		public void run()
		{
			
			try
			{
				/* Read the input stream until the stream is closed (i.e. null termination): */
				while(reader.read()!=-1);
			}
			catch (IOException e)
			{
			}
			
			/* If both input streams (i.e. stdout and stderr) are closed, notify the waiting parent thread: */
			synchronized (currentThreadLock)
			{
				if (++childThreadsDone == 2)
					currentThreadLock.notify();
			}
		}
	}

	/**
	 * @return Returns the offset within the file where we are currently reading.
	 */
	public long getFilePointer() {
		return lastBookmark;
	}

	/**
	 * @return Returns the confidenceBufferSize.
	 */	
	public int getConfidenceBufferSize() {
		return confidenceBufferSize;
	}
	/**
	 * @param confidenceBufferSize The confidenceBufferSize to set.
	 */
	public void setConfidenceBufferSize(int confidenceBufferSize) {
		this.confidenceBufferSize = confidenceBufferSize;
	}
	/**
	 * @return Returns the fileFooterSize.
	 */
	public int getFileFooterSize() {
		return fileFooterSize;
	}
	/**
	 * @param fileFooterSize The fileFooterSize to set.
	 */
	public void setFileFooterSize(int fileFooterSize) {
		this.fileFooterSize = fileFooterSize;
	}
	/**
	 * @return Returns the multiFile.
	 */
	public boolean isMultiFile() {
		return multiFile;
	}
	/**
	 * @param multiFile The multiFile to set.
	 */
	public void setMultiFile(boolean multiFile) {
		this.multiFile = multiFile;
	}
	/**
	 * @return Returns the converterArray.
	 */
	public String[] getConverterArray() {
		return converterArray;
	}
	/**
	 * @param converterArray The converterArray to set.
	 */
	public void setConverter(String[] converterArray) {
		this.converterArray = converterArray;
		this.converter = null;
	}
	/**
	 * @param filename The filename to set.
	 */
	public void setFilename(String filename) {
		this.filename = filename;
	}
	/**
	 * @param converter The converter to set.
	 */
	public void setConverter(String converter) {
		this.converter = converter;
		this.converterArray = null;
	}
	/**
	 * Returns the size of the file being read
	 * @return Returns the lastSize.
	 */
	public long getSize() {
		return lastSize;
	}
}