/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: MultipleFilesReader.java,v 1.7 2005/04/20 18:29:26 dnsmith Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.logging.adapter.util;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.Vector;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;


/**
 * This thread is used to read a list of files and append them to a single temp
 * file for the reader. It will keep looking for changes in the files and make
 * sure any updates are published to the temp file.
 */
public class MultipleFilesReader extends Thread {

	private String dirName; // directory containing the files
	private String fileName_regex; // regex of the file names
	private Vector files = null; // list of file names
	private int index;
	private static int timeout = 5; // 5 seconds
	private static int MAX_CHAR = 1024; // max 1024 char per line
	private RandomAccessFile tmpLog = null;
	
	/* line separator characters */
	protected String EOL_CHARS = System.getProperty("line.separator");
	
	/* Last line separator character
	 * bugzilla 70772 - on z/OS the line separator character is x'15' but the System.getProperty returns x'0A'
	 * so we explicitly set the last line separator character to x'15' or decimal 21 on z/OS.
	 */
	protected char EOL_LAST_CHAR= (System.getProperty("os.name", "Windows").equals("z/OS") || System.getProperty("os.name", "Windows").equals("OS/390")) ? 21 : EOL_CHARS.charAt(EOL_CHARS.length()-1);
	
	
	public MultipleFilesReader(String dir, String f, RandomAccessFile tfile) {
		dirName = dir;
		if (!dirName.endsWith(File.separator)) {
			dirName += File.separator;
		}
		tmpLog = tfile;
		fileName_regex = f;
		this.setName("MultipleFilesReader"); // create a thread with this name
	}
	/**
	 * This is used to initialize the reader so that the list of matched files are added to the
	 * list. It will sort the list such that the first element is the oldest file while the last
	 * element is the newest file. This will allow readLine() to navigate through the files in
	 * the correct order
	 *
	 */
	public void init() {
		Pattern pat;
		Matcher mat;
		File dir;
		File[] allFiles;

		// List all the files in the specified dir
		if(dirName != null) {
			dir = new File(dirName);
			if (!dir.isDirectory()) {
				reset();
				return;
			}
			allFiles = dir.listFiles();
			// Create a pattern using the given regex
			if(fileName_regex != null) {
				files = new Vector();
				try {
					pat = Pattern.compile(fileName_regex);
				}
				catch (PatternSyntaxException e) {
					// If the pattern is invalid then return
					reset();
					return;
				}
				// Check each file to see if its name matches the file name pattern
				for(int i = 0; i < allFiles.length; i++) {
					mat = pat.matcher(allFiles[i].getName());
					if(mat.matches()) {
						boolean added = false;
						// If the name matches, add to the list in a sorted manner
						for(int j = 0; (j < files.size()) && (!added); j++) {
							// Find the timestamp of the files in the list and compare
							File f = new File((String)files.elementAt(j));
							if(allFiles[i].lastModified() < f.lastModified()) {
								// Add to the current location if it is older
								files.insertElementAt(dirName + File.separator + allFiles[i].getName(), j);
								added = true;
							}
						}
						// if the timestamp is the newest, add to the tail of the list
						if(!added) {
							files.add(dirName + allFiles[i].getName());
						}
					}
				}
			}
		}
		reset();

		return;
	}

	/**
	 * Reset the pointer to 0
	 */
	public void reset() {
		index = 0;
	}

	/**
	 * Return the number of matched files given by the regex
	 * @return number of matched files
	 */
	public int size() {
		// Check for null files - it might not have been set
		if (files == null) {
			return 0;
		}
		return files.size();
	}

	/**
	 * Check if there is any file left in the list
	 * @return true if there is at least a file left in the list
	 */
	public boolean hasNext() {
		return (index < files.size());
	}

	/**
	 * Retrieve the next file in the list and move the pointer
	 * @return name of the next file in the list
	 */
	public String getNext() {
		if(index < files.size()) {
			return (String)files.elementAt(index++);
		}
		else {
			return null;
		}
	}

	/**
	 * Copies multiple files into a single temporary file
	 * 
	 */
	public void loadConsolidatedFile() {
		int byteCopied = 0;

		// make sure the temp file exists before proceeding
		if(tmpLog != null) {
			while(hasNext()) { // copy the contents in all matched files to the temp file
				byteCopied = appendToTempFile(getNext());
			}
		}
	}

	/**
	 * Thread loop
	 * There are 2 filename variables to point to the newest and the 2nd newest
	 * file in the list. This is used to handle the case when the content of the
	 * newest file is copied to the 2nd newest one and the newest one is cleared
	 * (log rotation scenario). Since we are sampling the changes, we might not be
	 * able to catch all the lines at the bottom of the newest file when rotation
	 * occurs - that's when we need to look at the 2nd newest file for those lines.
	 */
	public void run() {
		String previousFileName = null;
		String currentFileName = null;
		RandomAccessFile previousFile = null;
		RandomAccessFile currentFile = null;
		byte[] buf = null;
		int bufLen = 0;
		int byteCopied = 0;

		// make sure the temp file exists before proceeding
		if(tmpLog != null) {
			while(hasNext()) { // copy the contents in all matched files to the temp file
				previousFileName = currentFileName;
				currentFileName = getNext();
				byteCopied = appendToTempFile(currentFileName);
			}

			// currentFile now has the most recent log
			buf = new byte[MAX_CHAR];
			bzero(buf);
			bufLen = 0;

			// Keep querying the newest file for changes 
			while(true && (currentFileName != null)) {
				try {
					// Open it for read only
					currentFile = new RandomAccessFile(currentFileName, "r");

					// If somehow the files are rotated...
					if(currentFile.length() < byteCopied) {
						// Look at the 2nd newest file
						previousFile = new RandomAccessFile(previousFileName, "r");
						previousFile.seek(byteCopied);
						// Check if there are new lines that are not copied due to rotation
						if(byteCopied < previousFile.length()) {
							for(; byteCopied < previousFile.length(); byteCopied++) {
								tmpLog.write(previousFile.readByte());
							}
						}
						previousFile.close();
						byteCopied = 0;
					}

					currentFile.seek(byteCopied);
					// Check if the newest file is updated or not
					if(byteCopied < currentFile.length()) { // if the file is updated (file size grows)
						for(; byteCopied < currentFile.length(); byteCopied++) {
							// Cache the bytes until a linefeed is found
							byte a = currentFile.readByte();
							buf[bufLen++] = a;
							if(a == EOL_LAST_CHAR) {
								tmpLog.write(buf, 0, bufLen);
								bzero(buf); // empty the buffer
								bufLen = 0;
							}
						}
					}

					currentFile.close();

					/* Sleep for 5 seconds before reading the current file again to see if it has changed */
					Thread.sleep(timeout * 1000);
				} catch (Exception e) {
					return;
				}
			}
		}
		else {
			return;
		}
	}

	/**
	 * Empty a buffer
	 * @param buffer
	 */
	private void bzero(byte[] buffer) {
		for(int i = 0; i < buffer.length; i++) {
			buffer[i] = 0;
		}
	}

	/**
	 * Copy the content of a file to the single temp file
	 * @param src The file name which its content has to be copied
	 * @return Number of bytes copied
	 */
	private synchronized int appendToTempFile(String src) {
		int len = 0;
		byte[] buf = null;

		if(tmpLog != null) {
			RandomAccessFile in;
			try {
				in = new RandomAccessFile(src, "r");

				tmpLog.seek(tmpLog.length());
				len = (int)in.length();
				buf = new byte[len];
				len = in.read(buf, 0, len);
				tmpLog.write(buf);
			} catch (Exception e) {
				/* 
				 * TODO We should really be throwing an exception here
				 * instead of continuing without including this file.
				 */
				return 0;
			}
			try {
				in.close();
			} catch (IOException e) {
				/* Ignore this */
			}
		}
		return len;
	}
}