/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.statistical.ui.editor.internal;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Arrays;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

public class ColumnPrinter {

	private int header_rowcount = 1;
	private int column_count = 0;

	private String[][] headers = new String[column_count][header_rowcount];
	private ZipStringBuffer[] columns = new ZipStringBuffer[column_count];

	private String cell_prefix = "";
	private String cell_postfix = "";
	private String cell_empty = "";

	private int getHeaderRowCount() {
		return header_rowcount;	
	}

	private int getColumnCount() {
		return column_count;	
	}

	private void addHeaderRow() {
		for (int i = 0; i < column_count; i++) {
			String[] tmp = new String[header_rowcount+1];
			if (headers[i].length > 0) {
				System.arraycopy(headers[i],0,tmp,1,header_rowcount);
			}
			headers[i] = tmp;
		}
		header_rowcount++;
	}	

	public void setHeaderText(int col, int row, String val) {
		while (getHeaderRowCount() <= row) {
			addHeaderRow();	
		}
		while (getColumnCount() <= col) {
			addColumn();	
		}
		//row should be from the bottom, up
		//---row 2
		//---row 1
		//---row 0
		row+=1;
		headers[col][header_rowcount-row] = val;
	}	

	private void addColumn() {
		String[][] tmp = new String[column_count+1][header_rowcount];
		System.arraycopy(headers,0,tmp,0,column_count);
		headers = tmp;

		ZipStringBuffer[] tmp2 = new ZipStringBuffer[column_count+1];
		System.arraycopy(columns,0,tmp2,0,column_count);
		columns = tmp2;

		column_count++;
	}

	public void setColumnText(int col, String val) {
		while (getColumnCount() <= col) {
			addColumn();	
		}
		columns[col] = new ZipStringBuffer(val);
	}

	public void appendColumnText(int col, String val) {
		while (getColumnCount() <= col) {
			addColumn();	
		}
		if (columns[col] == null) {
			setColumnText(col,val);
		} else {
			columns[col].append(val);
		}
	}

	public void setCellPrefix(String val) {
		cell_prefix = val;	
	}
	public void setCellPostfix(String val) {
		cell_postfix = val;
	}
	public void setCellEmpty(String val) {
		cell_empty = val;
	}

	public void printTo(OutputStream out) throws IOException {
		byte[] prefix = cell_prefix.getBytes();
		byte[] postfix = cell_postfix.getBytes();
		byte[] empty = cell_empty.getBytes();
		
		for (int r = 0; r < header_rowcount; r++) {
			for (int c = 0; c < column_count; c++) {
				out.write(prefix);
				if (headers[c][r] != null) {
					out.write(headers[c][r].getBytes());
				} else {
					out.write(empty);	
				}
				out.write(postfix);
			}
			out.write('\r');	//?
			out.write('\n');
		}
		
		boolean moreToGo = true;
		int[] indexes = new int[column_count];
		Arrays.fill(indexes,0);

		StringBuffer row = new StringBuffer();

		while (moreToGo) {
			moreToGo = false;

			row.setLength(0);

			for (int c = 0; c < column_count; c++) {
				row.append(cell_prefix);
	
				if (indexes[c] != -1) {
					if (columns[c] != null) {
						int index = indexOf(columns[c],'\n',indexes[c]);
						if (index == -1) {
							row.append(columns[c].substring(indexes[c]));
							indexes[c] = -1;
						} else {
							row.append(columns[c].substring(indexes[c],index));
							indexes[c] = index+1;
						}
						moreToGo = true;
					}
				} else {
					row.append(cell_empty);					
				}
	
				row.append(cell_postfix);					
			}
			if (moreToGo) {
				out.write(row.toString().getBytes());
				out.write('\r');	//?
				out.write('\n');
			}
		}
	}
	
	public int indexOf(ZipStringBuffer sb, char c, int index) {
		while (index < sb.length()) {
			if (sb.charAt(index) == c) return index;
			index++;	
		}
		return -1;
	}
	
	public String toString() {
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		try {
			printTo(bout);
			return new String(bout.toByteArray());
		} catch (IOException e) {
			return e.toString();
		}
	}	

	private byte[] compress(byte[] dat) throws IOException {
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		GZIPOutputStream gzout = new GZIPOutputStream(bout);
		
		gzout.write(dat,0,dat.length);
		
		gzout.finish();
		gzout.close();
		
		return bout.toByteArray();
	}
	
	private byte[] decompress(byte[] dat) throws IOException {
		ByteArrayInputStream bin = new ByteArrayInputStream(dat);
		GZIPInputStream gzin = new GZIPInputStream(bin);
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
				
		byte[] buf = new byte[1024];
		int n = 0;
		while (n != -1) {
			n = gzin.read(buf,0,1024);
			if (n > 0) {
				bout.write(buf,0,n);	
			}	
		}
		return bout.toByteArray();
	}
		
}